# -*- coding: UTF-8 -*-
import cliUtil
import common
import config

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

def execute(cli):
    '''
        块设备状态检查： 
        1.如果创建了虚拟机文件系统，且Health Status为Normal表示正常；
        2.如果没有查询到虚拟机块设备，表示正常；
        3.如果虚拟机块设备的Health Status为Normal，表示正常；
        4.其它情况均属于不正常
    '''
    flag = True
    cliRet = ""
    errMsg = ""
    
    try:
        dev_info = py_java_env.get("devInfo")
        p_version = str(dev_info.getProductVersion())
        dev_type = str(dev_info.getDeviceType())
        if p_version.startswith("V500R007C70") and \
                "Kunpeng" in p_version and \
                dev_type not in config.V5R7C70_SUPPORT_VM_MODEL:
            cli_ret = "Product Version:{0}\n Product Model:{1} " \
                      "Not Support VM.".format(p_version, dev_type)
            return cliUtil.RESULT_NOSUPPORT, cli_ret, ''

        #检查虚拟机服务是否开启
        checkRet = isVMStarted(cli)
        cliRet += checkRet[1]+"\n"
        
       #命令执行失败或虚拟机服务未开启
        if checkRet[0] != True:
            return (checkRet[0],checkRet[1],checkRet[2])
        elif checkRet[-1] == False:
            return (True,cliRet,checkRet[2])
        
        for ret in  (checkFileSystem(cli), checkBlockDevice(cli)):
            flag = flag and ret[0]
            cliRet += ret[1]
            errMsg += ret[2]
            
        if flag:
            LOGGER.logPass()
            
        return (flag, cliRet, errMsg)
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(LANG, "query.result.abnormal"))

def checkFileSystem(cli):
    '''
    @summary: 虚拟机文件系统检查    
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag: 
            True: 状态健康
            False: 状态不健康
        cliRet: 
            vmFlag为True时，返回设备上的信息
            vmFlag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    errMsg = ""
    flag = True
    
    cmd = "show vm_fs general"
    LOGGER.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if not checkRet[0] or checkRet[0] in ["NOCHECK", "NOSUPPORT"]: 
        LOGGER.logSysAbnormal()
        return checkRet
    
    cliRet = checkRet[1]
    if 'file system does not exist'.lower() in cliRet.lower():
        errMsg = common.getMsg(LANG, "cannot.get.vm.file.system.info")
        LOGGER.logNoPass("Cannot get information about VM file system")
        return(False, cliRet, errMsg)
    
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet) 
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.vm.file.system.info")
        LOGGER.logNoPass("Cannot get information about VM file system")
        return (False, cliRet, errMsg)
        
    for line in cliRetLinesList:
        Id = line.get("ID")
        healthStatus = line.get("Health Status")
        if healthStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(LANG, "port.vm.file.system.current.status.abnormal", 
                (Id, healthStatus))
            LOGGER.logNoPass("The health status of VM file system ID [%s] is abnormal(Health Status:%s)" % 
                (Id, healthStatus))
    
    return(flag, cliRet, errMsg)

def checkBlockDevice(cli):
    '''
    @summary: 虚拟机块设备检查    
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag: 
            True: 状态健康
            False: 状态不健康
        cliRet: 
            vmFlag为True时，返回设备上的信息
            vmFlag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    errMsg = ""
    flag = True
    
    cmd = "show virtual_machine block"
    LOGGER.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if not checkRet[0] or checkRet[0] in ["NOCHECK", "NOSUPPORT"]: 
        LOGGER.logSysAbnormal()
        return checkRet
     
    cliRet = checkRet[1]
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
        
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)       
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(LANG, "cannot.get.block.device.info")
        LOGGER.logNoPass("No controller [%s] VM block device has been queried")
        return (False, cliRet, errMsg)
        
    for line in cliRetLinesList:
        Id = line.get("ID")
        healthStatus = line.get("Health Status")
        if healthStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(LANG, "port.block.device.health.status.abnormal", 
                (Id, healthStatus))
            LOGGER.logNoPass("The health status of VM block device [%s] is abnormal (Health Status:%s)" % 
                (Id, healthStatus))
                        
    return(flag, cliRet, errMsg)

def isVMStarted(cli):
    '''
    @summary: 检查虚拟机服务是否开启
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        isStarted: 
            True: 虚拟机服务已开启
            False: 虚拟机服务未开启
        cliRet: 
                                检查虚拟机服务的命令回显
        errMsg: 错误消息
    '''
    flag = True
    isStarted = True
    cliRet = ""
    errMsg = ""
    cmd = "show vm_service general"
    
    execRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    # 校验是否包含错误码，如果包含则不涉及
    if cliUtil.has_unknown_error(execRet[1]):
        return cliUtil.RESULT_NOSUPPORT, execRet[1], cliUtil.getMsg(LANG, "has.not.license"), True

    if not cliUtil.hasCliExecPrivilege(execRet[1]):
        return (cliUtil.RESULT_NOSUPPORT, execRet[1], execRet[2], True)
    
    if execRet[0] != True:
        return (execRet[0],execRet[1],execRet[2],isStarted)
    
    cliRet = execRet[1]
    dictList = cliUtil.getHorizontalCliRet(cliRet)
    status = dictList[0].get("Status")
    if status != "Running":
        isStarted = False
        
    return (flag,cliRet,errMsg,isStarted)