# -*- coding: UTF-8 -*-
import cliUtil
import common
import re
from common_utils import get_err_msg
from com.huawei.ism.tool.obase.exception import ToolException


def execute(cli):
    dual_write_check = DualWriteCheck(py_java_env, PY_LOGGER, cli)
    flag, err_msg = dual_write_check.do_check()
    return flag, "\n".join(dual_write_check.cli_ret_list), err_msg


class DualWriteCheck:
    def __init__(self, py_java_env, logger, cli):
        self.logger = common.getLogger(logger, __file__)
        self.py_java_env = py_java_env
        self.lang = common.getLang(py_java_env)
        self.cli_ret_list = list()
        self.cli = cli

    def do_check(self):
        try:
            if not self.check_hot_patch():
                return True, ""
            flag, cli_ret, err_msg = cliUtil.enterDeveloperMode(
                self.cli, self.lang
            )
            self.cli_ret_list.append(cli_ret)
            if not flag:
                return flag, err_msg
            return self.check_controller_concur()
        except (Exception, ToolException) as e:
            self.logger.logException(e)
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        finally:
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)
            self.logger.logInfo(
                "enter cli mode from some model ret is %s" % str(ret))
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, self.logger)

    def check_controller_concur(self):
        """
        检查每个控制下的Remain Concur字段值
        :return:
        """
        (
            flag,
            control_id_list,
            err_msg,
            cli_ret,
        ) = cliUtil.getControllerIdListWithRet(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if not flag:
            return False, err_msg
        err_controller_list = list()
        for control_id in control_id_list:
            cmd = "show qos concurrency_obj controller={}".format(control_id)
            flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
                self.cli, cmd, True, self.lang
            )
            self.cli_ret_list.append(cli_ret)
            if not flag:
                return False, err_msg
            cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
            for cli_ret_line in cli_ret_lines_list:
                name = cli_ret_line.get("Name", "")
                if name != "concurrHostIoObj":
                    continue
                remain_concur = int(cli_ret_line.get("Remain Concur", ""))
                if remain_concur <= 0 or remain_concur > 8192:
                    err_controller_list.append(control_id)
                    break
        if len(err_controller_list) == 0:
            return True, ""
        return (
            False,
            get_err_msg(
                self.lang, "check.dual.write", ",".join(err_controller_list)
            ),
        )

    def check_hot_patch(self):
        """
        检查是否打了120以上的版本补丁

        :return: true 有风险
        """
        (
            cli_ret,
            __,
            hot_patch_ver,
        ) = common.getHotPatchVersionAndCurrentVersion(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        pattern_hot_patch = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
        match_hot_path = pattern_hot_patch.search(hot_patch_ver)
        # 如果为特殊补丁，没有补丁，或者补丁小于V300R002C10SPH120，则存在风险
        if (
            not match_hot_path
            or re.search(r"SPH\d[6-9]\d", hot_patch_ver)
            or int(match_hot_path.group(1)) < 120
        ):
            return True
        return False
