# -*- coding: UTF-8 -*-
import common
import cliUtil
from utils import Products

DEVICE_TYPE = "2600 V3 for Video"
# 支持NAS的版本号：V300R006C20及之后版本；之前的版本支持NoNAS
PATCH_KEY_FOR_26_V_VER_DICT = {"V300R006C20": "V300R006C20SPH030",
                               "V300R006C50SPC100": "V300R006C50SPH125"}

PY_JAVA_ENV = py_java_env
LOGGER = common.getLogger(PY_LOGGER, __file__)
LANG = common.getLang(PY_JAVA_ENV)

# 进度总剩余时间
LIMIT_TIME = 120
# 进度刷新间隔
INTERVAL = 2


def execute(cli):
    """
    检查方法(中文
    步骤1、获取当前设备的型号及版本号；
    步骤2、执行命令show file_system general，查询文件系统信息。
    步骤3、小系统下执行ls -l /OSM/coffer_bin/hotpatch/patch_cur，查询是否是NAS的补丁。\]
    检查标准(中文)
    1.若当前设备型号为2600 V3 for Video，系统版本为V300R006C20和V300R006C50SPC100版本，继续检查；否则检查通过；
    2.若当前设备没有安装补丁，则检查通过；
    3.V300R006C50SPC100安装了SPH125或之后补丁，或者V300R006C20安装了SPH030或之后补丁，则检查通过；否则继续检查；
    4.若步骤2存在文件系统，继续检查；否则检查通过；
    5.若步骤3没有查到以cmr_auth_、cmr_store_、cmr_cm_、cmr_kagent、ndmp_开头的文件，则检查不通过；否则检查通过；
    :param cli:
    :return:
    """
    dev_node = PY_JAVA_ENV.get("devInfo")
    fs_check = FSConfigCheck(cli, dev_node)
    try:
        # 进度条刷新
        common.threadUpProcess(PY_JAVA_ENV, LIMIT_TIME, INTERVAL, LOGGER)
        # 进度开始
        common.inProcess(PY_JAVA_ENV)
        return fs_check.check()
    except Exception as exception:
        LOGGER.logException(exception)
        return (
            cliUtil.RESULT_NOCHECK,
            fs_check.get_all_ret(),
            common.getMsg(LANG, "query.result.abnormal")
            )
    finally:
        common.finishProcess(PY_JAVA_ENV)
        LOGGER.logInfo("finish process!")


class FSConfigCheck:
    def __init__(self, cli, dev_node):
        self.cli = cli
        self.lang = LANG
        self.log = PY_LOGGER
        self.dev_type = str(dev_node.getDeviceType())
        self.product_version = dev_node.getProductVersion()
        self.product_hotpatch = dev_node.getHotPatchVersion()
        self.all_cli_ret = []

    def check(self):
        self.log.info("current device type:{},current patch version:{}, current version:{}"
                      .format(self.dev_type, self.product_hotpatch, self.product_version))
        # 当前设备型号为2600 V3 for Video
        self.all_cli_ret.append(self.dev_type)
        if self.dev_type != DEVICE_TYPE:
            return True, self.get_all_ret(), ""
        # 1、系统版本不为V300R006C20和V300R006C50SPC100版本，检查通过
        self.all_cli_ret.append(self.product_version)
        if self.product_version not in PATCH_KEY_FOR_26_V_VER_DICT.keys():
            return True, self.get_all_ret(), ""
        # 2、若当前设备没有安装补丁，则检查通过；
        self.all_cli_ret.append(self.product_hotpatch)
        if not self.product_hotpatch or self.product_hotpatch == "--":
            self.log.info("not install patch ,patch version:{}".format(self.product_hotpatch))
            return True, self.get_all_ret(), ""
        # 3、.V300R006C50SPC100安装了SPH125或之后补丁，或者V300R006C20安装了SPH030或之后补丁，则检查通过；
        if Products.compareVersion(self.product_hotpatch, PATCH_KEY_FOR_26_V_VER_DICT.get(self.product_version)) >= 0:
            self.log.info(
                "current patch version:{}, there is more than version".format(self.product_hotpatch))
            return True, self.get_all_ret(), ""
        flag, err_msg = self.check_file_system_exist()
        if flag is not False:
            return flag, self.get_all_ret(), err_msg
        # 4、查询是否安装NAS补丁。
        flag, err_msg = self.check_install_nas_patch()
        if flag is not True:
            return flag, self.get_all_ret(), err_msg
        return True, self.get_all_ret(), ""

    def check_file_system_exist(self):
        """
        根据是否存在文件系统获取补丁包关键字；如果存在文件系统，提示联系技术支持工程师；
        如果没有文件系统，选择NoNAS补丁包（前提条件当前设备已经安装有补丁）
        :return: flag,err_msg;
        """
        self.log.info("check_file_system_exist start.")
        cmd = "show file_system general"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True, LANG)
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            self.log.error("check_file_system_exist failed.")
            return cliUtil.RESULT_NOCHECK, err_msg
        file_sys_list = cliUtil.getHorizontalCliRet(cli_ret)
        # 如果存在文件系统，则继续检查
        if file_sys_list:
            self.log.info("exist file system or license.")
            return False, common.getMsg(self.lang, "check.file.system.exist.tip")
        return True, ""

    def check_install_nas_patch(self):
        """ 检查是否安装nas补丁：小系统ls -l /OSM/coffer_bin/hotpatch/patch_cur
            查看是否有cmr_auth_、cmr_store_、cmr_cm_、cmr_kagent、ndmp_其中的一个，
            如果有则是NAS补丁，否则为NO-NAS的补丁
            """

        self.log.info("get is_install_nas_patch start.")
        keylist = ("cmr_auth_", "cmr_store_", "cmr_cm_", "cmr_kagent", "ndmp_")
        cmd = "ls -l /OSM/coffer_bin/hotpatch/patch_cur"
        flag, cli_ret, err_msg = cliUtil.excute_cmd_in_minisystem_with_check(self.cli, cmd, self.lang)
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            self.log.error("get is_install_nas_patch failed.")
            return cliUtil.RESULT_NOCHECK, err_msg

        for key in keylist:
            if key in cli_ret:
                return True, ""
        err_msg = common.getMsg(self.lang, "check.file.system.config.tip",
                                (self.product_version, self.product_hotpatch,
                                 PATCH_KEY_FOR_26_V_VER_DICT.get(self.product_version)))
        return False, err_msg

    def get_all_ret(self):
        return "\n".join(self.all_cli_ret)
