# -*- coding: UTF-8 -*-
import traceback
import re

import cliUtil
import common
from common import UnCheckException
from com.huawei.ism.tool.obase.exception import ToolException
from risk_version_config import (
    RISK_SYSTEM_MAX_VOLUME_ID_MODEL as risk_product_model,
)

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)

RISK_VERSION = re.compile("6.0.1.*")
VOLUME_ID = re.compile("Volume ID\s*: (\d+)")


def execute(cli):
    system_max_volume_id = SystemMaxVolumeId(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = system_max_volume_id.execute_check()
    return flag, "\n".join(system_max_volume_id.all_ret_list), msg


class SystemMaxVolumeId:
    """
    LUN的卷ID检查
    """

    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []
        self.product_model = ""
        self.software_version = ""
        self.hot_patch_version = ""

    def execute_check(self):
        try:
            if not self.is_risk_version():
                return True, ""
            (flag, ret, message, lun_id_list) = cliUtil.getLunIdList(
                self.cli, self.lang
            )
            self.all_ret_list.append(ret)
            if flag is not True:
                raise common.UnCheckException(message, ret)
            if not lun_id_list:
                return True, ""
            risk_volume_id_list = self.get_risk_volume_id(lun_id_list)
            if not risk_volume_id_list:
                return True, ""
            error_message = common.getMsg(
                self.lang,
                "system.max.volume.risk_id",
                ",".join(risk_volume_id_list),
            )
            return False, error_message
        except common.UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except ToolException:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        finally:
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, LANG)
            self.logger.logInfo(
                "enter cli mode from some model ret is %s" % str(ret)
            )
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, LOGGER)

    def get_risk_volume_id(self, lun_id_list):
        """
        遍历lun_id_list执行lun showctrl {lun_id},获取Volume ID的值并判断是否有风险
        返回[有风险的lun_id]
        :param lun_id_list:
        :return:list[]
        """
        risk_lun_id_list = []
        for lun_id in lun_id_list:
            cmd = "lun showctrl {}".format(lun_id)
            flag, ret, err_msg = cliUtil.excuteCmdInDebugModel(
                self.cli, cmd, self.lang
            )
            self.all_ret_list.append(ret)
            if flag is not True:
                raise UnCheckException(err_msg, ret)
            match_flag = VOLUME_ID.search(ret)
            if match_flag and int(match_flag.group(1)) > 100000:
                risk_lun_id_list.append(lun_id)
        return risk_lun_id_list

    def is_risk_version(self):
        (
            flag,
            self.product_model,
            ret,
            msg,
        ) = cliUtil.getProductModelWithCliRet(self.cli, self.lang)
        self.all_ret_list.append(ret)
        if flag is not True:
            raise UnCheckException(msg, ret)
        if self.product_model not in risk_product_model:
            return False
        (
            flag,
            cli_ret,
            err_msg,
            self.software_version,
            self.hot_patch_version,
        ) = common.getVersion(self.cli, self.lang)
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, cli_ret)
        if self.software_version == "6.0.1":
            pattern = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
            match_hot_path = pattern.search(self.hot_patch_version)
            if not match_hot_path:
                return True
            if int(match_hot_path.group(1)) < 6:
                return True
        return False
