# coding=UTF-8
import common
import java.lang.Exception as JException

import cliUtil
import common_cache
from cbb.frame.base.exception import CliCmdException
PY_JAVA_ENV = py_java_env


def execute(cli):
    """
    存储池分条深度检查
    :return: (检查结果，CLI回显，错误提示信息）
    """
    return StoragepoolStripeDepth(cli).run_check()


class StoragepoolStripeDepth(object):
    LANG = common.getLang(PY_JAVA_ENV)
    LOGGER = common.getLogger(PY_LOGGER, __file__)
    NO_PASSED_RANGE_VERSIONS = (
        ('V300R006C50SPC100', 'V300R006C61'),
        ('V500R007C30SPC100', 'V500R007C60SPC300'),
    )
    WARN_NAME_STRIP_DEPTH_DICT = {
        'Performance': 512,     # 单位KB
        'Capacity': 512,
        'Extreme Performance': 256,
    }
    all_cli_ret = ''
    flag = None
    err_msg = ''
    disk_domain_id_list = []

    def __init__(self, cli):
        self.cli = cli
        return

    def is_passed_version(self, version, hot_ver):
        """
若步骤2中软件版本满足以下情况，则检查通过，否则继续检查；
小于 V300R006C50SPC100/V500R007C30SPC100
V300R006C50SPC100：安装高于 V300R006C50SPH108 补丁
V500R007C30SPC100：安装高于 V500R007C30SPH108 补丁
大于等于V300R006C61/V500R007C60SPC300
        :param version:
        :return:
        """
        if version == "V300R006C50SPC100" and hot_ver > 'V300R006C50SPH108':
            return True
        if version == 'V500R007C30SPC100' and hot_ver > 'V500R007C30SPH108':
            return True
        if version == 'V500R007C60SPC100' and hot_ver >= 'V500R007C60SPH105':
            return True
        for low_ver, hi_ver in self.NO_PASSED_RANGE_VERSIONS:
            if low_ver <= version < hi_ver:
                return False
        return True

    def get_disk_domain_id(self):

        cmd = "show disk_domain general"
        self.flag, cli_ret, self.err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.LANG)
        self.all_cli_ret += cli_ret

        if self.flag is not True:
            raise CliCmdException(
                'Get disk domain info error. flag:{} err_msg:{} cli_ret:{}'
                ''.format(self.flag, self.err_msg, cli_ret)
            )

        info_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
        for line in info_dict_list:
            self.disk_domain_id_list.append(line.get('ID'))
        return self.disk_domain_id_list

    def enter_developer_view(self):
        flag, cli_ret, self.err_msg = cliUtil.enterDeveloperMode(
            self.cli, self.LANG)
        self.all_cli_ret += cli_ret
        if self.flag is not True:
            raise CliCmdException(
                'Enter developer view error. flag:{} err_msg:{} cli_ret:{}'
                ''.format(self.flag, self.err_msg, cli_ret)
            )

        return

    def get_all_disk_domain_on(self):
        """
        检查所有的硬盘域状态
        :return: (检查是否成功，所有的硬盘域都是关闭状态)
        """
        status_on_disk_domain_id_list = []
        for dd_id in self.disk_domain_id_list:
            cmd = "show disk_domain lid_switch disk_domain_id={}".format(dd_id)
            self.flag, cli_ret, self.err_msg = cliUtil.excuteCmdInCliMode(
                self.cli, cmd, True, self.LANG)
            self.all_cli_ret += cli_ret

            if self.flag is not True:
                raise CliCmdException(
                    'Get disk domain status error. '
                    'flag:{} err_msg:{} cli_ret:{}'.format(
                        self.flag, self.err_msg, cli_ret)
                )

            info_dict = cliUtil.getCliRetDict(cli_ret)
            if info_dict.get('LID Switch') == 'On':
                status_on_disk_domain_id_list.append(dd_id)

        return status_on_disk_domain_id_list

    def get_warning_storage_pool(self, on_status_disk_domain_id_set):
        """
        查询阵列配置的存储池ID
        :param on_status_disk_domain_id_set:
        :return:
        """
        cmd = "show storage_pool general"
        self.flag, cli_ret, self.err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.LANG)
        self.all_cli_ret += cli_ret

        if self.flag is not True:
            raise CliCmdException(
                'Get storage pool general error. flag:{} err_msg:{} cli_ret:{}'
                ''.format(self.flag, self.err_msg, cli_ret)
            )

        info_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
        common.refreshProcess(PY_JAVA_ENV, 30, self.LOGGER)
        stor_pool_id_err_msg = {}
        process = 30
        each_process = 70.0 / len(info_dict_list) if info_dict_list else 0
        for stor_pool_info in info_dict_list:
            disk_domain_id = stor_pool_info.get('Disk Domain ID')
            if disk_domain_id in on_status_disk_domain_id_set:
                stor_id = stor_pool_info.get('ID')
                err_list = self.is_warning_storage_pool(stor_id)
                if err_list:
                    stor_pool_id_err_msg[stor_id] = {
                        "disk_domain_id": disk_domain_id,
                        "err_list": err_list,
                    }
            process += each_process
            process = process if process < 100 else 100
            common.refreshProcess(PY_JAVA_ENV, process, self.LOGGER)

        if stor_pool_id_err_msg:
            # 报错信息待确定
            err_msg = ""
            for s_p_id, info in stor_pool_id_err_msg.items():
                err_msg_list = map(
                    lambda x: common.getMsg(
                        self.LANG,
                        "storage_pool_strip_depth_iter_error",
                        x),
                    info['err_list']
                )
                iter_err = ', '.join(list(err_msg_list))
                err_msg += common.getMsg(
                    self.LANG,
                    "storage_pool_strip_depth_error",
                    (info['disk_domain_id'], s_p_id, iter_err)
                )
            return False, self.all_cli_ret, err_msg
        return True, self.all_cli_ret, ""

    def is_warning_storage_pool(self, storage_pool_id):
        """
        判断是否风险存储池
        :param storage_pool_id: 存储池ID
        :return: True:是  False:否
        """
        cmd = "show storage_pool tier pool_id={}".format(storage_pool_id)
        self.flag, cli_ret, self.err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.LANG)
        self.all_cli_ret += cli_ret

        if self.flag is not True:
            raise CliCmdException(
                'Storage pool info get error. flag:{} err_msg:{} cli_ret:{}'
                ''.format(self.flag, self.err_msg, cli_ret)
            )

        info_dict_list = cliUtil.getVerticalCliRet(cli_ret)
        err_list = []
        for info_dict in info_dict_list:
            name = info_dict.get('Name')
            strip_depth = info_dict.get('Stripe Depth')
            warn_depth_value = self.WARN_NAME_STRIP_DEPTH_DICT.get(name)
            strip_depth_value = cliUtil.transfer_capacity(strip_depth, 'KB')
            if bool(warn_depth_value and
                    strip_depth_value >= warn_depth_value):
                err_list.append((name, strip_depth))
        return err_list

    def _run_check(self):
        """运行检查项"""

        # 版本检查
        flag, vr, h_v, ret, err_msg = common_cache.get_version_and_patch_cache(
            PY_JAVA_ENV, self.cli, self.LOGGER)
        self.all_cli_ret += ret
        if flag is not True:
            return flag, self.all_cli_ret, err_msg
        if self.is_passed_version(vr, h_v):
            return True, self.all_cli_ret, ""
        common.refreshProcess(PY_JAVA_ENV, 5, self.LOGGER)

        # 硬盘域的LID状态检查
        if not self.get_disk_domain_id():
            # 无硬盘域情况直接通过
            return True, self.all_cli_ret, ""

        # 进入开发者视图
        self.enter_developer_view()
        common.refreshProcess(PY_JAVA_ENV, 10, self.LOGGER)
        # 查看硬盘域的lid状态
        on_status_disk_domain_ids = self.get_all_disk_domain_on()

        if not on_status_disk_domain_ids:
            # 均关闭状态，则检查通过
            return True, self.all_cli_ret, ""
        common.refreshProcess(PY_JAVA_ENV, 20, self.LOGGER)
        # 查询阵列配置的存储池ID
        return self.get_warning_storage_pool(
            set(on_status_disk_domain_ids)
        )

    def run_check(self):

        try:
            return self._run_check()
        except CliCmdException:
            return self.flag, self.all_cli_ret, self.err_msg
        except (Exception, JException) as exception:
            self.LOGGER.logException(exception)
            return cliUtil.RESULT_NOCHECK, self.all_cli_ret, ""
        finally:
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.LANG)
            self.LOGGER.logInfo(
                "Storagepool strip depth check end."
                "Enter cli mode from some model, ret: %s" % str(ret)
            )
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, self.LOGGER)
