# -*- coding: UTF-8 -*-

import re
import json
import traceback
import common
import cliUtil
import resource_error_msg
from common import (
    UnCheckException,
    getVstoreNameById,
    enterVstoreMode,
    joinLines,
    exitVstoreMode,
    getMsg,
    reConnectionCli,
)
from risk_version_config import FC_FAULT_RISK_VERSION_DICT

from cbb.frame.checkitem.base_dsl_check import CheckStatus
from cbb.frame.cli.cliUtil import get_user_level_and_role_id
from cbb.frame.cli.cliUtil import get_system_version_with_ret

from frameone.rest import restData
from frameone.rest import restUtil
from frameone.util import contextUtil


DIAGNOSE_END_FLAG = "diagnose>"


def get_result_bureau(env, ret, msg):
    """
    开局交付，除命令执行失败未完成检查需要处理，其他未完成检查场景优化为建议优化。
    :param env: 上下文
    :param ret: 原始信息
    :param msg: 错误消息
    :return: 结果
    """
    return cliUtil.RESULT_WARNING if common.is_opening_delivery_inspect(env) else cliUtil.RESULT_NOCHECK, ret, msg


def get_err_msg(lang, msg, args=""):
    """
    @summary: 消息国际化
    :param lang: 语言lang
    :param msg: 消息
    :param args: 消息参数
    :return: 经过国际化处理后的消息
    """
    return cliUtil.getMsg(
        lang, msg, args, resource_error_msg.ERROR_MESSAGES_DICT
    )


def check_all_key_in_line(line, keys):
    """
    检查是否所有的key都在此行中，如果都在话判定为True
    :param line: 待确认行
    :param keys: 字段
    :return: True 是， False 不是
    """
    for key in keys:
        if key not in line:
            return False
    return True


def ana_form_by_space(
        ret, key_list, logger, end_flag=DIAGNOSE_END_FLAG, split_key=None
):
    """
        专门处理表头不是用空格区分的，而是制表符区分的场景，
        正常按位无法解析的场景。使用python自带的split划分表头和值,
        注意：不能用于处理值中有空值的场景。
        :param ret: 回文
        :param key_list: 需要解析的字段
        :param end_flag: 结束符
        :param split_key: title分隔符
        :return: 字典列表
        """
    hor_header_dict = {}
    res_dict_list = []
    ret_list = ret.splitlines()
    try:
        title_find_flag = False
        for ret in ret_list:
            if not title_find_flag and check_all_key_in_line(ret, key_list):
                if not split_key:
                    head_keys = ret.split()
                else:
                    head_keys = list(
                        map(lambda t: t.strip(), ret.split(split_key))
                    )
                for key in key_list:
                    hor_header_dict[key] = head_keys.index(key)
                title_find_flag = True
                continue

            # 排除 title下面为-- 的空行
            if not ret.replace("-", "").strip():
                continue

            # 解析数据行
            if title_find_flag and not ret.endswith(end_flag):
                res_dict = {}
                values = ret.split()
                if len(values) < len(hor_header_dict):
                    continue
                for key in hor_header_dict:
                    value_index = hor_header_dict.get(key)
                    res_dict[key] = values[value_index]
                res_dict_list.append(res_dict)
    except Exception:
        logger.logError(traceback.format_exc())
    return res_dict_list


def ana_special_form(ret, query_key_list, logger, end_flag=DIAGNOSE_END_FLAG):
    """
    对于标准的对齐的格式解析，非制表符的场景，但值里可能带空格。
    又不带 -- 用于找表头的解析方法
    :param ret:
    :param query_key_list: 需要解析的字段
    :param end_flag: 结束符
    :return: 字典列表
    """
    start_pos_key = "start_pos"
    end_pos_key = "end_pos"
    hor_header_dict = {}
    res_dict_list = []
    ret_list = ret.splitlines()
    try:
        title_find_flag = False
        for ret in ret_list:
            if not title_find_flag and check_all_key_in_line(
                ret, query_key_list
            ):
                hor_header_dict = get_title_data(
                    query_key_list, ret, start_pos_key, end_pos_key
                )
                title_find_flag = True
                continue

            # 排除 title下面为-- 的空行
            if not ret.replace("-", "").strip():
                continue

            # 解析数据行
            if title_find_flag and not ret.endswith(end_flag):
                res_dict = {}
                for key in hor_header_dict:
                    start_pos = hor_header_dict.get(key, {}).get(
                        start_pos_key, 0
                    )
                    end_pos = hor_header_dict.get(key, {}).get(end_pos_key, 0)
                    value = ret[start_pos:end_pos].strip()
                    res_dict[key] = value
                res_dict_list.append(res_dict)
    except Exception:
        logger.logError(traceback.format_exc())
    return res_dict_list


def get_title_data(query_key_list, ret, start_pos_key, end_pos_key):
    """
    解析title
    :param query_key_list: title 字段
    :param ret: title 行
    :return: title字典
    """
    hor_header_dict = {}
    for key in query_key_list:
        reg_split = re.compile(r"{}\s*".format(key))
        start_pos = ret.find(key)
        match = reg_split.search(ret)
        if match:
            end_pos = match.end()
            hor_header_dict[key] = {
                start_pos_key: start_pos,
                end_pos_key: end_pos,
            }

    return hor_header_dict


def check_in_vstore_mode_wrap(func):
    """
    vstore 租户模式装饰器
    :param func:
    :return:
    """

    def wrap(cls, param_dict, *args, **kargs):
        cli = param_dict.get("cli")
        vstore_id = param_dict.get("vstore_id")
        logger = param_dict.get("logger")
        lang = param_dict.get("lang")
        all_ret_list = param_dict["cli_ret_all"]
        enter_ret = ""
        # 获取Vstore Name
        flag, vstore_name, cli_ret, err_msg = getVstoreNameById(
            cli, vstore_id, lang
        )
        if flag is not True:
            enter_ret = joinLines(enter_ret, cli_ret)
            all_ret_list.append(enter_ret)
            raise UnCheckException(err_msg, enter_ret)
        try:
            flag, cli_ret, err_msg = enterVstoreMode(
                cli, vstore_id, vstore_name, lang, logger
            )
            enter_ret = joinLines(enter_ret, cli_ret)
            all_ret_list.append(enter_ret)
            if flag is not True:
                raise UnCheckException(err_msg, enter_ret)
            func_ret = func(cls, param_dict)
            return func_ret
        except Exception:
            logger.logError(str(traceback.format_exc()))
            raise UnCheckException(
                getMsg(lang, "query.result.abnormal"), enter_ret
            )
        finally:
            _, cli_ret, _ = exitVstoreMode(cli, lang, logger, vstore_name)
            all_ret_list.append(cli_ret)
            logger.logInfo("exit vstore mode success.")
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(cli, lang)
            logger.logInfo("enter cli mode from some model  %s" % str(ret))
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                reConnectionCli(cli, logger)

    return wrap


def is_special_patch(patch_version):
    """
    是否特殊补丁
    :param patch_version:
    :return: True 是，False 不是
    """
    return re.search(r"SPH\d[6-9]\d", patch_version) is not None

def get_p_patch_num(p_patch):
    """
        @summary: 获取热补丁版本的数字，仅限SPH(\d+)格式的热补丁版本。
        :param p_patch: 热补丁版本
        :return: 热补丁版本的数字
        """
    reg = re.compile(r"SPH(\d+)")
    match_res = reg.search(p_patch)
    if match_res:
        p_patch_num = match_res.group(1)
        return int(p_patch_num)
    else:
        return 0


def check_conn_and_mode(cli, lang, logger):
    """
    退出模式，检查连接并重连。
    :param cli:
    :param lang:
    :param logger:
    :return:
    """
    # 退出到cli模式
    flag, ret, msg = cliUtil.enterCliModeFromSomeModel(cli, lang)
    logger.logInfo(
        "enter cli mode from some model ret is {}".format(str(ret))
    )
    # 退出失败后为不影响后续检查项重新连接cli
    if flag is True:
        return True

    if cli is None:
        return False
    try:
        cli.reConnect()
        logger.logInfo("it is afresh to connect to device by ssh gallery")
        return True
    except:
        logger.logError(str(traceback.format_exc()))
        return False


def is_expansion_scene(env, logger):
    """
    是否扩容扩容前巡检
    因扩容场景无法评估在哪个存储池上，所以直接通过。
    :return:
    """
    scene_data = env.get("sceneData")
    if not scene_data:
        return False

    logger.logInfo("scene data :{}".format(scene_data))
    if scene_data.get("mainScene") != "Expansion":
        return False

    tool_scene = scene_data.get("toolScene")
    if not tool_scene:
        return False

    # 是否是扩容前巡检
    if tool_scene == "perInspector":
        return True

    return False


def is_risk_version_fc_cause_risk(cli, current_version,
                                  current_patch, logger):
    """
    工具规避在收集日志时触发的问题
    @param current_version: 当前阵列的版本所有控制器的版本列表
    @param current_patch: 当前阵列的版本所有控制器的补丁版本列表
    @param logger:
    @return：False：表示是风险版本 True：表示不是风险版本
    """
    if current_version not in FC_FAULT_RISK_VERSION_DICT:
        return False, ''

    need_install_patch = FC_FAULT_RISK_VERSION_DICT.get(
        current_version)
    # 如果热补丁版本匹配则通过
    if current_patch != '--' and current_patch >= need_install_patch:
        return False, ''

    cmd = "show initiator initiator_type=FC"
    cli_ret = cli.execCmdNoLogTimout(cmd, 10*60)
    ini_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
    normal_count = 0
    for ini_dict in ini_dict_list:
        if ini_dict.get("Running Status", '') == "Online":
            normal_count += 1

    is_all_normal = True
    cmd = "show controller general |filterColumn include " \
          "columnList=Health\sStatus,Running\sStatus,Controller"
    cli_ret = cli.execCmdNoLogTimout(cmd, 10*60)
    controller_dict_list = cliUtil.getVerticalCliRet(cli_ret)
    for controller_dict in controller_dict_list:
        if controller_dict.get("Health Status", '') != "Normal":
            is_all_normal = False
            break
        if controller_dict.get("Running Status", '') != "Online":
            is_all_normal = False
            break

    logger.logInfo("Initiator Count:%s, isAllCtrlNormal:%s" % (
        normal_count, is_all_normal))

    if normal_count <= 200 and is_all_normal:
        return False, ''

    return True, need_install_patch


def is_support_read_only_user_enter_debug(product_version, hot_patch):
    """
    判断是否支持只读用户进去debug,如果支持返回true
    :param product_version:
    :param hot_patch:
    :return:
    """
    # 支持的版本列表，后续可以在此列表中新增
    support_read_only_version = {'V300R006C20': 30, 'V500R007C61': 30}
    if product_version in support_read_only_version:
        support_patch = support_read_only_version.get(product_version)

        pattern_hot_patch = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
        match_hot_path = pattern_hot_patch.search(hot_patch)
        if match_hot_path and support_patch <= int(match_hot_path.group(1)):
            return True
    return False


def merge_result(not_pass, suggestion, no_check, ret_list, env):
    """
    合并结果
    :param not_pass: 不通过
    :param suggestion: 建议优化
    :param no_check: 未完成检查
    :param ret_list: 回文列表
    :param env: 上下文
    :return:
    """
    not_pass = list(set(not_pass))
    suggestion = list(set(suggestion))
    no_check = list(set(no_check))
    ret = "\n".join(ret_list)
    error_msg = "".join(not_pass + suggestion + no_check)
    if not_pass:
        return False, ret, error_msg

    if suggestion:
        return cliUtil.RESULT_WARNING, ret, error_msg

    if no_check:
        return get_result_bureau(env, ret, error_msg)

    return True, ret, ''


def get_patch_value(hot_patch):
    """
    获取补丁的值
    :param hot_patch: 补丁
    :return: 补丁的数字版本
    """
    pattern_hot_patch = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
    match_hot_path = pattern_hot_patch.search(hot_patch)
    if match_hot_path:
        return int(match_hot_path.group(1))
    return 0


def check_conn_wrap(call_func):
    """
    1. 检查是否18000设备,如果不能连接设备，提前拦截报错。
    2. 最后重连工作
    :param call_func:
    :return:
    """
    def wrap(instance_obj):
        conn_cli = ''
        env = instance_obj.env
        logger = instance_obj.local_logger
        cli = instance_obj.cli
        lang = common.getLang(env)
        try:
            flag, conn_cli, msg = common.createDeviceCliContFor18000(
                cli, env, logger, lang)
            if flag is not True:
                return CheckStatus.NOCHECK, msg
            return call_func(instance_obj)
        except Exception:
            logger.logInfo(traceback.format_exc())
            return CheckStatus.NOCHECK, common.getMsg(
                lang, "query.result.abnormal")
        finally:
            if conn_cli != "":
                if common.is18000(env, conn_cli) and conn_cli is not cli:
                    common.closeConnection(conn_cli, env, logger)
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(cli, lang)
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(cli, logger)
    return wrap


def is_super_administrator(cli, user_name, lang):
    """
    是否超级管理员角色
    :param cli:
    :param user_name:
    :param lang:
    :return: True: 是超级管理员， False: 非超级管理员
    """
    cmd = "show user user_name=%s" % user_name
    flag, ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        raise UnCheckException(msg, ret)
    user_dict_list = cliUtil.getHorizontalCliRet(ret)
    for user_item in user_dict_list:
        # 兼容老版本不存在“Role ID”场景"Super_admin"
        if user_item.get("Role ID") == '1':
            return True
        if user_item.get("Level") == "Super_admin":
            return True
    return can_enter_debug_mode_for_read_only_user(cli, lang, None)


def can_enter_debug_mode_for_read_only_user(cli, lang, logger):
    """
    判断是否支持只读用户采集
    :param params_dict:
    :return: True 非超级管理员并且支持只读用户采集
    """
    flag, product_version, hot_patch, _ = get_system_version_with_ret(
        cli, lang
    )
    if not is_support_read_only_user_enter_debug(product_version, hot_patch):
        return False
    flag, _, level, role_id, _ = get_user_level_and_role_id(cli, lang)
    if not flag:
        if logger:
            logger.logError("Query user privilege failed, check terminated!")
        return False
    if logger:
        logger.logInfo(
            "Current user level is {},role_id is {}.".format(level, role_id))
    if level == "Admin" and role_id == "10":
        return True

    return False


def is_hypermetro_work_mode(domain_info):
    """
    是否是双活模式的domain
    hypermetro: 双活模式
    其他值：非双活模式（Synchronous 同步模式）
    因TR5版本没有同步模式，所以没有work mode字段。TR6新增的work mode字段。
    所以：当不存在work mode字段时，默认双活模式。
    当存在work mode字段，但值为Hypermetro，则为双活模式，否则不为双活模式
    :param domain_info:
    :return:
    """
    work_mode = domain_info.get("Work Mode", '')
    return not work_mode or work_mode.lower() == 'hypermetro'


def get_hyper_metro_pair_id_list(dev_sn, domain_id, env, logger, lang):
    """
    @summary: 获取双活pair
    @return: pair_id_list 双活pair ID列表
    """
    import common_cache
    flag, ret, msg, pair_list = common_cache.get_san_pair_from_cache(
        env, logger, dev_sn, lang)
    if flag is not True:
        logger.logInfo("Failed to get information about HyperMetro Pair")
        raise UnCheckException(msg)

    pair_info_list = []
    pair_id_list = []
    for pair_info in pair_list:
        if pair_info.get("Domain ID") == domain_id:
            pair_info_list.append(pair_info)
            pair_id_list.append(pair_info.get("ID"))

    return pair_id_list, pair_info_list


def get_domain_info(env, dev_sn, logger, lang):
    """
    获取domain信息
    :param env: 上下文
    :param dev_sn: 设备SN
    :param logger: 日志句柄
    :param lang: 语言
    :return:
    """
    domain_dict = {}
    cmd = "show hyper_metro_domain general"
    flag, cli_ret, err_msg = common.getObjFromFile(
        env, logger, dev_sn, cmd, lang)
    if flag is not True:
        raise UnCheckException(err_msg, cli_ret)

    res_list = cliUtil.getHorizontalCliRet(cli_ret)
    for domain_info in res_list:
        domain_id = domain_info.get("ID", '')
        remote_device_id = domain_info.get("Remote Device ID", '')
        domain_dict[domain_id] = remote_device_id

    return domain_dict


def get_remote_device_info(env, dev_sn, logger, lang):
    """
    获取远端设备和domain的对应关系
    :param env: 上下文
    :param dev_sn: 设备SN
    :param logger: 日志句柄
    :param lang: 语言
    :return:
    """
    remote_device_dict = {}
    domain_dict = get_domain_info(env, dev_sn, logger, lang)
    if not domain_dict:
        return remote_device_dict

    cmd = "show remote_device general"
    flag, cli_ret, err_msg = common.getObjFromFile(env, logger, dev_sn, cmd, lang)
    if flag is not True:
        raise UnCheckException(err_msg, cli_ret)

    res_list = cliUtil.getHorizontalCliRet(cli_ret)
    for remote_dev in res_list:
        remote_id = remote_dev.get("ID", '')
        remote_sn = remote_dev.get("SN", '')
        for domain_id in domain_dict:
            if remote_id == domain_dict.get(domain_id):
                remote_device_dict[remote_sn] = domain_id

    return remote_device_dict


def not_support_nas_domain(ret):
    """
    判断不支持NAS双活
    :param ret:
    :return: True：不支持，False 支持
    """
    if not cliUtil.hasCliExecPrivilege(ret):
        return True
    return "the current device does not support nas" in str(ret).lower()


def get_internal_product_model(dev_obj):
    """
    查询内部产品型号
    :param dev_obj:
    :return:
    """
    lang = str(dev_obj.get("lang"))
    rest = contextUtil.getRest(contextUtil.getContext(dev_obj))
    err_msg = cliUtil.getMsg(lang, "failed.to.get.internal.product.model")
    params = {}
    param_dict = restUtil.CommonRest.getUriParamDict(restData.RestCfg.SpecialUri.INTERNAL_DEVICE_INFO)
    record = restUtil.CommonRest.execCmd(rest, param_dict, params, restData.RestCfg.RestMethod.GET)
    ret = "{}\n{}".format(restData.RestCfg.SpecialUri.INTERNAL_DEVICE_INFO, str(record))
    if not record:
        # 如果取得为空的话，直接返回None
        return err_msg, None, ret
    else:
        internal_model = restUtil.CommonRest.getRecordValue(
            restUtil.CommonRest.getData(record), restData.InternalDeviceInfo.INTERNAL_PRODUCT_MODEL
        )
        return err_msg, internal_model, ret


def is_flush_through_mode(cli, lang, logger):
    """
    查询微存储设备是否为直通模式。
    @return:直通模式返回True,否则为False，
    """
    logger.logInfo("query disk rw Mode start.")
    cmd = "show system work_mode"
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        logger.logWarning("excute cmd show system work_mode failed.")
        return False
    if "System mode: pass-through mode" in cli_ret:
        return True
    return False


def get_dpa_rest_record(dpa_rest, url):
    """
    获取dpa巡检rest返回值
    :param dpa_rest:
    :param url:
    :return:
    """
    response_info = dpa_rest.execGet(dpa_rest.getBaseUrl() + url)
    data_string = response_info.getContent()
    return json.loads(data_string)
