# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.
import re

import cliUtil
import common
import expandconfig
from cbb.common.conf.productConfig import CHECK_DEV_CAPACITY
from cbb.frame.base import baseUtil

# 字典存取key值
ID_KEY = "ID"  # ID
CONTROLLER_KEY = "Controller"  # 所属控制器
CONTROLLER_CACHE_KEY = "Cache Capacity"  # 控制器内存大小
DOMAIN_ID_KEY = "Disk Domain ID"  # 硬盘域ID
TOTAL_CAPACITY_KEY = "Total Capacity"  # 总容量
TOTAL_CAPACITY_AFTER_KEY = "total_capacity_after"  # 扩容后总容量
SUBSCRIBED_CAPACITY_KEY = "Subscribed Capacity"  # 已写入容量
LUN_SUBSCRIBED_CAPACITY_KEY = "LUN Subscribed Capacity"  # LUN已写入容量
ON_CTRL_LIST_KEY = "on_ctrl_list"  # 归属控制器字符串
ON_LOGIC_ENG_LIST_KEY = "on_logic_eng"  # 归属逻辑引擎列表
EST_MAX_CAP_KEY = "estimate_max_capacity"  # 预估最大写入容量
FACT_CAP_KEY = "fact_capacity"  # 实际写入容量

# 扩容配置的硬盘容量加入存储池总容量的折算比
EXP_DISK_CAPACITY_COEFFICIENT = 0.5
# 控制器内存单位转换进制
CTRL_CACHE_CAPACITY_COEFFICIENT = 1024
# 比值：80%
EIGHTY_PERCENT = 0.8
# 比值：50%
FIFTY_PERCENT = 0.5

# DoradoV3控制框高度
CTRL_HEIGHT_PDT_MAP = {
    2: ["Dorado3000 V3", "Dorado5000 V3"],
    3: ["Dorado6000 V3", ],
    6: ["Dorado18000 V3", ],
}


class ExpDiskCapacityCheck:
    """扩容评估有效容量超规格检查类
    """

    def __init__(self, py_java_env, cli, lang, logger):
        self._py_java_env = py_java_env
        self._cli = cli
        self._lang = lang
        self._logger = logger
        self._product_model = str(py_java_env.get("devInfo").getDeviceType())
        self._pdt_vrc_version = baseUtil.getVRCVersion(
            str(py_java_env.get("devInfo").getProductVersion()))
        self._ctrl_height = 0  # 控制框高度
        self._ctrl_cache_memory = 0  # 控制器内存大小
        self._dev_capacity = 0  # 用户有效容量
        self._origin_pool_info = tuple()  # 存储池原始信息字典的元组
        self._pool_info_list = []  # 可定义的存储池信息字典的列表，用于计算
        self._logic_eng_cap_dict = {}  # 逻辑引擎的容量信息字典
        self._direct_pass = False  # 是否直接通过检查
        self._flag = True  # 评估结果
        self._all_cli_ret = ""  # 回显记录
        self._err_msg = ""  # 错误消息
        self.init_ctrl_height()
        self.init_pool_info()
        self.init_ctrl_cache_info()
        self.init_dev_capacity()
        self.init_domain_eng_info()
        self.add_exp_cap_to_pool()

    def execute(self):
        """检查执行入口

        :return:
        """
        try:
            self._logger.logInfo(
                "Ctrl Enclosure height: %s" % self._ctrl_height)
            self._logger.logInfo(
                "Controller cache: %s, device capacity: %s" %
                (self._ctrl_cache_memory, self._dev_capacity))
            self._logger.logInfo("Pool info: %s" % str(self._pool_info_list))

            # 判断初始化结果，失败或直接通过，则返回
            if self._flag is not True or self._direct_pass:
                return

            # 以存储池为单位计算各容量
            self.calculate_on_pool()
            if self._flag is not True or self._direct_pass:
                return

            # 以逻辑引擎为单位计算各容量，判断是否超规格
            self.calculate_on_logic_eng()
            return
        except Exception as ex:
            self._flag = cliUtil.RESULT_NOCHECK
            self._err_msg = common.getMsg(self._lang, "query.result.abnormal")
            self._logger.logException(ex)
            return

    def init_ctrl_height(self):
        """初始化控制框高度信息

        :return:
        """
        for height, pdt_list in CTRL_HEIGHT_PDT_MAP.items():
            if self._product_model in pdt_list:
                self._ctrl_height = height
                break
        return

    def init_ctrl_cache_info(self):
        """初始化控制器内存大小

        :return:
        """
        cmd = "show controller general"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self._cli, cmd, True, self._lang)
        self._all_cli_ret = common.joinLines(self._all_cli_ret, cli_ret)
        ctrl_info_list = cliUtil.getVerticalCliRet(cli_ret)
        # 命令执行成功，且回显解析成功
        if flag is True and ctrl_info_list:
            # 所有控制器内存大小一致，取第一个即可
            ctrl_cache_memory = ctrl_info_list[0].get(CONTROLLER_CACHE_KEY, "")

            # 内存保留整数，TB转为GB
            if re.search("GB", ctrl_cache_memory) and "." in ctrl_cache_memory:
                self._ctrl_cache_memory = int(ctrl_cache_memory.split(".")[0])
                return
            if re.search("TB", ctrl_cache_memory) and "." in ctrl_cache_memory:
                self._ctrl_cache_memory = int(ctrl_cache_memory.split(".")[0])
                self._ctrl_cache_memory *= CTRL_CACHE_CAPACITY_COEFFICIENT
            return
        else:
            self._flag = cliUtil.RESULT_NOCHECK
            err_msg = common.getMsg(self._lang, "cannot.get.controller.info")
            self._err_msg = common.joinLines(self._err_msg, err_msg)
            return

    def init_pool_info(self):
        """初始化存储池相关信息

        :return:
        """
        cmd = r"show storage_pool general|filterColumn include columnList=" \
              r"ID,Disk\sDomain\sID,Total\sCapacity,Subscribed\sCapacity," \
              r"LUN\sSubscribed\sCapacity"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self._cli, cmd, True, self._lang)
        self._all_cli_ret = common.joinLines(self._all_cli_ret, cli_ret)
        pool_info_list = cliUtil.getHorizontalCliRet(cli_ret)
        # 如果系统中不存在存储池，则直接通过
        if cliUtil.queryResultWithNoRecord(cli_ret):
            self._direct_pass = True
            return
        # 命令执行失败，或回显解析失败
        if flag is not True or not pool_info_list:
            self._flag = cliUtil.RESULT_NOCHECK
            err_msg = common.getMsg(self._lang, "can.not.get.pool.detail")
            self._err_msg = common.joinLines(self._err_msg, err_msg)
            return

        pool_info_tuple = tuple(pool_info_list)
        self._origin_pool_info = pool_info_tuple
        # 容量进行单位转换并保存新的信息字典
        for pool_info in pool_info_tuple:
            diy_pool_info = {}
            flag0, total_cap = common.changUnit2GB(
                pool_info.get(TOTAL_CAPACITY_KEY, ""), isLabelCap=True)
            flag1, sub_cap = common.changUnit2GB(
                pool_info.get(SUBSCRIBED_CAPACITY_KEY, ""), isLabelCap=True)
            flag2, lun_sub_cap = common.changUnit2GB(
                pool_info.get(LUN_SUBSCRIBED_CAPACITY_KEY, ""),
                isLabelCap=True)
            # 容量全部转换成功
            if flag0 and flag1 and flag2:
                diy_pool_info[ID_KEY] = pool_info.get(ID_KEY, "")
                diy_pool_info[DOMAIN_ID_KEY] = pool_info.get(DOMAIN_ID_KEY, "")
                diy_pool_info[TOTAL_CAPACITY_KEY] = total_cap
                diy_pool_info[SUBSCRIBED_CAPACITY_KEY] = sub_cap
                diy_pool_info[LUN_SUBSCRIBED_CAPACITY_KEY] = lun_sub_cap
                self._pool_info_list.append(diy_pool_info)
            else:
                self._logger.logNoPass(
                    "Fail to change pool capacity.ID: %s, result: %s, %s, %s."
                    % (pool_info.get(ID_KEY), flag0, flag1, flag2))
                self._flag = cliUtil.RESULT_NOCHECK
                err_msg = common.getMsg(self._lang, "can.not.get.pool.detail")
                self._err_msg = common.joinLines(self._err_msg, err_msg)
                return

        return

    def init_dev_capacity(self):
        """初始化用户有效容量

        :return:
        """
        self._dev_capacity = CHECK_DEV_CAPACITY.get(
            self._product_model, {}).get(self._ctrl_cache_memory, {}).get(
            self._pdt_vrc_version, 0)
        # 有效容量获取失败
        if self._dev_capacity == 0:
            self._logger.logNoPass("Fail to get device capacity.")
            self._flag = cliUtil.RESULT_NOCHECK
            err_msg = common.getMsg(self._lang, "query.result.abnormal")
            self._err_msg = common.joinLines(self._err_msg, err_msg)
        return

    def add_exp_cap_to_pool(self):
        """将扩容配置中容量加到存储池总容量中
            硬盘数量 * 硬盘容量 * 折算系数

        :return:
        """
        # 扩容配置中硬盘域新增硬盘总容量字典
        exp_domain_cap_dict = {}
        exp_disk_dict = \
            expandconfig.ExpandConfig(self._py_java_env).getExpDiskInfo()

        for domain_id, exp_obj_list in exp_disk_dict.items():
            exp_domain_cap = 0
            for exp_obj in exp_obj_list:
                disk_cap = exp_obj.getDiskCapacity()
                disk_num = exp_obj.getDiskNum()
                all_disk_cap = \
                    disk_cap * EXP_DISK_CAPACITY_COEFFICIENT * disk_num
                exp_domain_cap += all_disk_cap

            exp_domain_cap_dict[domain_id] = exp_domain_cap
        self._logger.logInfo("new expand domain capacity dict is: %s" %
                             str(exp_domain_cap_dict))

        # 将硬盘域新扩容量与原有存储池总容量相加，用新的key保存
        for pool_info in self._pool_info_list:
            domain_id = pool_info.get(DOMAIN_ID_KEY, "")
            pool_info[TOTAL_CAPACITY_AFTER_KEY] = \
                pool_info.get(TOTAL_CAPACITY_KEY, 0) + \
                exp_domain_cap_dict.get(domain_id, 0)
        return

    def init_domain_eng_info(self):
        """初始化存储池归属引擎信息

        :return:
        """
        cmd = "show disk_domain general|filterColumn include " \
              "columnList=ID,Controller"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self._cli, cmd, True, self._lang)
        self._all_cli_ret = common.joinLines(self._all_cli_ret, cli_ret)
        domain_ctrl_info_list = cliUtil.getHorizontalCliRet(cli_ret)
        if flag is not True or not domain_ctrl_info_list:
            self._flag = cliUtil.RESULT_NOCHECK
            err_msg = common.getMsg(self._lang, "cannot.get.domain.info")
            self._err_msg = common.joinLines(self._err_msg, err_msg)
            return

        for domain_ctrl_info in domain_ctrl_info_list:
            domain_id = domain_ctrl_info.get(ID_KEY, "")
            on_ctrl_string = domain_ctrl_info.get(CONTROLLER_KEY, "")
            for pool_info in self._pool_info_list:
                if domain_id == pool_info.get(DOMAIN_ID_KEY):
                    # 归属控制器列表
                    on_ctrl_list = on_ctrl_string.split(",")
                    # 获取到归属逻辑引擎列表
                    on_logic_eng_list = baseUtil.get_logic_eng_id(
                        self._ctrl_height, on_ctrl_list)
                    pool_info[ON_LOGIC_ENG_LIST_KEY] = on_logic_eng_list
                    break

    def calculate_on_pool(self):
        """计算存储池在对应引擎的LUN预估最大写入容量、LUN实际写入容量
           LUN预估最大写入容量：LUN Subscribed Capacity / Subscribed Capacity *
                              Total Capacity / ENGINE_ID_NUM
           LUN实际写入容量：LUN Subscribed Capacity / ENGINE_ID_NUM
        :return:
        """
        for pool_info in self._pool_info_list:
            lun_sub_cap = pool_info.get(LUN_SUBSCRIBED_CAPACITY_KEY)
            sub_cap = pool_info.get(SUBSCRIBED_CAPACITY_KEY)
            total_cap = pool_info.get(TOTAL_CAPACITY_AFTER_KEY)
            on_logic_eng_list = pool_info.get(ON_LOGIC_ENG_LIST_KEY)
            if lun_sub_cap is not None \
                    and sub_cap is not None \
                    and total_cap \
                    and on_logic_eng_list:
                # 存储池已写入容量为0，跳过，继续计算下一个存储池
                if not lun_sub_cap or not sub_cap:
                    pool_info[EST_MAX_CAP_KEY] = 0
                    pool_info[FACT_CAP_KEY] = 0
                    continue
                # LUN预估最大写入容量
                estimate_max_capacity = \
                    lun_sub_cap / sub_cap * total_cap / len(on_logic_eng_list)
                # LUN实际写入容量
                fact_capacity = lun_sub_cap / len(on_logic_eng_list)
                pool_info[EST_MAX_CAP_KEY] = estimate_max_capacity
                pool_info[FACT_CAP_KEY] = fact_capacity
            else:
                self._logger.logNoPass("Calculate capacity on pool error. "
                                       "Pool info: %s" % str(pool_info))
                self._flag = cliUtil.RESULT_NOCHECK
                err_msg = common.getMsg(self._lang, "query.result.abnormal")
                self._err_msg = common.joinLines(self._err_msg, err_msg)
                break
        return

    def calculate_on_logic_eng(self):
        """逻辑引擎的LUN预估最大写入容量: 所有存储池在此引擎的LUN预估最大写入容量之和
           逻辑引擎的LUN实际写入容量: 所有存储池在此引擎的LUN实际写入容量之和

           若任一逻辑引擎的LUN实际写入容量 / 用户有效容量 > 80%，则检查不通过。
           若任一逻辑引擎的LUN实际写入容量 / 用户有效容量 > 50%，\
           且逻辑引擎的LUN预估最大写入容量 / 用户有效容量 > 80%，则检查不通过，
           否则检查通过。

        :return:
        """
        for pool_info in self._pool_info_list:
            estimate_max_capacity = pool_info.get(EST_MAX_CAP_KEY, 0)
            fact_capacity = pool_info.get(FACT_CAP_KEY, 0)
            on_logic_eng_list = pool_info.get(ON_LOGIC_ENG_LIST_KEY)
            if on_logic_eng_list:
                for logic_eng in on_logic_eng_list:
                    self._logic_eng_cap_dict.setdefault(logic_eng, {})
                    eng_cap_info = self._logic_eng_cap_dict.get(logic_eng)
                    on_ctrl_list = \
                        baseUtil.get_ctrl_by_logic_eng_id(self._ctrl_height,
                                                          logic_eng)
                    on_ctrl_list.sort()
                    # 单个逻辑引擎对应的控制器信息
                    eng_cap_info[ON_CTRL_LIST_KEY] = \
                        "[{}]".format(",".join(on_ctrl_list))
                    # 累加当前存储池在该逻辑引擎上的预估最大写入容量
                    eng_cap_info[EST_MAX_CAP_KEY] = \
                        eng_cap_info.get(EST_MAX_CAP_KEY, 0) + \
                        estimate_max_capacity
                    # 累加当前存储池在该逻辑引擎上的实际最大写入容量
                    eng_cap_info[FACT_CAP_KEY] = \
                        eng_cap_info.get(FACT_CAP_KEY, 0) + fact_capacity
            else:
                self._logger.logNoPass(
                    "Calculate capacity on logic engine error. "
                    "Pool info: %s." % str(pool_info))
                self._flag = cliUtil.RESULT_NOCHECK
                err_msg = common.getMsg(self._lang, "query.result.abnormal")
                self._err_msg = common.joinLines(self._err_msg, err_msg)
                return
        self._logger.logInfo("Capacity on all logic engine dict is: %s" %
                             str(self._logic_eng_cap_dict))

        # 超规格的控制器列表
        over_cap_ctrl_list = []
        for logic_eng, eng_cap_info in self._logic_eng_cap_dict.items():
            on_ctrl_list = eng_cap_info.get(ON_CTRL_LIST_KEY)
            est_max_cap_on_eng = eng_cap_info.get(EST_MAX_CAP_KEY)
            fact_cap_on_eng = eng_cap_info.get(FACT_CAP_KEY)
            if (fact_cap_on_eng / self._dev_capacity) > EIGHTY_PERCENT:
                over_cap_ctrl_list.append(on_ctrl_list)
            if (fact_cap_on_eng / self._dev_capacity) > FIFTY_PERCENT and \
                    (est_max_cap_on_eng / self._dev_capacity) > EIGHTY_PERCENT:
                over_cap_ctrl_list.append(on_ctrl_list)

        if over_cap_ctrl_list:
            over_cap_ctrl_list = list(set(over_cap_ctrl_list))
            over_cap_ctrl_list.sort()
            self._flag = False
            if self._lang == "en":
                join_flag = ","
            else:
                join_flag = u"，"
            err_msg = common.getMsg(self._lang, "check.capacity.too.high",
                                    join_flag.join(over_cap_ctrl_list))
            self._err_msg = common.joinLines(self._err_msg, err_msg)
        return

    def get_eval_result(self):
        """获取评估结果

        :return:
        """
        return self._flag, self._all_cli_ret, self._err_msg
