# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2012-. All rights reserved.

import re
import os
from memory_inspect.rule import memory_rule


class RuleEngine(object):
    def __init__(self, rules, logger):
        self.rules = rules
        self.logger = logger
        self.origin_info = {}

    def execute(self, rule_context):
        """
        根据配置文件中的规则和表达式，选择满足条件的表达式执行，获取执行结果
        :param rule_context: 规则执行上下文
        :return: 执行结果（是否命中，错误信息， 原始信息）
        """
        express_rules = self._get_express_rules()
        for express_rule_name in express_rules:
            express_rule_params = self.rules.get(express_rule_name)
            condition_name = express_rule_params.get("condition")

            if condition_name not in self.rules:
                self.logger.info("use default express.")
                return self._run_express(express_rule_params.get("express"), rule_context)

            condition_params = self.rules.get(condition_name)
            rule_obj = getattr(memory_rule, self._get_fun_name(condition_name))(rule_context)
            if rule_obj.execute(**condition_params):
                self.logger.info("use condition express. condition name={}".format(condition_name))
                return self._run_express(express_rule_params.get("express"), rule_context)
        raise Exception("rule configure error.")

    def _run_express(self, express_str, rule_context):
        """
        在规则上下文中，运行特定的表达式
        :param express_str: 带执行的表达式
        :param rule_context: 规则上下文
        :return: 表达式执行结果
        """
        err_msg = []
        all_rules = re.findall(r"([\w_]+)", express_str)
        for rule_name in all_rules:
            rule_params = self.rules.get(rule_name)
            if not rule_params:
                continue
            rule_obj = getattr(memory_rule, self._get_fun_name(rule_name))(rule_context)
            is_rule_hit = rule_obj.execute(**rule_params)
            self.origin_info[rule_name] = rule_obj.origin_info
            self.logger.info("run rule={}, result={}".format(rule_name, is_rule_hit))
            if is_rule_hit:
                err_msg.extend(rule_obj.err_msg)
            express_str = express_str.replace(rule_name, str(is_rule_hit))
        # eval 仅用于执行表达式的值，没有外部输入值，所有参数都是可信的，不存在注入风险
        ret = eval(express_str)
        if ret:
            return True, os.linesep.join(err_msg), self.origin_info
        return False, "", self.origin_info

    @staticmethod
    def _get_fun_name(rule_name):
        """
        根据配置的规则名称，获取执行的函数名称（取_前半部分）
        :param rule_name: 规则名称
        :return: 在memory_rule中定义的函数名称，便于更加函数名称反射执行
        """
        return rule_name.split("_")[0]

    def _get_express_rules(self):
        """
        获取表达式的配置
        :return: [express_1, express_2，...]
        """
        express_rules = []
        for rule_name, rule_params in self.rules.items():
            if "express_" in rule_name:
                express_rules.append(rule_name)
        return express_rules
