# -*- coding: UTF-8 -*-
import time
import cliUtil
import common
from common_utils import get_err_msg

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env


def execute(cli):
    """
    成员盘重构状态检查
    """
    member_disk_reconstruction_check = MemberDiskReconstructionCheck(cli)
    flag, err_msg = member_disk_reconstruction_check.execute_check()
    return \
        flag, "\n".join(member_disk_reconstruction_check.cli_ret_list), err_msg


class MemberDiskReconstructionCheck:
    def __init__(self, cli):
        self.cli = cli
        self.LANG = LANG
        self.LOGGER = LOGGER
        self.cli_ret_list = []
        # 检查次数，避免重构盘状态未刷新
        self.CHECK_TIMES = 2
        self.abnormal_ids = []

    def execute_check(self):
        try:
            # 获取另外一个引擎的node_id
            another_engine_node_id = self.get_node_id_on_another_engine()
            LOGGER.logInfo(
                "The another_engine_node_id: {}".format(another_engine_node_id))

            for i in range(self.CHECK_TIMES):
                # 获取重构状态的盘ID
                disk_ids = self.get_reconstruction_disk()

                another_engine_disk_ids = []
                if another_engine_node_id:
                    another_engine_disk_ids = \
                        self.get_reconstruction_disk_by_node_id(
                            another_engine_node_id[0])
                disk_ids.extend(another_engine_disk_ids)
                if not disk_ids:
                    return True, ""

                # 检查重构盘的健康状态
                self.abnormal_ids = \
                    self.check_reconstruction_disk_status(disk_ids)
                if not self.abnormal_ids:
                    return True, ""
                time.sleep(10)

            return False, get_err_msg(
                LANG,
                "reconstruction.disk.status.abnormal",
                ",".join(self.abnormal_ids))

        except common.UnCheckException as e:
            LOGGER.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            err_msg = common.getMsg(self.LANG, "query.result.abnormal")
            return cliUtil.RESULT_NOCHECK, err_msg

    def get_reconstruction_disk(self):
        """
        获取重构状态下的盘
        """
        cmd = "show disk_mgr disk_list"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInDeveloper(self.cli, cmd, True, self.LANG)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)
        ret_data = cliUtil.getHorizontalCliRet(cli_ret)
        LOGGER.logInfo("The ret_data: {}".format(ret_data))
        disk_ids = []
        if not ret_data:
            return disk_ids
        for item in ret_data:
            reconstruct_finish = item.get("Reconstruct Finish", "")
            if reconstruct_finish == "yes":
                disk_ids.append(item.get("User Disk ID", ""))
        return disk_ids

    def check_reconstruction_disk_status(self, disk_ids):
        """
        检查重构状态盘的健康状态,当重构盘的Health Status为normal时，状态为异常
        """
        cmd = "show disk general"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInDeveloper(self.cli, cmd, True, self.LANG)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)
        ret_data = cliUtil.getHorizontalCliRet(cli_ret)
        LOGGER.logInfo("The ret_data: {}".format(ret_data))
        abnormal_disk_ids = []
        if not ret_data:
            return abnormal_disk_ids
        for item in ret_data:
            disk_id = item.get("ID")
            health_status = item.get("Health Status", "")
            if disk_id in disk_ids and health_status == "Normal":
                abnormal_disk_ids.append(disk_id)
        return abnormal_disk_ids

    def get_node_id_on_another_engine(self):
        """
        获取另外一个引擎上的node_id,如果不存在，返回 “ ”
        """
        flag, err_msg, cli_ret, current_ctrl, \
            engine_ctrl_mapping_dict, current_engine, node_num = \
            common.getEngineCtrlNodeInfo(self.cli, self.LOGGER, self.LANG)
        LOGGER.logInfo("The current_ctrl: {}".format(current_ctrl))
        LOGGER.logInfo(
            "The engine_ctrl_mapping_dict: {}".format(engine_ctrl_mapping_dict))
        LOGGER.logInfo("The current_engine: {}".format(current_engine))
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)

        for engine, node_ids in engine_ctrl_mapping_dict.items():
            if str(engine) != str(current_engine):
                return node_ids
        return []

    def get_reconstruction_disk_by_node_id(self, node_id):
        """
        通过node_id获取重构状态下的盘
        """
        # 心跳到指定node_id的控制器
        flag, cli_ret, err_msg = common.heartBeatToOtherCtrl(self.cli, node_id,
                                                             PY_JAVA_ENV,
                                                             self.LOGGER,
                                                             self.LANG)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            self.LOGGER.logError('SSH to remote controller failed.')
            raise common.UnCheckException(err_msg, cli_ret)
        # 获取重构状态下的盘
        disk_ids = self.get_reconstruction_disk()
        cliUtil.exitHeartbeatCli(self.cli, self.LANG)
        return disk_ids
