# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import time
from datetime import datetime

import cliUtil
import common
import dpa_common
from java.lang import Exception as JException
import com.huawei.ism.tool.obase.entity.EntityUtils as EntityUtils

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)
ENVIRONMENT_FLAG = "no such user"


def execute(dpa_rest):
    check_item = CheckItem(dpa_rest, LANG, LOGGER)
    return check_item.execute()


class CheckItem:
    def __init__(self, dpa_rest, lang, logger):
        self.dpa_rest = dpa_rest
        self.lang = lang
        self.logger = logger
        self.err_msg = str()
        self.cli_ret = list()
        self.date_list = list()
        self.time_zone = set()

    @staticmethod
    def need_check_dev():
        cur_device_ip = PY_JAVA_ENV.get("devInfo").getIp()
        return [dev_node
                for dev_node in PY_JAVA_ENV.get('selectDevs')
                if dev_node.getIp() in list(PY_JAVA_ENV.get('dpaRelationshipHost').get(cur_device_ip))]

    def execute(self):
        need_check_dev_list = CheckItem.need_check_dev()
        self.cli_ret.append(str(PY_JAVA_ENV.get('dpaRelationshipHost').get(PY_JAVA_ENV.get("devInfo").getIp())))
        self.get_date_and_time_zone_info(need_check_dev_list)
        if self.err_msg:
            return False, "\n".join(self.cli_ret), self.err_msg
        # 环境仅一个节点，则环境为单机环境。升级前检查场景直接通过，开局场景返回人工确认建议。
        if len(need_check_dev_list) == 1:
            if PY_JAVA_ENV.get("isSelectBureauCheck"):
                return cliUtil.RESULT_WARNING, "\n".join(self.cli_ret), dpa_common.get_dpa_msg(self.lang,
                                                                                               "dpa.manually.determine")
            return True, "\n".join(self.cli_ret), ""
        self.check_result()
        if self.err_msg:
            return False, "\n".join(self.cli_ret), self.err_msg
        if PY_JAVA_ENV.get("isSelectBureauCheck"):
            return cliUtil.RESULT_WARNING, "\n".join(self.cli_ret), dpa_common.get_dpa_msg(self.lang,
                                                                                           "dpa.manually.determine")
        return True, "\n".join(self.cli_ret), ""

    def check_result(self):
        if not self.date_list:
            self.err_msg += dpa_common.get_dpa_msg(self.lang, "query.result.abnormal")
            return
        # 各个节点时间的最大值与最小值之差超过60s则报错
        if (max(self.date_list) - min(self.date_list)).total_seconds() > 60:
            self.err_msg += dpa_common.get_dpa_msg(self.lang, "dpa.time.difference.abnormal")
        # 各个节点时区不一致则报错
        if len(self.time_zone) > 1:
            self.err_msg += dpa_common.get_dpa_msg(self.lang, "dpa.time.zone.inconsistent.abnormal")

    def get_date_and_time_zone_info(self, need_check_dev_list):
        for dev_node in need_check_dev_list:
            connection = None
            try:
                dev_node = EntityUtils.cloneDevNode(dev_node)
                device_sn = dev_node.getIp() + str(time.time())
                dev_node.setDeviceSerialNumber(device_sn)
                connection = PY_JAVA_ENV.get("sshManager").getSshConnection(dev_node)
                self.query_date_info(connection)
                self.query_time_zone_info(connection)
            except JException as exp:
                self.err_msg += dpa_common.get_dpa_msg(LANG, "dpa.node.connection.abnormal", dev_node.getIp())
                self.logger.logNoPass("Create SSH {} connection catch exception:{}".format(dev_node.getIp(), exp))
            finally:
                PY_JAVA_ENV.get("sshManager").releaseConnection(connection)
                self.logger.logInfo("SSH {} connection is release".format(dev_node.getDeviceSerialNumber()))

    def query_time_zone_info(self, connection):
        time_zone_ret = connection.execCmd("timedatectl |grep 'Time zone' |awk '{print $3}'")
        self.cli_ret.append(time_zone_ret)
        self.time_zone.add(time_zone_ret.encode("utf8").splitlines()[1])

    def query_date_info(self, connection):
        date_ret = connection.execCmd('date +"%F %T"')
        self.cli_ret.append(date_ret)
        standard_time = datetime.strptime(date_ret.encode("utf8").splitlines()[1], '%Y-%m-%d %H:%M:%S')
        self.date_list.append(standard_time)
