# -*- coding: UTF-8 -*-
import common
import common_utils
import cliUtil
from risk_version_config import BOM_INFO_OF_INNTER_MODEL_3P

from cbb.frame.cli import cliUtil as cbbCliUtil

from utils import Products

INNER_MODEL_1P = "1P"
INNER_MODEL_3P = "3P"

# 支持非内双活组网引擎间3P扩1P扩控的设备型号
PRODUCT_MODEL_TUPLE = ("OceanProtect X8000", "OceanProtect X8000K", "OceanProtect E8000")

PY_JAVA_ENV = py_java_env


def execute(cli):
    """
    支持x8000，x8000k非内双活组网引擎间3P扩1P扩控
    @param cli: cli连接
    """
    lang = common.getLang(PY_JAVA_ENV)
    logger = common.getLogger(PY_LOGGER, __file__)
    exp_3p_to_1p_check = ExpCtrl3pTo1pCheck(cli, lang, logger, PY_JAVA_ENV)
    exp_3p_to_1p_check.execute_check()
    return exp_3p_to_1p_check.get_check_result()


class ExpCtrl3pTo1pCheck:
    def __init__(self, cli, lang, logger, env):
        self.cli = cli
        self.lang = lang
        self.logger = logger
        self.env = env
        self.all_ret_list = []
        self.p_version = str(self.env.get("devInfo").getProductVersion())
        self.new_ctrl_inner_model = self.get_new_ctrl_type_in_inspect_scene()
        self.res_flag = True
        self.err_msg = ""

    def execute_check(self):
        if not self.is_support_exp_3p_to_1p(self.get_product_model()):
            self.res_flag = cliUtil.RESULT_NOSUPPORT
            return
        msg, ctrl_inner_model, ret = common_utils.get_internal_product_model(self.env)
        self.all_ret_list.append(ret)
        if not ctrl_inner_model:
            self.res_flag, self.err_msg = False, msg
            return
        old_ctrl_inner_mode = INNER_MODEL_1P if str(ctrl_inner_model).endswith(INNER_MODEL_1P) else INNER_MODEL_3P
        self.logger.logInfo("old ctrl inner mode:{}".format(old_ctrl_inner_mode))
        self.res_flag, self.err_msg = self.is_exp_process_compatible(old_ctrl_inner_mode)

    def get_check_result(self):
        return self.res_flag, self.get_ret(), self.err_msg

    def is_exp_process_compatible(self, old_ctrl_inner_mode):
        # 1P扩3P 不支持
        if self.is_exp_1p_to_3p(old_ctrl_inner_mode):
            return False, common_utils.get_err_msg(self.lang, "not.support.exp.controller.1p.to.3p")
        # 3P扩1P 继续检查
        if not self.is_exp_3p_to_1p(old_ctrl_inner_mode):
            return True, ""
        return self.check_origin_software_and_kernel_version()

    def check_origin_software_and_kernel_version(self):
        """
        1. 如果软件版本1.3.RC1之前，提示滚动升级到1.3.RC1，不通过
        2. 如果软件版本1.3.RC1及其之后，内核版本1.1.6.0之前，提示升级内核版本，不通过
        :return:True：支持；False:不支持
        """
        _, ret, kernel_version = cliUtil.get_kernel_version(self.cli, self.lang)
        self.all_ret_list.append(ret)
        kernel_version = "1.0.0.1" if not kernel_version else kernel_version
        err_msg_key = ""
        err_msg = ""
        if Products.compareVersion(self.p_version, "1.3.RC1") < 0:
            err_msg_key = "exp.controller.roll.upgrade.to.130"
            err_msg = common_utils.get_err_msg(self.lang, err_msg_key, self.p_version)
        elif kernel_version < '1.1.6.0':
            err_msg_key = "exp.controller.need.apollo.upgrade"
            err_msg = common_utils.get_err_msg(self.lang, err_msg_key, kernel_version)
        return not err_msg_key, err_msg

    def get_product_model(self):
        """
        检查设备型号
        """
        # 查询设备型号
        (flag, product_model, err_msg, cli_ret,) = cbbCliUtil.get_product_model_with_ret(self.cli, self.lang)
        self.all_ret_list.append(cli_ret)
        return product_model

    def is_support_exp_3p_to_1p(self, product_model):
        """
        是否支持3P扩1P扩控
        :return: True：支持，False：不支持
        """
        return product_model in PRODUCT_MODEL_TUPLE

    def get_new_ctrl_type_in_inspect_scene(self):
        """
        扩容评估场景下获取待扩集群内部型号
        @return: 内部型号
        """
        enc_bom = self.env.get("newEncBom")
        if not enc_bom:
            return ""

        bom_list = enc_bom.upper().split(",")
        self.logger.logInfo("exp ctrl bom_list: {}".format(bom_list))
        for bom in bom_list:
            if bom in BOM_INFO_OF_INNTER_MODEL_3P:
                return INNER_MODEL_3P
        return INNER_MODEL_1P

    def is_exp_1p_to_3p(self, origin_internal_model):
        """
        1P设备：内部型号中包含_1P，3P设备不包含_1P(不包含则默认为3P)，是否为1P扩3P
        :param origin_internal_model: 原扩内部型号
        :param new_internal_model: 待扩内部型号
        :return: 是否为1P扩3P
        """
        return origin_internal_model.endswith(INNER_MODEL_1P) and not self.new_ctrl_inner_model.endswith(INNER_MODEL_1P)

    def is_exp_3p_to_1p(self, origin_internal_model):
        """
        1P设备：内部型号中包含_1P，3P设备不包含_1P(不包含1P则默认为3P)，是否为3P扩1P
        :param origin_internal_model: 原扩内部型号
        :param new_internal_model: 待扩内部型号
        :return: 是否为3P扩1P
        """
        return not origin_internal_model.endswith(INNER_MODEL_1P) and self.new_ctrl_inner_model.endswith(INNER_MODEL_1P)

    def get_ret(self):
        return "\n".join(self.all_ret_list)
