# -*- coding: UTF-8 -*-
import re

import cliUtil
import common
from common_utils import get_err_msg

from cbb.frame.base import product
from cbb.frame.base import baseUtil

from com.huawei.ism.tool.obase.exception import ToolException


def execute(cli):
    """
    归一化至check_item_service_number_of_initiators_for_ports
    高端共享卡端口启动器数量超规格检查：
    1 若步骤2中系统Product Model为：OceanStor Dorado 18X00/8000 V6
    或者18X10（F）/6810（F） V5，则继续检查，否则检查结果为通过;
    2 若步骤2中系统软件版本为6.0.1且安装SPH2或之后版本热补丁，则检查结果为通过，
    否则继续检查;
    3 若步骤3中存在存在“FC-SCSI”或者“FC-NVME”记录，且对应
    “Number Of Initiators”值大于等于250，则检查结果为不通过，
    否则检查结果为通过。
    :param cli:cli链接
    :return:
    """
    ports_initiator_check = PortsInitiatorCheck(cli, py_java_env,
                                                common.getLang(py_java_env),
                                                common.getLogger(PY_LOGGER,
                                                                 __file__))
    check_result, err_msg = ports_initiator_check.check_controller()
    return check_result, "\n".join(ports_initiator_check.all_ret_list), err_msg


class PortsInitiatorCheck:
    def __init__(self, cli, env, lang, logger):
        self.cli = cli
        self.env = env
        self.lang = lang
        self.logger = logger
        self.all_ret_list = list()

    def check_controller(self):
        try:
            flag, product_model, _, cli_ret, err_msg = \
                common.getProductModelAndCurSysDate(self.cli, self.lang)
            self.all_ret_list.append(cli_ret)
            if flag is not True:
                return cliUtil.RESULT_NOCHECK, err_msg

            # 若步骤2中系统Product Model为：OceanStor Dorado 18X00/8000 V6
            #     或者18X10（F）/6810（F） V5，则继续检查，否则检查结果为通过;
            if not baseUtil.isV5V6HighEnd(product_model):
                return True, ""

            flag, soft_version, patch_ver, cli_ret_version, err_msg = \
                common.get_soft_and_patch_version(self.cli, self.logger,
                                                  self.lang)
            self.all_ret_list.append(cli_ret_version)
            if flag is not True:
                return cliUtil.RESULT_NOCHECK, err_msg
            # 检查是否在风险版本中
            if not self.is_risky_version(soft_version, patch_ver):
                return True, ""

            # 检查是否启动器超规格
            err_msg_list = self.check_number_of_initiators_among_fc_ports()
            if err_msg_list:
                return False, "".join(err_msg_list)
            return True, ""
        except common.UnCheckException as uncheck:
            return (cliUtil.RESULT_NOCHECK,
                    uncheck.errorMsg)
        except (ToolException, Exception) as exception:
            self.logger.logException(exception)
            return cliUtil.RESULT_NOCHECK, common.getMsg(
                self.lang, "query.result.abnormal")

    def is_risky_version(self, soft_version, patch_ver):
        """
        检查是否为风险版本，若步骤3中系统软件版本为V500R007C60SPC200 Kunpeng
        及之后版本，或6.0且低于6.0.1.SPH2，则继续检查，否则检查结果为通过;
        :param soft_version: 版本
        :param patch_ver: 补丁
        :return: True：是风险版本
        """
        product_version = str(self.env.get("devInfo").getProductVersion())
        if all([
            product.isKunpeng(product_version),
            'V500R007C60SPC200' <= soft_version < 'V500R007C70'
        ]):
            return True
        if product.isDigitalVer(product_version) and "6.0" in product_version:
            if "6.0.1" not in product_version:
                return True
            pattern_hot_patch = re.compile(r'SPH(\d+)',
                                           flags=re.IGNORECASE)
            match_hot_path = pattern_hot_patch.search(patch_ver)
            if match_hot_path and int(
                    match_hot_path.group(1)) >= 2:
                return False
            return True
        return False

    def check_number_of_initiators_among_fc_ports(self):
        """
            高端共享卡端口启动器数量检查：
            “Number Of Initiators”记录数值中有大于等于250，
            则检查结果为不通过，否则检查结果为通过。
        :return:
        """

        err_meg_list = list()

        cmd = "show port general physical_type=FC"
        flag, ret, msg = cliUtil.excuteCmdInCliMode(self.cli, cmd, True,
                                                    self.lang)
        self.all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)

        if cliUtil.queryResultWithNoRecord(ret):
            return err_meg_list

        cli_ret_lines_list = cliUtil.getHorizontalCliRet(ret)

        for info_dict in cli_ret_lines_list:

            initiator_num = info_dict.get("Number Of Initiators", "")
            protocol = info_dict.get("Protocol", "")
            # “Number Of Initiators”记录数值中有大于等于250

            if initiator_num and int(initiator_num) >= 250 and \
                    protocol in ["FC-SCSI", "FC-NVME"]:
                initiator_id = info_dict.get("ID", "")
                if initiator_id:
                    err_meg_list.append(
                        get_err_msg(self.lang, "number.of.initiators.not.pass",
                                    (initiator_id, initiator_num)))
        return err_meg_list
