# -*- coding: UTF-8 -*-
from __future__ import division
import decimal
import cliUtil
import common
from cbb.frame.cli.cli_with_cache import execute_cmd_in_cli_mode_with_cache
from com.huawei.ism.tool.obase.exception import ToolException

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)

def execute(cli):
    '''
            异步远程复制空间预留检查 ； 
            检查步骤：
                步骤1 以admin用户登录设备；
                步骤2 执行命令：show remote_replication general，查询异步远程复制配置，记录Replication Mode字段为Asynchronous的远程复制ID；
                步骤3 执行命令：show remote_replication general remote_replication_id=ID（ID步骤2中查询到的远程复制ID），并记录主LUN ID（Local LUN ID）；
                步骤4 执行命令：show lun general lun_id=ID（ID为步骤3中查询到的主LUN ID），并记录Pool ID字段值以及Capacity字段值；
                步骤5 执行命令：show storage_pool general，获取各Pool的剩余容量（Free Capacity）；
            检查标准：
              注：若存在异步远程复制配置，则远端阵列也需要检查该检查项，请添加远端阵列到工具并选择该检查项进行检查。
        1. 若不存在异步远程复制配置则通过，否则继续检查；
        2. 检查步骤4中涉及的lun和Pool满足：检查步骤5中对应Pool的剩余容量大于同一Pool下远程复制LUN的容量之和的30%，则检查通过，否者检查不通过。
    '''
    
    allCliRet = ""
    errMsg = ""
    
    try:
        #获取异步远程复制ID列表
        execRet = getAsynchIDList(cli, LANG)
        allCliRet += execRet[1]
        
        #获取异步远程复制失败（下发命令失败或发生异常）
        if not execRet[0]:
            errMsg += execRet[2]
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        if execRet[0] != True:
            errMsg += execRet[2]
            return (execRet[0], allCliRet, errMsg)
        
        #不存在异步远程复制
        if execRet[0] and not execRet[-1]:
            return (True, allCliRet, errMsg)
        
        repIDList = execRet[-1]
        
        #获取异步远程复制的主LUN ID(Local LUN ID)列表
        execRet = getLocalLunIDList(cli, LANG, repIDList)
        allCliRet += execRet[1]
        
        #获取异步远程复制的主LUN ID列表失败
        if not execRet[0] or not execRet[-1]:
            errMsg += execRet[2]
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        
        lunIDList = execRet[-1]
        lunIDList = list(set(lunIDList))
        
        #获取各Local LUN的详细信息，将容量累加到存储池LUN容量字典中对应的poolID下
        execRet = getPoolLunCapInfo(cli, LANG, lunIDList)
        allCliRet += execRet[1]
        
        #获取各Local LUN的详细信息失败
        if not execRet[0] or not execRet[-1]:
            errMsg += execRet[2]
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
                
        lunCapInfo = execRet[-1]
        LOGGER.logInfo("detail of pool LUN cap:")
        LOGGER.logInfo(lunCapInfo)
        
        #检查各存储池的剩余容量是否大于远程复制LUN总容量的30%
        execRet = checkPoolCap(cli, LANG, lunCapInfo)
        allCliRet += execRet[1]
        errMsg += execRet[2]
        
        return (execRet[0], allCliRet, errMsg)

    except (ToolException, Exception) as exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(
            LANG, "query.result.abnormal")

def getAsynchIDList(cli, lang):
    """
    @summary: 执行命令show remote_replication general，获取Replication Mode字段为Asynchronous的远程复制ID
    @return: 
        isSucc：True/False，方法是否正常结束
        cliRet：CLI回显
        errMsg：方法异常结束时的错误消息
        idList: 远程复制ID列表
    """
    id_list = []
    try:
        cmd = "show remote_replication general"
        flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
            py_java_env, cli, cmd, LOGGER)

        if flag is not True:
            err_msg = common.getMsg(lang, "can.not.get.asynch.id")
            return flag, cli_ret, err_msg, id_list

        if cliUtil.queryResultWithNoRecord(cli_ret):
            return (True, cli_ret, "", id_list)
        
        cliRetLinesList = cliUtil.getHorizontalCliRet(cli_ret)
        for line in cliRetLinesList:
            repMode = line.get("Replication Mode")
            repID = line.get("ID")
            if repMode == "Asynchronous" :
                id_list.append(repID)
                
        return True, cli_ret, "", id_list
    
    except Exception, exception:
        LOGGER.logException(exception)
        # 修改备注：捕捉到异常应该是未检查
        return (cliUtil.RESULT_NOCHECK, cli_ret,
                common.getMsg(lang, "query.result.abnormal"),
                [])


def getLocalLunIDList(cli, lang, repIDList):
    """
    @summary: 执行命令show remote_replication general remote_replication_id=ID，
                                    获取主LUN ID（Local LUN ID）
    @param repIDList: 包含需要查询的所有异步远程复制的ID的列表
    @return: 
        isSucc：True/False，方法是否正常结束
        cliRet：CLI回显
        errMsg：方法异常结束时的错误消息
        idList: 所有异步远程复制的Local LUN ID列表
    """
    
    isSucc = True
    cliRet = ""
    errMsg = ""
    idList = []
    
    try:
        cmd = "show remote_replication general remote_replication_id=%s"
        for repID in repIDList:
            execRet = cliUtil.excuteCmdInCliMode(cli, cmd % repID, True, lang)
            cliRet += execRet[1]
            
            #命令执行失败或无法查询远程复制详情
            if execRet[0] != True or cliUtil.queryResultWithNoRecord(execRet[1]):
                errMsg += common.getMsg(lang, "can.not.get.replication.detail", repID)
                return (False, cliRet, errMsg, idList)
                        
            cliRetLinesList = cliUtil.getVerticalCliRet(execRet[1])
            for line in cliRetLinesList:
                localLunID = line.get("Local LUN ID")
                if localLunID:
                    idList.append(localLunID)
        LOGGER.logInfo("local LUN ID list:")
        LOGGER.logInfo(idList)        
        return (isSucc, cliRet, errMsg, idList)
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(lang, "query.result.abnormal"), [])  # 修改备注：捕捉到异常应该是未检查
    
def getPoolLunCapInfo(cli, lang, lunIDList):
    """
    @summary: 执行命令show lun general lun_id=ID，获取Pool ID和Capacity
    @param lunIDList: 所有异步远程复制的主LUN ID列表
    @return: 
        isSucc：True/False，方法是否正常结束
        cliRet：CLI回显
        errMsg：方法异常结束时的错误消息
        lunCapInfo: 各存储池的LUN容量信息字典{poolID : totalLunCap}
    """

    cli_ret_list = []
    err_msg_total = ""
    lun_cap_info = {}

    try:
        cmd = "show lun general lun_id=%s"
        for lunID in lunIDList:
            flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
                py_java_env, cli, cmd % lunID, LOGGER)
            cli_ret_list.append(cli_ret)

            #命令执行失败或无法查询LUN详情
            if flag is not True or cliUtil.queryResultWithNoRecord(cli_ret):
                err_msg_total += common.getMsg(lang, "can.not.get.lun.detail",
                                               lunID)
                return (False, "\n".join(cli_ret_list),
                        err_msg_total, lun_cap_info)

            cli_ret_line_list = cliUtil.getVerticalCliRet(cli_ret)
            for line in cli_ret_line_list:
                pool_id = line.get("Pool ID")
                lun_cap = line.get("Capacity")
                #确认容量转换单位
                trans_flag, trans_ret = common.changUnit2GBDecimal(lun_cap)
                if trans_flag:
                    lun_cap = trans_ret
                else :
                    #容量没有单位或是其他非规范的数值
                    err_msg_total += common.getMsg(LANG,
                                                   "get.LUN.capacity.failed",
                                                   lunID)
                    return (False, "\n".join(cli_ret_list),
                            err_msg_total, lun_cap_info)

                #存储池的LUN容量累加
                total_lun_cap = lun_cap_info.get(pool_id, decimal.Decimal("0"))
                total_lun_cap += lun_cap
                lun_cap_info.update({pool_id: total_lun_cap})

        return (True, "".join(cli_ret_list), err_msg_total, lun_cap_info)

    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, "\n".join(cli_ret_list),
                common.getMsg(lang, "query.result.abnormal"),
                {})  # 修改备注：捕捉到异常应该是未检查
    
def checkPoolCap(cli, lang, lunCapInfo):
    """
    @summary: 执行命令show storage_pool general，获取各存储池的空闲容量
    @param lunCapInfo: 各存储池的LUN占用容量字典
    @return: 
        isSucc：True/False，方法是否正常结束
        cliRet：CLI回显
        errMsg：方法异常结束时的错误消息
    """
    
    passed = True
    cliRet = ""
    errMsg = ""
    poolLunInfo = {}
    pool_warning_dict = dict()
    standardNum = decimal.Decimal("0.10")
    pool_warning_num = decimal.Decimal("0.30")
    
    try:
        cmd = "show storage_pool general"
        execRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        cliRet += execRet[1]
        
        #命令执行失败或无法查询存储池详情
        if execRet[0] != True or cliUtil.queryResultWithNoRecord(cliRet):
            errMsg += common.getMsg(lang, "can.not.get.pool.detail")
            return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)
                    
        cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
        for line in cliRetLinesList:
            poolID = line.get("ID")
            poolFreeCap = line.get("Free Capacity")
            transRet = common.changUnit2GBDecimal(poolFreeCap)
            if transRet[0] :
                poolFreeCap = transRet[1]
            else:
                #容量没有单位或是其他非规范的数值
                errMsg += common.getMsg(LANG, "can.not.get.free.capacity", poolID)
                return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)
                
            poolLunCap = lunCapInfo.get(poolID, decimal.Decimal("0.0"))
            
            #LUN总容量为0，说明不存在远程复制LUN特性，跳过该存储池
            if poolLunCap == decimal.Decimal("0.0"):
                continue

            if poolFreeCap / poolLunCap <= standardNum :
                poolLunInfo.setdefault(poolID, (poolFreeCap, poolLunCap))
                continue
            if poolFreeCap / poolLunCap <= pool_warning_num :
                pool_warning_dict.setdefault(poolID, (poolFreeCap, poolLunCap))
        passed, errMsg = build_result(passed, errMsg, poolLunInfo,
                                      pool_warning_dict, lang)
            
        return (passed, cliRet, errMsg)
    except (ToolException, Exception) as exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(
            LANG, "query.result.abnormal")

def build_result(flag, err_meg, pool_not_pass_dict, pool_warning_dict, lang):
    """
    构建返回结果
    :param flag:  结果
    :param err_meg: 错误消息
    :param pool_not_pass_dict: 不通过的POOL
    :param pool_warning_dict:  建议优化的POOL
    :param lang: 中英文
    :return:
    """
    if len(pool_warning_dict) > 0:
        flag = cliUtil.RESULT_WARNING
        err_meg += build_err_meg(pool_warning_dict, lang, flag)
    if len(pool_not_pass_dict) > 0:
        flag = False
        err_meg += build_err_meg(pool_not_pass_dict, lang, flag)
    return flag, err_meg

def build_err_meg(err_dict, lang, flag):
    """
    构建错误信息
    :param err_dict: 异常pool列表
    :param lang: 中英文
    :param flag:结果
    :return:
    """
    err_meg_list = list()
    meg = "pool.lun.capacity.beyond.limit.notpass" \
        if flag is False else "pool.lun.capacity.beyond.limit"
    for pool_id in err_dict:
        cap_tuple = err_dict.get(pool_id)
        pool_free_cap = cap_tuple[0]
        pool_lun_cap = cap_tuple[1]

        err_meg_list.append(common.getMsg(lang, meg,
                                          (pool_id, pool_free_cap,
                                           pool_lun_cap)))
    return "".join(err_meg_list)
    