# -*- coding: UTF-8 -*-
from __future__ import division
import decimal
import cliUtil
import common
from common import AsynProgress
from cbb.frame.cli.cli_with_cache import execute_cmd_in_cli_mode_with_cache
from com.huawei.ism.tool.obase.exception import ToolException

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env

def execute(cli):
    '''
            快照容量检查 ； 
            检查步骤：
                步骤1 以admin用户登录设备
                步骤2 执行如下命令：show snapshot general，记录全部Source LUN ID字段值。
                步骤3 执行如下命令：show lun general lun_id=ID（ID为步骤2中查询到的Source LUN ID），记录lun的容量（Capacity）及所属的Pool（Pool ID）。
                步骤4 执行如下命令：show storage_pool general，获取所有Pool的ID及剩余容量（Free Capacity）。
            检查标准：
    1. 检查步骤3中涉及的lun和Pool满足：同一Pool下涉及快照源lun的容量总和的21%应小于检查步骤4中对应Pool的剩余容量，则检查通过，否则检查不通过。
    '''
    allCliRet = ""
    errMsg = ""
    
    #获取Source LUN ID列表
    myPthread = AsynProgress(PY_JAVA_ENV, LOGGER)
    myPthread.start_thread()
    try:
        execRet = getSourceLunIDList(cli, LANG)
        allCliRet += execRet[1]
        #获取Source LUN ID失败
        if execRet[0] !=True:
            errMsg += execRet[2]
            common.refreshProcess(PY_JAVA_ENV, 100, LOGGER)
            return (execRet[0], allCliRet, errMsg)
        
        #命令成功执行，但无快照记录
        if not execRet[-1] :
            common.refreshProcess(PY_JAVA_ENV, 100, LOGGER)
            return (True, allCliRet, errMsg)
        
        lunIDList = execRet[-1]
        
        #获取各Source LUN的详细信息，将容量累加到存储池LUN容量字典中对应的poolID下
        execRet = getPoolLunCapInfo(cli, LANG, lunIDList)
        allCliRet += execRet[1]
        
        #获取各Source LUN的详细信息失败
        if not execRet[0] or not execRet[-1]:
            errMsg += execRet[2]
            common.refreshProcess(PY_JAVA_ENV, 100, LOGGER)
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
                
        lunCapInfo = execRet[-1]
        
        #检查各存储池快照源LUN总容量的21%是否小于存储池的剩余容量
        execRet = checkPoolCap(cli, LANG, lunCapInfo)
        allCliRet += execRet[1]
        errMsg += execRet[2]
        
        return (execRet[0], allCliRet, errMsg)
    
    except Exception, exception:
        LOGGER.logException(exception)
        common.refreshProcess(PY_JAVA_ENV, 100, LOGGER)
        return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(LANG, "query.result.abnormal"))
    
    finally:
        myPthread.setStopFlag(True)
    
def getSourceLunIDList(cli, lang):
    """
    @summary: 执行命令show snapshot general，获取全部Source LUN ID字段值
    @return: 
        isSucc：True/False，方法是否正常结束
        cliRet：CLI回显
        errMsg：方法异常结束时的错误消息
        idList: 所有异步远程复制的Source LUN ID列表
    """
    
    cli_ret = ""
    err_msg = ""
    id_list = []
    
    try:
        cmd = "show snapshot general"
        flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
            py_java_env, cli, cmd, LOGGER)

        #命令执行失败或无法查询快照信息
        if flag is False:
            err_msg = common.getMsg(lang, "can.not.get.snapshot.info")
            return cliUtil.RESULT_NOCHECK, cli_ret, err_msg, id_list

        if flag is not True:
            return flag, cli_ret, err_msg, id_list
        
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return True, cli_ret, err_msg, id_list
                    
        cli_ret_list = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        for line in cli_ret_list:
            local_lun_id = line.get("Source LUN ID")
            if local_lun_id:
                id_list.append(local_lun_id)
        
        #兼容多个快照的源LUN是同一个的情况
        id_list = list(set(id_list))
                   
        return True, cli_ret, err_msg, id_list
    
    except Exception, exception:
        LOGGER.logException(exception)
        return (cliUtil.RESULT_NOCHECK, cli_ret,
                common.getMsg(lang, "query.result.abnormal"), id_list)
    
def getPoolLunCapInfo(cli, lang, lunIDList):
    """
    @summary: 执行命令show lun general lun_id=ID，获取Pool ID和Capacity
    @param lunIDList: Source LUN ID列表
    @return: 
        isSucc：True/False，方法是否正常结束
        cliRet：CLI回显
        errMsg：方法异常结束时的错误消息
        lunCapInfo: 各存储池的LUN容量信息字典{poolID : totalLunCap}
    """
    is_succ = True
    cli_ret_list = []
    err_msg_total = ''
    lun_cap_info = {}
    err_luns = []
    err_caps = []
    
    try:
        cmd = "show lun general lun_id=%s"
        for lun_id in lunIDList:
            cmd_str = cmd % lun_id
            flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
                py_java_env, cli, cmd_str, LOGGER)
            cli_ret_list.append(cli_ret)
            #命令执行失败或无法查询LUN详情
            if flag is not True or cliUtil.queryResultWithNoRecord(cli_ret):
                err_luns.append(lun_id)
                continue
                        
            cli_ret_lines_list = cliUtil.getVerticalCliRet(cli_ret)
            for line in cli_ret_lines_list:
                pool_id = line.get("Pool ID")
                lun_cap = line.get("Capacity")
                
                transRet = common.changUnit2GBDecimal(lun_cap)
                if transRet[0] :
                    lun_cap = transRet[1]
                else :
                    #容量没有单位或是其他非规范的数值
                    err_caps.append(lun_id)
                    break
                
                #存储池的LUN容量累加
                total_lun_cap = lun_cap_info.get(pool_id, decimal.Decimal("0"))
                total_lun_cap += lun_cap
                lun_cap_info.update({pool_id: total_lun_cap})
        
        #对所有获取信息失败和获取容量失败的LUN做错误消息处理        
        if err_luns:
            is_succ = False
            err_msg_total += common.getMsg(lang, "can.not.get.lun.detail",
                                           ", ".join(err_luns))
            
        if err_caps:
            is_succ = False
            err_msg_total += common.getMsg(LANG, "get.LUN.capacity.failed",
                                           ", ".join(err_caps))
        
        return is_succ, "\n".join(cli_ret_list), err_msg_total, lun_cap_info
    
    except Exception, exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, "\n".join(cli_ret_list), common.getMsg(
            lang, "query.result.abnormal"), {}
    
def checkPoolCap(cli, lang, lunCapInfo):
    """
    @summary: 执行命令show storage_pool general，获取各存储池的空闲容量
    @param lunCapInfo: 各存储池的LUN占用容量字典
    @return: 
        isSucc：True/False，方法是否正常结束
        cliRet：CLI回显
        errMsg：方法异常结束时的错误消息
    """
    passed = True
    cliRet = ""
    errMsg = ""
    poolLunInfo = {}
    pool_warning_dict = dict()
    standardNum = decimal.Decimal("0.10")
    pool_warning_num = decimal.Decimal("0.30")
    
    try:
        cmd = "show storage_pool general"
        execRet = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        cliRet += execRet[1]
        #命令执行失败或无法查询存储池详情
        if execRet[0] != True or cliUtil.queryResultWithNoRecord(cliRet):
            errMsg += common.getMsg(lang, "can.not.get.pool.detail")
            return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)
                    
        cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
        for line in cliRetLinesList:
            poolID = line.get("ID")
            
            if poolID not in lunCapInfo:
                continue
            
            poolFreeCap = line.get("Free Capacity")
            transRet = common.changUnit2GBDecimal(poolFreeCap)
            if transRet[0] :
                poolFreeCap = transRet[1]
            else:
                #容量没有单位或是其他非规范的数值
                errMsg += common.getMsg(LANG, "can.not.get.free.capacity", poolID)
                return (cliUtil.RESULT_NOCHECK, cliRet, errMsg)
                
            poolLunCap = lunCapInfo.get(poolID, decimal.Decimal("0.0"))

            if poolLunCap * standardNum >= poolFreeCap :
                poolLunInfo.setdefault(poolID, (poolFreeCap, poolLunCap))
                continue
            if poolLunCap * pool_warning_num >= poolFreeCap :
                pool_warning_dict.setdefault(poolID, (poolFreeCap, poolLunCap))
        passed, errMsg = build_result(passed, errMsg, poolLunInfo,
                                      pool_warning_dict, lang)
        return (passed, cliRet, errMsg)
    except (ToolException, Exception) as exception:
        LOGGER.logException(exception)
        return cliUtil.RESULT_NOCHECK, cliRet, common.getMsg(
            LANG, "query.result.abnormal")

def build_result(flag, err_meg, pool_not_pass_dict, pool_warning_dict, lang):
    """
    构建返回结果
    :param flag:  结果
    :param err_meg: 错误消息
    :param pool_not_pass_dict: 不通过的POOL
    :param pool_warning_dict:  建议优化的POOL
    :param lang: 中英文
    :return:
    """
    if len(pool_warning_dict) > 0:
        flag = cliUtil.RESULT_WARNING
        err_meg += build_err_meg(pool_warning_dict, lang, flag)
    if len(pool_not_pass_dict) > 0:
        flag = False
        err_meg += build_err_meg(pool_not_pass_dict, lang, flag)
    return flag, err_meg

def build_err_meg(err_dict, lang, flag):
    """
    构建错误信息
    :param err_dict: 异常pool列表
    :param lang: 中英文
    :param flag:结果
    :return:
    """
    err_meg_list = list()
    meg = "pool.lun.capacity.beyond.limit.notpass" if \
        flag is False else "pool.lun.capacity.beyond.limit.snap"
    for pool_id in err_dict:
        cap_tuple = err_dict.get(pool_id)
        pool_free_cap = cap_tuple[0]
        pool_lun_cap = cap_tuple[1]
        err_meg_list.append(common.getMsg(lang, meg,
                                          (pool_id, pool_free_cap,
                                           pool_lun_cap)))
    return "".join(err_meg_list)