# -*- coding: UTF-8 -*-

import traceback

from cbb.frame.checkitem.context_adapter import InspectContext
from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.adapter import replace_adapter

import common_utils
import common
from risk_version_config import DISK_DOMAIN_TOTAL_DISK_NUM_VERSION

PY_JAVA_ENV = py_java_env
JAVA_LOGGER = PY_LOGGER
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(JAVA_LOGGER, __file__)

MAX_DOMAIN_DISK_NUMBER = 256


def execute(cli):
    """
    硬盘域的硬盘总数检查
    主要检查硬盘域中成员盘数量是否大于256。
    :param cli:
    :return:
    """
    context = InspectContext(PY_JAVA_ENV, cli)
    inspect_context = context.get_context()
    inspect_context["logger"] = JAVA_LOGGER
    check_obj = CheckItem(inspect_context, cli)
    status, err_msg = check_obj.check()
    all_ret = check_obj.get_origin_info()
    LOGGER.logInfo("status={},err_msg={}".format(status, err_msg))
    return context.get_result(status, err_msg, all_ret)


class CheckItem(BaseCheckItem):
    def __init__(self, context, cli):
        super(CheckItem, self).__init__(context)
        self.dev_version = ""
        self.patch_version = ""
        self.cli = cli
        self.disk_domain_id_list = list()

    def execute(self):
        try:
            self.init_version_and_patch()
            if not self.is_risk_version():
                return CheckStatus.PASS, ""

            self.query_all_domain()
            if not self.disk_domain_id_list:
                return CheckStatus.PASS, ""

            if not self.is_domain_disk_number_risk():
                return CheckStatus.PASS, ""

            return CheckStatus.NOTPASS, self.deal_error_msg()
        except Exception:
            LOGGER.logError(traceback.format_exc())
            return CheckStatus.NOCHECK, common.getMsg(
                LANG, "query.result.abnormal")

    def init_version_and_patch(self):
        """
        初始化版本、补丁信息
        :return:
        """
        (
            flag,
            self.dev_version,
            self.patch_version,
            ret,
        ) = replace_adapter.get_system_version_with_ret(self.context)
        self.patch_version = "--" if not self.patch_version \
            else self.patch_version
        self.add_origin_info(ret)

    def query_all_domain(self):
        """
        查询全部存储池
        :return:
        """
        pool_info_list = self.dsl(
            "exec_cli 'show storage_pool general |filterColumn include "
            "columnList=ID,Disk\sDomain\sID' | horizontal_parser",
        )

        self.disk_domain_id_list = set([
            pool.get("Disk Domain ID")
            for pool in pool_info_list
        ])

    def is_domain_disk_number_risk(self):
        """
        检查硬盘域硬盘数量是否超过规格
        若步骤4中存在一个硬盘域中成员盘数量大于256，则检查结果为不通过
        :return: False: 无风险; True: 有风险
        """
        for domain_id in self.disk_domain_id_list:
            domain_info = self.dsl(
                "exec_cli 'show disk_domain general disk_domain_id={}'"
                " | vertical_parser".format(domain_id),
            )
            disk_num_str = domain_info.get("Disk Number", '0')
            if not disk_num_str.isdigit():
                continue
            if int(disk_num_str) > MAX_DOMAIN_DISK_NUMBER:
                return True
        return False

    def deal_error_msg(self):
        """
        根据版本处理修复建议
        :return:
        """
        target_patch = DISK_DOMAIN_TOTAL_DISK_NUM_VERSION.get(
            self.dev_version)
        if not target_patch:
            return common_utils.get_err_msg(
                LANG,
                "contact.technical.support.engineers"
            )
        return common_utils.get_err_msg(
            LANG,
            "software.io.delay.risk.patch.not.pass",
            (self.dev_version, self.patch_version, target_patch),
        )

    def is_risk_version(self):
        """
        检查是否风险版本。
        :return: True 风险版本, False 非风险。
        """
        if self.dev_version not in DISK_DOMAIN_TOTAL_DISK_NUM_VERSION:
            return False

        target_patch = DISK_DOMAIN_TOTAL_DISK_NUM_VERSION.get(self.dev_version)
        if not target_patch:
            return True

        if common_utils.get_patch_value(self.patch_version) >= \
                common_utils.get_patch_value(target_patch):
            return False
        return True
