# -*- coding: UTF-8 -*-
import re
import traceback

from com.huawei.ism.tool.obase.exception import ToolException

from cbb.frame.base import product
from cbb.frame.util.common import fakeProgress
from cbb.frame.context import sqlite_context

import cliUtil
import common
import common_utils
import common_cache
from common import UnCheckException

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)

KUNPENG_PROMPT = "diskId:{}"
NORMAL_PROMPT = "ID:{}"


@fakeProgress(PY_JAVA_ENV, totalSeconds=120, logger=LOGGER, interval=1)
def execute(cli):
    """
    硬盘recovery状态检查
    software_disk_recovery_state_check
    """
    all_cli_ret = []
    conn_cli = None
    try:
        risk_flag, p_version = is_risk_version(cli, all_cli_ret)
        LOGGER.logInfo("risk is{},version is:{}".format(risk_flag, p_version))
        if not risk_flag:
            return True, all_cli_ret[0], ""

        # 如果是18000执行debug命令获取阵列链接
        flag, conn_cli, err_msg = common.createDeviceCliContFor18000(
            cli, PY_JAVA_ENV, LOGGER, LANG
        )
        if flag is not True:
            return cliUtil.RESULT_NOCHECK, "\n".join(all_cli_ret), err_msg
        # 每个引擎只检查一控
        disk_state_check = DiskRecoverStateCheck(conn_cli, p_version)
        check_disk_in_ctrl(
            conn_cli, PY_JAVA_ENV, LOGGER,
            LANG, "", all_cli_ret, disk_state_check,
        )
        all_ret = "\n".join(all_cli_ret)
        # 存在风险硬盘
        if disk_state_check.risk_disk_id_list:
            msg = get_err_prompt_info(
                get_disk_location_by_disk_id(disk_state_check.risk_disk_id_list, cli),
                p_version
            )
            return False, all_ret, msg
        return True, all_ret, ""
    except UnCheckException as unCheckException:
        return (
            cliUtil.RESULT_NOCHECK, "\n".join(all_cli_ret),
            unCheckException.errorMsg,
        )
    except (ToolException, Exception):
        LOGGER.logInfo(str(traceback.format_exc()))
        return (
            cliUtil.RESULT_NOCHECK, "\n".join(all_cli_ret),
            common.getMsg(LANG, "query.result.abnormal"),
        )
    finally:
        if conn_cli is not cli:
            common.closeDeviceCliContFor18000(conn_cli, PY_JAVA_ENV, LOGGER, LANG)
        # 退出到cli模式
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode from some model ret is {}".format(ret))

        # 退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)


def get_disk_location_by_disk_id(risk_disk_id_list, cli):
    """
    将风险硬盘ID 转换为 location
    :param risk_disk_id_list:
    :return:
    """
    dev_info = PY_JAVA_ENV.get("devInfo")
    object_for_py = PY_JAVA_ENV.get("objectForPy")
    sn = common.get_sn_from_env(PY_JAVA_ENV)
    p_version = str(dev_info.getProductVersion())
    p_patch = str(dev_info.getHotPatchVersion())
    is_risk, need_patch = common_utils.is_risk_version_fc_cause_risk(
        cli, p_version, p_patch, LOGGER
    )
    if is_risk:
        return risk_disk_id_list

    sqlite_conn = sqlite_context.get_sqlite_conn_from_context(
        object_for_py, sn, LOGGER
    )
    flag = common_cache.ana_running_data(
        cli, sn, p_version, LANG, sqlite_conn, LOGGER, object_for_py
    )
    if not flag:
        return risk_disk_id_list

    tmp_dict = object_for_py.get(common_cache.DISK_INFO_FROM_CONFIG.format(sn))
    if not tmp_dict:
        return risk_disk_id_list
    disk_info_list = tmp_dict.get("disk_info", [])
    risk_disk_location = []
    for disk_info_dict in disk_info_list:
        if disk_info_dict.get("Disk ID") in risk_disk_id_list:
            risk_disk_location.append(
                disk_info_dict.get("Disk Frame and Slot ID")
            )
    if not risk_disk_location:
        return risk_disk_id_list
    return risk_disk_location


def get_err_prompt_info(risk_disk_id_list, p_version):
    """
    获取错误提示
    :param risk_disk_id_list: 风险硬盘ID
    :param p_version: 版本型号
    :return: 提示信息
    """
    risk_disk_str = ",".join(list(set(risk_disk_id_list)))
    prompt_msg = (
        NORMAL_PROMPT.format(risk_disk_str)
        if product.isKunpeng(p_version)
        else KUNPENG_PROMPT.format(risk_disk_str)
    )
    err_msg = common_utils.get_err_msg(
        LANG, "disk.recover.status.check.no.pass", prompt_msg
    )
    return err_msg


def is_risk_version(cli, ret_list):
    """
    检查是否风险版本
    :param cli: ssh连接
    :param ret_list: 回文列表
    :return: True 是风险版本，False 非风险版本
    """
    (
        flag,
        p_version,
        p_patch,
        ret,
        err_msg,
    ) = common_cache.get_version_and_patch_cache(PY_JAVA_ENV, cli, LOGGER)
    ret_list.append(ret)
    if flag is not True:
        raise UnCheckException(err_msg, ret)
    LOGGER.logInfo("version is {}, patch is {}".format(p_version, p_patch))
    # 若步骤2中系统软件版本低于V300R006C50SPC100或高于V300R006C61SPC300，
    # 或者低于V500R007C30SPC100或高于V500R007C60SPC300，检查结果为通过，否则继续检查
    if p_version.startswith("V300R006") and (
        p_version > "V300R006C61SPC300" or p_version < "V300R006C50SPC100"
    ):
        return False, p_version
    if p_version.startswith("V500R007") and (
        p_version > "V500R007C60SPC300" or p_version < "V500R007C30SPC100"
    ):
        return False, p_version

    return True, p_version


class DiskRecoverStateCheck:
    def __init__(self, cli, product_version):
        self.cli = cli
        # 返回框架的参数
        self.err_msg = ""
        self.product_version = product_version
        self.risk_disk_id_list = []

    def get_recovery_disks(self, all_cli_ret):
        """
        获取等待状态的盘
        :param all_cli_ret: 回文列表
        :return: 列表
        """
        recovery_disk_list = []
        cmd = "ckgiof ehstate"
        flag, ret, err_msg = cliUtil.excuteCmdInDebugModel(self.cli, cmd, LANG)
        all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(err_msg, ret)
        query_key_list = ["diskId", "isRecovery"]
        res_list = common_utils.ana_form_by_space(ret, query_key_list, LOGGER)
        if not res_list:
            return recovery_disk_list

        for res_info in res_list:
            if res_info.get("isRecovery", "").lower() == "yes":
                recovery_disk_list.append(res_info.get("diskId"))

        return recovery_disk_list

    def get_task_disks(self, all_cli_ret):
        """
        获取任务中的硬盘
        :param all_cli_ret: 回文列表
        :return: 列表
        """
        cmd = "pmgr showalltask"
        flag, ret, err_msg = cliUtil.excuteCmdInDebugModel(self.cli, cmd, LANG)
        all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(err_msg, ret)

        regx = re.compile(r"Disk Id\s*:\s*(\d*)")
        return regx.findall(ret)

    def get_disk_sds_info(self, all_cli_ret):
        """
        获取硬盘的LDID和ID信息
        :param all_cli_ret: 回文列表
        :return: ID对应的LDID信息字典
        """
        id_ldid_dict = {}
        cmd = "ld show sds"
        flag, ret, err_msg = cliUtil.excuteCmdInDebugModel(self.cli, cmd, LANG)
        all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(err_msg, ret)
        query_key_list = ["LDID", "ID"]
        res_dict_list = common_utils.ana_form_by_space(
            ret, query_key_list, LOGGER
        )
        for res_info in res_dict_list:
            disk_id = res_info.get("ID")
            ldid = res_info.get("LDID")
            id_ldid_dict[ldid] = disk_id

        return id_ldid_dict


@common.checkAllEngineInClusterWarp
def check_disk_in_ctrl(
    cli, env, logger, lang, heart_beat_cli_ret, all_cli_ret, cls_obj
):
    """
    每个引擎只检查任意一个控制器
    :param cli: ssh连接
    :param env: 上下文
    :param logger: 日志
    :param lang: 语言
    :param heart_beat_cli_ret: 心跳回文
    :param all_cli_ret: 回文列表
    :param cls_obj: 检查类对象
    :return: 不再继续检查当前引擎其他控制器
    """
    all_cli_ret.append(heart_beat_cli_ret)
    recovery_disk_list = cls_obj.get_recovery_disks(all_cli_ret)
    LOGGER.logInfo("recovery_disk_list:{}".format(recovery_disk_list))
    if not recovery_disk_list:
        return common.CHECK_FLAG_ONLY_ONE_CTRL

    task_disk_ids = cls_obj.get_task_disks(all_cli_ret)
    LOGGER.logInfo("task_disk_ids:{}".format(task_disk_ids))
    # 兼容py3加一个list
    risk_disk_ids = list(
        filter(
            lambda disk_id: disk_id not in task_disk_ids, recovery_disk_list
        )
    )
    LOGGER.logInfo("risk_disk_ids:{}".format(risk_disk_ids))
    if not product.isKunpeng(cls_obj.product_version):
        id_ldid_dict = cls_obj.get_disk_sds_info(all_cli_ret)
        risk_disk_ids = [
            id_ldid_dict.get(disk_id, disk_id) for disk_id in risk_disk_ids
        ]
    cls_obj.risk_disk_id_list.extend(risk_disk_ids)
    return common.CHECK_FLAG_ONLY_ONE_CTRL
