# -*- coding: UTF-8 -*-
import traceback

from com.huawei.ism.tool.obase.exception import ToolException

import cliUtil
import common
import common_utils
from common import UnCheckException
from common import AsynProgress

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """
    硬盘wait access状态检查
    """
    all_cli_ret = []
    risk_disk_list = []
    conn_cli = None
    my_pthread = AsynProgress(PY_JAVA_ENV, LOGGER, sleepTime=0.3)
    my_pthread.start_thread()
    try:
        # 进入developer模式
        flag, ret, err_msg = cliUtil.enterDeveloperMode(cli, LANG)
        all_cli_ret.append(ret)
        if flag is not True:
            return False, "\n".join(all_cli_ret), err_msg

        # 如果是18000执行debug命令获取阵列链接
        flag, conn_cli, err_msg = common.createDeviceCliContFor18000(
            cli, PY_JAVA_ENV, LOGGER, LANG
        )
        if flag is not True:
            return cliUtil.RESULT_NOCHECK, "\n".join(all_cli_ret), err_msg

        heart_beat_cli_ret = ''
        check_wait_access_status(
            conn_cli,
            PY_JAVA_ENV,
            LOGGER,
            LANG,
            heart_beat_cli_ret,
            all_cli_ret,
            risk_disk_list,
        )
        LOGGER.logInfo("risk_disk_list is:{}".format(risk_disk_list))
        if risk_disk_list:
            err_msg = common_utils.get_err_msg(
                LANG,
                "software.disk.waitaccess.not.pass",
                ",".join(list(set(risk_disk_list)))
            )
            return False, "\n".join(all_cli_ret), err_msg

        return True, "\n".join(all_cli_ret), ""
    except UnCheckException as unCheckException:
        return (
            cliUtil.RESULT_NOCHECK,
            "\n".join(all_cli_ret),
            unCheckException.errorMsg,
        )
    except (ToolException, Exception):
        LOGGER.logInfo(str(traceback.format_exc()))
        return (
            cliUtil.RESULT_NOCHECK,
            "\n".join(all_cli_ret),
            common.getMsg(LANG, "query.result.abnormal"),
        )

    finally:
        my_pthread.setStopFlag(True)
        if conn_cli is not cli:
            common.closeDeviceCliContFor18000(
                conn_cli, PY_JAVA_ENV, LOGGER, LANG
            )
        # 退出到cli模式
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode from some model ret is %s" % str(ret))

        # 退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)


@common.checkAllEngineInClusterWarp
def check_wait_access_status(
    cli,
    env,
    logger,
    lang,
    heart_beat_cli_ret,
    all_cli_ret_list,
    risk_disk_list,
):
    """
    步骤4的diskId对应步骤5的LDID，步骤6的ID对应步骤5的SDID
    :param cli: ssh连接
    :param env: 上下文
    :param logger: 日志
    :param lang: 语言
    :param heart_beat_cli_ret: 心跳回文
    :param all_cli_ret_list: 回文列表
    :return: 是否继续
    """
    LOGGER.logInfo("heart beat ret:{}".format(heart_beat_cli_ret))
    all_cli_ret_list.append(heart_beat_cli_ret)
    # 步骤4 获取等待状态的盘
    wait_access_disk_list = get_wait_access_disk(cli, all_cli_ret_list)
    LOGGER.logInfo("wait access disk are:{}".format(wait_access_disk_list))
    if not wait_access_disk_list:
        return common.CHECK_FLAG_CONTINUE
    # 步骤5 获取硬盘LDID对应的SDID
    sdid_ldid_dict = get_disk_sds_info(cli, all_cli_ret_list)
    LOGGER.logInfo("disk sdid and ldid mapping:{}".format(sdid_ldid_dict))
    # 步骤6 查询存在盘State为RUNNING的ID
    running_sdde_id_list = get_running_sdde_info(cli, all_cli_ret_list)
    LOGGER.logInfo("running_sdde_id_list:{}".format(running_sdde_id_list))
    if not running_sdde_id_list:
        return common.CHECK_FLAG_CONTINUE

    tmp_risk_disk_list = get_risk_disk_id(
        running_sdde_id_list, sdid_ldid_dict, wait_access_disk_list
    )
    risk_disk_list.extend(tmp_risk_disk_list)
    LOGGER.logInfo("tmp_risk_disk_list is:{}".format(tmp_risk_disk_list))
    return common.CHECK_FLAG_CONTINUE


def get_risk_disk_id(
    running_sdde_id_list, sdid_ldid_dict, wait_access_disk_list
):
    """
    获取风险盘id
    :param running_sdde_id_list: 运行的盘
    :param sdid_ldid_dict: 关系对应表sdid 和ldid
    :param wait_access_disk_list: 处于等待状态的盘
    :return: 风险盘
    """
    risk_disk_id_list = []
    for running_sdde_id in running_sdde_id_list:
        ldid = sdid_ldid_dict.get(running_sdde_id)
        if ldid and ldid in wait_access_disk_list and ldid.isdigit():
            risk_disk_id_list.append(int(running_sdde_id))
    risk_disk_id_list.sort()
    return [str(disk_id) for disk_id in risk_disk_id_list]


def get_wait_access_disk(cli, ret_list):
    """
    获取等待状态的盘
    :param cli: ssh连接
    :param ret_list: 回文列表
    :return: 列表
    """
    wait_access_disk_list = []
    cmd = "ckgiof ehstate"
    flag, ret, err_msg = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
    ret_list.append(ret)
    if flag is not True:
        raise UnCheckException(err_msg, ret)
    query_key_list = ["diskId", "isWaitAccess"]
    res_list = common_utils.ana_form_by_space(ret, query_key_list, LOGGER)
    if not res_list:
        return wait_access_disk_list

    for res_info in res_list:
        if res_info.get("isWaitAccess") == "YES":
            wait_access_disk_list.append(res_info.get("diskId"))

    return wait_access_disk_list


def get_running_sdde_info(cli, ret_list):
    """
    获取硬盘的LDID和SDID信息
    :param cli: ssh连接
    :param ret_list: 回文列表
    :return: 列表
    """
    sdde_id_list = []
    cmd = "sddebug show device"
    flag, ret, err_msg = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
    ret_list.append(ret)
    if flag is not True:
        raise UnCheckException(err_msg, ret)
    query_key_list = ["ID", "State"]
    res_list = common_utils.ana_form_by_space(ret, query_key_list, LOGGER)
    if not res_list:
        return sdde_id_list

    for res_info in res_list:
        if res_info.get("State") == "RUNNING":
            sdde_id_list.append(res_info.get("ID"))

    return sdde_id_list


def get_disk_sds_info(cli, ret_list):
    """
    获取硬盘的LDID和SDID信息
    :param cli: ssh连接
    :param ret_list: 回文列表
    :return: SDID对应的LDID信息字典
    """
    sdid_ldid_dict = {}
    cmd = "ld show sds"
    flag, ret, err_msg = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
    ret_list.append(ret)
    if flag is not True:
        raise UnCheckException(err_msg, ret)
    query_key_list = ["LDID", "SDID"]
    res_dict_list = common_utils.ana_form_by_space(ret, query_key_list, LOGGER)
    for res_info in res_dict_list:
        sdid = res_info.get("SDID")
        ldid = res_info.get("LDID")
        sdid_ldid_dict[sdid] = ldid

    return sdid_ldid_dict
