# -*- coding: UTF-8 -*-
import re

from cbb.frame.base import product

import common
import cliUtil
from risk_version_config import RISK_SHARE_NFS_VERSION_AND_PATCH
from common import UnCheckException
from common_utils import (
    check_in_vstore_mode_wrap,
    get_err_msg,
)

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)

# 进度总剩余时间
LIMIT_TIME = 120
# 进度刷新间隔
INTERVAL = 2

# 有风险的共享缓存数量
RISK_SHARE_NFS_COUNT = 1500


def execute(cli):
    """
    NFS共享检查
    :param cli:
    :return:
    """
    nfs_check = FsShareCheck(cli)
    flag, no_check_msg = nfs_check.check_nfs_count()
    all_ret = "\n".join(nfs_check.all_cli_ret_list)

    if flag is not True:
        return cliUtil.RESULT_NOCHECK, all_ret, no_check_msg

    if nfs_check.err_msg:
        return False, all_ret, nfs_check.err_msg

    return True, all_ret, ""


class FsShareCheck:
    def __init__(self, cli):
        self.cli = cli
        # 返回框架的参数
        self.all_cli_ret_list = []
        self.err_msg = ""

        # 默认租户下NFS共享数量
        self.share_nfs_count_default = 0

        # 所有租户的NFS共享数量
        self.all_vstore_nfs_count = 0

        # 所有vstore id
        self.vstore_ids = []

        # 所有控制器
        self.controller_list = []

    def check_nfs_count(self):
        """
        检查主流程入口
        :return:
        """
        try:
            # 进度条刷新
            common.threadUpProcess(PY_JAVA_ENV, LIMIT_TIME, INTERVAL, LOGGER)
            # 进度开始
            common.inProcess(PY_JAVA_ENV)
            if not self.is_risk_version():
                return True, ""

            # 查看默认租户下NFS共享数量
            self.share_nfs_count_default = self.get_share_nfs_count()
            # 获取所有租户ID
            self.get_all_vstore()
            # 获取全部租户视图下NFS共享数量
            self.get_all_vstore_nfs_count()

            self.all_vstore_nfs_count += self.share_nfs_count_default

            if self.all_vstore_nfs_count == 0:
                return True, ""

            # 进入developer
            flag, cli_ret, msg = cliUtil.enterDeveloperMode(self.cli, LANG)
            self.all_cli_ret_list.append(cli_ret)
            if flag is not True:
                return flag, msg
            (
                flag,
                controller_list,
                err_msg,
                cli_ret,
            ) = cliUtil.getControllerIdListWithRet(self.cli, LANG)
            self.all_cli_ret_list.append(cli_ret)
            self.controller_list = controller_list
            if flag is not True:
                return flag, msg
            # 检查每控的链接数量
            self.check_nfs_connection_count()

            return True, ""
        except UnCheckException as un:
            LOGGER.logException(un)
            return cliUtil.RESULT_NOCHECK, un.errorMsg
        except Exception as e:
            LOGGER.logException(e)
            err_key = "query.result.abnormal"
            return cliUtil.RESULT_NOCHECK, common.getMsg(LANG, err_key)
        finally:
            common.finishProcess(PY_JAVA_ENV)
            LOGGER.logInfo("finish process!")

    def is_risk_version(self):
        """
        检查是否风险版本
        请安装对应版本热补丁，对应补丁关系如下：
        (1) V500R007C60SPC300：V500R007C60SPH301或之后版本补丁；
        (2) V500R007C60SPC300 Kunpeng：V500R007C60SPH302或之后版本补丁。
        :return:
        """
        (
            flag,
            soft_version,
            patch_version,
            cli_ret,
            err_msg,
        ) = common.get_soft_and_patch_version(self.cli, LANG, LOGGER)
        self.all_cli_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg, "")

        dev_info = py_java_env.get("devInfo")
        p_version = str(dev_info.getProductVersion())
        if (
            p_version in RISK_SHARE_NFS_VERSION_AND_PATCH
            and patch_version
            < RISK_SHARE_NFS_VERSION_AND_PATCH.get(p_version, "")
        ):
            return True

        return False

    def get_all_vstore_nfs_count(self):
        """
        获取全部租户视图下NFS共享数量
        :return:
        """
        # 遍历租户检查
        for vstore_id in self.vstore_ids:
            param_dict = {}
            param_dict["cli"] = self.cli
            param_dict["vstore_id"] = vstore_id
            param_dict["logger"] = LOGGER
            param_dict["lang"] = LANG
            param_dict["cli_ret_all"] = self.all_cli_ret_list
            # 在当前租户下检查控制器IP
            self.get_vstore_nfs_count(param_dict)

    def get_share_nfs_count(self):
        """
        查看默认租户下NFS共享数量
        :return: NFS共享数量
        """
        cmd = "show share nfs_count"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, LANG
        )
        self.all_cli_ret_list.append(cli_ret)
        # 某些型号可能无相关命令，如果命令不存在，则不再继续执行。
        if not cliUtil.hasCliExecPrivilege(cli_ret):
            return 0
        if flag is not True:
            raise UnCheckException(err_msg, "")

        regx = re.compile(r"Number\s*:\s*(\d*)")
        res = regx.findall(cli_ret)
        return int(res[0]) if res else 0

    def get_all_vstore(self):
        cmd = "show vstore |filterColumn include columnList=ID"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, LANG
        )
        self.all_cli_ret_list.append(cli_ret)
        # 某些型号可能无相关命令，如果命令不存在，则不再继续执行。
        if not cliUtil.hasCliExecPrivilege(cli_ret):
            return
        if flag is not True:
            raise UnCheckException(err_msg, "")
        ret_list = cliUtil.getHorizontalCliRet(cli_ret)
        self.vstore_ids = [ret.get("ID") for ret in ret_list]

    @check_in_vstore_mode_wrap
    def get_vstore_nfs_count(self, param_dict, *args, **kargs):
        """
        步骤6 执行命令：show share nfs_count，检查租户视图下NFS共享数量；
        :param param_dict: 参数字典
        :param args: 可变参数
        :param kargs:
        :return:
        """
        self.all_vstore_nfs_count += self.get_share_nfs_count()

    def check_nfs_connection_count(self):
        """
        检查每控的链接数量
        :return:
        """
        cmd_str = "show nfs server_info controller={}"
        for ctrl_id in self.controller_list:
            cmd = cmd_str.format(ctrl_id)
            flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
                self.cli, cmd, True, LANG
            )
            self.all_cli_ret_list.append(cli_ret)
            # 某些型号可能无相关命令，如果命令不存在，则不再继续执行。
            if not cliUtil.hasCliExecPrivilege(cli_ret):
                return

            if flag is not True:
                raise UnCheckException(err_msg, "")

            ret_list = cliUtil.getVerticalCliRet(cli_ret)
            current_ctrl_count = 0
            for ret in ret_list:
                nfs3 = int(ret.get("NFS3 Connection Count", 0))
                nfs4 = int(ret.get("NFS4 Connection Count", 0))
                nfs4_1 = int(ret.get("NFS4.1 Connection Count", 0))
                current_ctrl_count = sum([nfs3, nfs4, nfs4_1])
            LOGGER.logInfo(
                "ctrl {} nfs count is {}".format(ctrl_id, current_ctrl_count)
            )
            if (
                current_ctrl_count * self.all_vstore_nfs_count
                > RISK_SHARE_NFS_COUNT
            ):
                msg_key = "share.nfs.connection.count.no.pass"
                self.err_msg += get_err_msg(LANG, msg_key, ctrl_id)
                return

        return
