# -*- coding: UTF-8 -*-
import traceback

import cliUtil
import common
from cbb.frame.util.common import fakeProgress
from common_utils import get_err_msg

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env

RISK_VERSION_DICT = {
    "V300R002C20": "V300R002C20SPH005",
    "V300R006C60": "",
    "V300R006C61": "V300R006C61SPH005",
    "V500R007C50": "",
    "V500R007C61": "V500R007C61SPH005",
}


@fakeProgress(py_java_env, totalSeconds=120, logger=LOGGER, interval=1)
def execute(cli):
    """
    smartIO2.0FC卡检查
    步骤1 以admin用户登录设备；
    步骤2 执行命令：show upgrade package，获取系统软件版本及热补丁版本信息；
    步骤3 执行命令：show port general physical_type=FC，
    检查FC端口的Max Speed是否存在32000的记录

    1 若步骤2中系统软件版本不为DoradoV300R002C20，V300R006C60，
    V300R006C61，V500R007C50，V500R007C61，则检查结果为不涉及，否则继续检查；
    2 若步骤2中系统热补丁为V300R002C20SPH005,V500R007C61SPH005,
    V300R006C61SPH005或更高版本热补丁，则检查结果为通过，否则继续检查；
    3 若步骤3中存在FC端口Max Speed为32000的记录，则检查结果为不通过，
    否则检查结果为通过。

    :param cli:
    :return:
    """
    smart_fc_check = SmartIo1822Fc(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = smart_fc_check.execute_check()
    return flag, "\n".join(smart_fc_check.all_ret_list), msg


class SmartIo1822Fc:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []

    def execute_check(self):
        try:

            (
                flag,
                soft_version,
                patch_version,
                cli_ret,
                err_msg,
            ) = common.get_soft_and_patch_version(self.cli, LOGGER, LANG)
            self.all_ret_list.append(cli_ret)
            # 检查是否为风险版本
            if self.check_risk_version(soft_version, patch_version):
                return True, ""
            err_msg_id_list = self.get_1822_fc_card_id()
            if not err_msg_id_list:
                return True, ""

            return (
                False,
                get_err_msg(
                    self.lang,
                    "fc.smart.io.check.warning",
                    (soft_version, patch_version, ", ".join(err_msg_id_list)),
                ),
            )
        except common.UnCheckException as e:
            LOGGER.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            LOGGER.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )

    def check_risk_version(self, soft_version, patch_version):
        """
        检查是否为风险版本
        :return: 有风险的为False
        """
        soft_version_no_spc = soft_version.split("SPC")[0]
        if soft_version_no_spc in RISK_VERSION_DICT:
            risk_hot_patch = RISK_VERSION_DICT.get(soft_version)
            if risk_hot_patch and patch_version >= risk_hot_patch:
                return True
            return False
        return True

    def get_1822_fc_card_id(self):
        """获取1822的FC卡信息
        :return: 返回1822的fc卡ID
        """
        not_pass_id_list = list()
        cmd = "show port general physical_type=FC"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(err_msg, cli_ret)
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return not_pass_id_list

        cards = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        for card in cards:
            card_id = card.get("ID", "--")
            max_speed = card.get("Max Speed(Mbps)", "")
            if max_speed.strip() == "32000":
                not_pass_id_list.append(card_id)

        self.logger.logInfo(
            "err 1822 FC Card is {}".format(str(not_pass_id_list))
        )
        return not_pass_id_list
