# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import cliUtil
import cli_util_cache as cache_util
import common
import common_cache as cache_common

from cbb.frame.cli.cli_with_cache import execute_cmd_in_cli_mode


# 查询双活信息的公共命令
COMMON_CMD_LIST = [
    "show hyper_metro_pair general |filterRow column=Type predict="
    "equal_to value=LUN|filterColumn include columnList=Local\sID,"
    "Remote\sID,ID,Domain\sID",
    "show hyper_metro_pair general |filterRow column=Type predict="
    "equal_to value=FS|filterColumn include columnList=Local\sID,"
    "Remote\sID,ID,Domain\sID",
    "show remote_device general",
    "show host general |filterColumn include columnList="
    "Operating\sSystem,Access\sMode,ID",
    "show initiator initiator_type=FC",
    "show initiator initiator_type=iSCSI",
    "show nvme_over_roce_initiator general",
    "show host general",
    "show upgrade package",
    "show system general",
    "show controller general",
    "show hyper_metro_domain general",
    "show hyper_metro_pair general",
    "show fs_hyper_metro_domain general",
    "show logical_port general|filterColumn include columnList="
    "Logical\sPort\sName,IPv4\sAddress,IPv4\sMask,IPv6\sAddress,IPv6\sMask",
    "show hyper_metro_pair general |filterRow column=Type predict=equal_to value=LUN|filterColumn include"
    " columnList=Local\sID,Remote\sID,ID,Domain\sID,Consistency\sGroup\sID"
]


class CreateHyperMetroCmd:
    def __init__(self, env, logger, cli, cmd_echo_dict, sn):
        self.env = env
        self.logger = logger
        self.cli = cli
        self.lang = common.getLang(env)
        self.cmd_echo_dict = cmd_echo_dict
        self.sn = sn
        py_obj = self.env.get("objectForPy")
        py_obj.put("productMode_{}".format(sn),
            self.env.get("devInfo").getProductModel())

    def create_command(self):
        """
        生成提前查询的命令
        :return:
        """
        cmd_list = self.get_all_hyper_cmd(
            self.cli, self.cmd_echo_dict, self.sn)
        cmd_list.extend(COMMON_CMD_LIST)
        self.logger.logInfo("multi thread cmd list:{}".format(cmd_list))
        return cmd_list

    def create_command_developer(self):
        """
        生成提前查询的命令(developer命令)
        :return:
        """
        cmd_list = self.get_all_developer_cli()
        self.logger.logInfo("multi developer thread cmd list:{}".format(cmd_list))
        return cmd_list

    def get_all_hyper_cmd(self, cli, cmd_echo_dict, sn):
        """
        待执行双活命令
        :param cli:
        :param cmd_echo_dict:
        :param sn:
        :return:
        """
        need_execute_cmd_list = []
        local_id_list, remote_id_list = self.get_pair_local_and_remote_id_list(
            cli, cmd_echo_dict, sn
        )
        # 查询双活主机详情
        need_execute_cmd_list.extend(
            self.get_hyper_host_cmd_list(cli, cmd_echo_dict, sn, local_id_list)
        )

        # 查询双活lun详情
        need_execute_cmd_list.extend(
            self.query_pair_local_obj(
                cli, cmd_echo_dict, sn, cache_common.CMD_SAN_PAIR,
                "show lun general lun_id={}",
            )
        )

        # 查询文件系统详情
        need_execute_cmd_list.extend(
            self.query_pair_local_obj(
                cli, cmd_echo_dict, sn, cache_common.CMD_NAS_PAIR,
                "show file_system general file_system_id={}",
            )
        )

        return need_execute_cmd_list

    def get_all_developer_cli(self):
        need_execute_cmd_list = []
        # 查询文件系统统计信息
        need_execute_cmd_list.extend(
            self.get_fs_statistic_info()
        )
        return need_execute_cmd_list

    def get_pair_local_id_list(self, cli, cmd, cmd_echo_dict, sn):
        """
        获取本地 local id
        :param cli:
        :param cmd:
        :param cmd_echo_dict:
        :param sn:
        :return:
        """
        if cmd in cmd_echo_dict:
            cli_ret = cmd_echo_dict.get(cmd, {}).get(common.CLI_RET_KEY)
        else:
            flag, cli_ret, err_msg = self.update_cmd_cache(cli, cmd, cmd_echo_dict, sn)
            cmd_echo_dict[cmd] = {
                common.FLAG_KEY: flag,
                common.CLI_RET_KEY: cli_ret,
                common.ERROR_MSG_KEY: err_msg,
            }
        pair_info_list = cliUtil.getHorizontalCliRet(cli_ret)
        return [pair_info.get("Local ID") for pair_info in pair_info_list]

    def get_san_pair_local_id_list(self, cli, cmd_echo_dict, sn):
        """
        获取SAN pair 本端id，有缓存，不会重复执行命令
        :param cli:
        :param cmd_echo_dict:
        :param sn:
        :return:
        """
        return self.get_pair_local_id_list(
            cli, cache_common.CMD_SAN_PAIR, cmd_echo_dict, sn
        )

    def get_nas_pair_local_id_list(self, cli, cmd_echo_dict, sn):
        """
        获取NAS pair本端id，有缓存，不会重复执行命令
        :param cli:
        :param cmd_echo_dict:
        :param sn:
        :return:
        """
        return self.get_pair_local_id_list(
            cli, cache_common.CMD_NAS_PAIR, cmd_echo_dict, sn
        )

    def get_pair_local_and_remote_id_list(
        self, cli, cmd_echo_dict, sn, cmd=cache_common.CMD_SAN_PAIR
    ):
        flag, ret, msg = self.update_cmd_cache(cli, cmd, cmd_echo_dict, sn)
        ret_list = cliUtil.getHorizontalNostandardCliRet(ret)
        local_id_list = []
        remote_id_list = []
        for pair_info in ret_list:
            local_id_list.append(pair_info.get("Local ID"))
            remote_id_list.append(pair_info.get("Remote ID"))
        return local_id_list, remote_id_list

    def query_pair_local_obj(self, cli, cmd_echo_dict, sn, pair_cmd,
                             obj_detail_cmd):
        need_execute_cmd_list = []
        local_id_list, remote_id_list = self.get_pair_local_and_remote_id_list(
            cli, cmd_echo_dict, sn, pair_cmd
        )

        need_execute_cmd_list.extend(
            [obj_detail_cmd.format(local_id) for local_id in local_id_list]
        )
        return need_execute_cmd_list

    def get_hyper_host_cmd_list(self, cli, cmd_echo_dict, sn, lun_id_list):
        """
        获取双活主机相关信息
        :param cli:
        :param cmd_echo_dict:
        :param sn:
        :param lunIdList:
        :return:
        """
        host_lun_cmd_list = []
        hy_host_list = []
        py_obj = self.env.get("objectForPy")
        if cache_util.is_downloaded_config_back_end(py_obj, sn):
            self.logger.logInfo("pr has downloaded config.txt {0}!".format(sn))
            hy_host_list = cache_util.get_hyper_host_from_config(
                self.env, sn, self.logger, lun_id_list
            )
        self.logger.logInfo("hyper host sn:{0}: {1}".format(sn, str(hy_host_list)))
        cmd = "show host general |filterColumn include columnList=ID"
        flag, cli_ret, err_msg = self.update_cmd_cache(cli, cmd, cmd_echo_dict, sn)
        hosts = cliUtil.getHorizontalNostandardCliRet(cli_ret)
        tmp_host_id_list = [host.get("ID") for host in hosts]
        # 如果已经过滤了双活主机，则只查双活主机，查出Host lun id
        if hy_host_list:
            tmp_host_id_list = hy_host_list
        self.logger.logInfo("hy_host_list is {}".format(tmp_host_id_list))

        local_id_list, remote_id_list = self.get_pair_local_and_remote_id_list(
            cli, cmd_echo_dict, sn
        )
        if not local_id_list:
            return host_lun_cmd_list

        cmd = "show host lun host_id={}"
        for host_id in tmp_host_id_list:
            host_lun_cmd_list.append(cmd.format(host_id))

        return host_lun_cmd_list

    def update_cmd_cache(self, cli, cmd, cmd_dict, sn):
        """
        执行命令并缓存数据库
        :param cli: 连接
        :param cmd: 命令
        :param cmd_dict: 回文缓存
        :param sn:
        """
        context = self.env.get("objectForPy")
        flag, cli_ret, err_msg = execute_cmd_in_cli_mode(
            context, cli, cmd, self.lang, sn, self.logger
        )
        cmd_dict[cmd] = {
            common.FLAG_KEY: flag,
            common.CLI_RET_KEY: cli_ret,
            common.ERROR_MSG_KEY: err_msg,
        }
        return flag, cli_ret, err_msg

    # 查询各个文件系统的容量使用情况
    def get_fs_statistic_info(self):
        wrong_word_version = [
            "V300R006C00SPC100",
            "V300R005C00SPC300",
            "V300R005C01",
            "V300R003C00SPC100",
            "V300R003C10SPC100",
            "V300R003C20",
            "V300R003C20SPC100",
            "V300R003C20SPC200"
        ]
        cli_ret, current_version, current_patch = common.getHotPatchVersionAndCurrentVersion(self.cli, self.lang)
        special_word = "Reserved"
        if current_version in wrong_word_version:
            special_word = "Resvered"
        statistic_cmd_list = []
        fs_list = self.get_fs_list()
        for item in fs_list:
            fs_id = item.get("ID")
            fs_controller = item.get("Work Controller")
            # 获取文件系统的统计信息
            cmd_tmp = 'show space statistic_info file_system_id={} controller={}|filterColumn include' \
                     ' columnList=File\sSystem\sTotal,File\sSystem\sUsed,SPA\sQuota\sData,SPA\sQuota\sMeta,' \
                     'Storage\sPool\sAvailable,Grain\sReserve\sData,Grain\sReserve\sMeta,SPC\sReserve\sTotal,' \
                     'SPC\sReserve\sUsed,DDP\sSaved\sDedup,Snapshot\s{}\sFree,DDP\sSaved\sCompress'
            statistic_cmd = cmd_tmp.format(fs_id, fs_controller, special_word)
            self.logger.logInfo(statistic_cmd)
            statistic_cmd_list.append(statistic_cmd)
        return statistic_cmd_list

    # 获取所有文件系统的id，控制器，type信息
    def get_fs_list(self):
        cmd_echo_dict = self.cmd_echo_dict
        cmd = "show file_system general|filterColumn include columnList=ID,Work\sController,Type"
        flag, ret, msg = self.update_cmd_cache(self.cli, cmd, cmd_echo_dict, self.sn)
        # 所有文件系统列表
        if flag is not True:
            return []
        fs_list = cliUtil.getHorizontalNostandardCliRet(ret)
        return fs_list
