# -*- coding: UTF-8 -*-
from functools import wraps, partial
import time
import re
import traceback
# noinspection PyUnresolvedReferences,PyUnresolvedReferences
from com.huawei.ism.tool.obase.connection import SshConnection
import common
import cliUtil
from logger import Logger


class CliView:
    CLI_USR_VIEW = '\w+:/>'
    CLI_DEVELOPER_VIEW = 'developer:/>'
    CLI_DIAGNOSE_VIEW = '/diagnose>'
    CLI_MINISYSTEM_VIEW = 'minisystem>'

    CLI_VIEW_FLG_MAG = {CLI_USR_VIEW: 'admin:/>',
                        CLI_DEVELOPER_VIEW: 'developer:/>',
                        CLI_DIAGNOSE_VIEW: '/diagnose>',
                        CLI_MINISYSTEM_VIEW: 'minisystem>',
                        }

    def __init__(self):
        pass


class CliException(Exception):
    """Cli Exception.
    """
    def __init__(self, errMsgKey, errMsgPara=None):
        self.errMsgKey = errMsgKey
        self.errMsgPara = errMsgPara

    def getErrMsgKey(self):
        return self.errMsgKey

    def getErrMsgPara(self):
        return self.errMsgPara


def countTime(logger):
    """记录函数的执行时间。

    :param logger:
    :return:
    """
    def outer(func):
        @wraps(func)
        def inner(*args, **kwargs):
            start = time.time()
            result = func(*args, **kwargs)
            end = time.time()
            logger.logInfo('Time used(s):%f' % (end - start))
            return result
        return inner
    return outer


def isSshConnected(cliCon):
    return cliCon and cliCon.isConnected()


# noinspection PyBroadException
def reconnectSsh(cliCon):
    """重新连接SSH（如果连接断开）。

    :param cliCon: SSH connection
    :return: isConnected, isReconnected
    """
    if not cliCon:
        raise CliException('cli.connection.invalid')

    if cliCon and cliCon.isConnected():
        return True, False

    retryTimes = 0
    while not cliCon.isConnected() and retryTimes < 3:
        try:
            cliCon.reConnect()
        except:
            retryTimes += 1
            time.sleep(retryTimes * 10)
        else:
            return True, True
    else:
        return False, False


def reConnectAndHandleView(cli, cliView, logger):
    """重连SSH，并恢复到指定的视图（用户视图、研发视图、小系统视图、诊断视图）

    :param cli:
    :param cliView:
    :param logger:
    :return:
    """
    isConnected, isReconnected = reconnectSsh(cli)
    if isConnected:
        if isReconnected:
            safeLogInfo(logger, 'SSH connection reconnected success, entering given cli view.')
            return enterSpecifiedCliView(cli, cliView, logger)
        else:
            safeLogInfo(logger, 'SSH connection still connected.')
            return True
    else:
        safeLogError(logger, 'SSH connection reconnected failed, return now.')
        return False


def exeCmdSafely(cli, cmd, logger, logged=True, timeOut=3 * 60, reconnect=True, retry=True, retryTimes=3,
                 retryDelay=10):
    """执行命令，延迟重试3次（连接中断再重连3次），返回命令是否执行成功和命令回显。如果命令执行失败，则表明网络异常。

    :param cli: CLI连接
    :param cmd: 执行的命令
    :param logger: 日志对象
    :param logged: 是否记录日志
    :param timeOut: 超时时间
    :param reconnect: 是否重连
    :param retry: 是否重试
    :param retryTimes: 重试次数
    :param retryDelay: 重试时延
    :return: 命令是否执行成功、命令回显
    """
    if not cli:
        raise CliException('cli.connection.invalid')

    retryCnt = 0
    while retryCnt < retryTimes:
        # noinspection PyBroadException
        try:
            if logged:
                cliRet = cli.execCmdWithTimout(cmd, timeOut)
            else:
                cliRet = cli.execCmdNoLogTimout(cmd, timeOut)
        except:
            if not retry:
                return False, ''

            retryCnt += 1
            time.sleep(retryCnt * retryDelay)
            if reconnect:
                isConnected, isReconnected = reconnectSsh(cli)
                if isConnected:
                    if isReconnected:
                        safeLogInfo(logger, 'SSH connection reconnected success.')
                    else:
                        safeLogInfo(logger, 'SSH connection still connected.')
                else:
                    safeLogError(logger, 'SSH connection reconnected failed.')
                    return False, ''
            elif not isSshConnected(cli):
                safeLogError(logger, 'SSH connection closed, return now.')
                return False, ''
        else:
            return True, cliRet

    return False, ''


exeCmdWithRetryAndReconnect = partial(exeCmdSafely, timeOut=3 * 60, reconnect=True, retry=True, retryTimes=3,
                                      retryDelay=10)
"""
执行命令并重试或者重连。
"""

exeCmdWithRetry = partial(exeCmdSafely, timeOut=1 * 60, reconnect=False, retry=True, retryTimes=3, retryDelay=10)
"""
执行CLI命令，失败会重试（连接断开后不会重连）
"""

exeCmdWithRetryNoLog = partial(exeCmdSafely, logged=False, timeOut=1 * 60, reconnect=False, retry=True, retryTimes=3,
                               retryDelay=10)
"""
执行CLI命令，失败会重试（连接断开后不会重连），不记录日志（输入敏感信息）
"""


def enterSpecifiedCliView(cli, cliView=None, logger=None):
    """进入到指定的CLI视图（用户视图、研发视图、诊断视图）

    :param cli:
    :param cliView:
    :param logger:
    :return:
    """
    if not cli:
        raise CliException('cli.connection.invalid')

    if not cliView:
        return False

    if cliView == CliView.CLI_USR_VIEW:
        return enterUserView(cli, logger)
    elif cliView == CliView.CLI_DEVELOPER_VIEW:
        return enterDeveloperView(cli, logger)
    elif cliView == CliView.CLI_DIAGNOSE_VIEW:
        return enterDiagnoseView(cli, logger)
    elif cliView == CliView.CLI_MINISYSTEM_VIEW:
        return enterMinisystemView(cli, logger)
    else:
        raise CliException('invalid.cli.view.type')


def isInUserView(cliRet):
    """根据CLI回显信息判断当前CLI是否在用户视图。

    :param cliRet:
    :return:
    """
    if re.search(CliView.CLI_USR_VIEW, cliRet, re.I) and not re.search(CliView.CLI_DEVELOPER_VIEW, cliRet, re.I):
        return True
    return False


def isInDeveloperView(cliRet):
    """根据CLI回显信息判断当前CLI是否在研发视图。

    :param cliRet:
    :return:
    """
    if re.search(CliView.CLI_DEVELOPER_VIEW, cliRet, re.IGNORECASE):
        return True
    return False


def isInMinisystemView(cliRet):
    """根据CLI回显信息判断当前CLI是否在小系统视图。

    :param cliRet:
    :return:
    """
    if re.search(CliView.CLI_MINISYSTEM_VIEW, cliRet, re.IGNORECASE):
        return True
    return False


def isInDiagnoseView(cliRet):
    """根据CLI回显信息判断当前CLI是否在诊断视图。

    :param cliRet:
    :return:
    """
    if re.search(CliView.CLI_DIAGNOSE_VIEW, cliRet, re.IGNORECASE):
        return True
    return False


def isDeveloperSwitchOn(cli, logger=None):
    """查询研发视图开关是否打开。

    :param cli: CLI连接。
    :param logger:日志对象。
    :return: 查询是否成功、视图是否打开
    """
    isCmdSucc, cliRet = exeCmdInUsrView(cli, 'show user_mode enabled', logger)
    if not isCmdSucc:
        return False, False

    if re.search('Enabled', cliRet, re.I):
        return True, True
    else:
        return True, False


def openDeveloperSwitch(cli, logger=None):
    """打开研发视图开关。

    :param cli:
    :param logger:
    :return:视图是否成功打开。
    """
    isCmdSucc, cliRet = exeCmdInUsrView(cli, 'show user_mode enabled', logger)
    if not isCmdSucc:
        return False

    if re.search('Enabled', cliRet, re.I):
        return True

    cliRet = exeCmdInUsrView(cli, 'change user_mode enabled user_mode=developer enabled=yes', logger)
    if re.search('success', cliRet, re.I):
        return True

    raise CliException('cannot.access.developer.mode')


def closeDeveloperSwitch(cli, logger=None):
    """关闭研发视图开关。

    :param cli:
    :param logger:
    :return:
    """
    isCmdSucc, cliRet = exeCmdInUsrView(cli, 'show user_mode enabled', logger)
    if not isCmdSucc:
        safeLogError(logger, 'Close developer switch failed.')
        return

    if re.search('Disabled', cliRet, re.I):
        return

    cliRet = exeCmdInUsrView(cli, 'change user_mode enabled user_mode=developer enabled=no', logger)
    if re.search('success', cliRet, re.I):
        return


def enterUserView(cli, logger=None):
    """进入用户视图。

    :param cli:
    :param logger:
    :return:是否成功进入用户视图。
    """
    isCmdSucc, cliRet = exeCmdWithRetryAndReconnect(cli, 'show system general', logger)
    if not isCmdSucc:
        safeLogError(logger, 'Enter user view failed.')
        return False

    while not isInUserView(cliRet):
        isCmdSucc, cliRet = exeCmdWithRetry(cli, 'exit', logger)
        if not isCmdSucc:
            safeLogError(logger, 'Executing exit command failed, return.')
            return False

        while 'y/n' in cliRet:
            isCmdSucc, cliRet = exeCmdWithRetry(cli, 'y', logger)
            if not isCmdSucc:
                safeLogError(logger, 'Executing exit command failed, return.')
                return False

    return True


def wrapException(lan, logger):
    def outer(func):
        def inner(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except CliException, ce:
                safeLogError(logger, str(traceback.format_exc()))
                errMsgKey = ce.getErrMsgKey()
                errMsgPara = ce.getErrMsgPara()
                errMsg = common.getMsg(lan, errMsgKey, errMsgPara) \
                    if errMsgPara else common.getMsg(lan, errMsgKey)
                return cliUtil.RESULT_NOCHECK, '', errMsg
        return inner
    return outer


def usrPrivilegeRequired(javaEnv, lvlList, logger):
    def outer(func):
        @wraps(func)
        def inner(*args, **kwargs):
            cliCon = getCliArg(*args)
            if not cliCon:
                raise CliException('CLI.connection.invalid')
            usrLvl = getUserLevel(cliCon, javaEnv, logger)
            if usrLvl not in lvlList:
                raise CliException('loginUser.name.level.guest.warning')

            return func(*args, **kwargs)
        return inner
    return outer


def dependingDeveloperSwitchOn(func):
    """Decorator for opening developer view switch.

    :param func:
    :return:
    """
    @wraps(func)
    def inner(*args, **kwargs):
        cliCon = getCliArg(*args)
        if not cliCon:
            raise CliException('CLI.connection.invalid')

        logger = getLoggerArg(args)
        openDeveloperSwitch(cliCon, logger)
        return func(*args, **kwargs)
    return inner


@dependingDeveloperSwitchOn
def enterMinisystemView(cli, logger):
    """Enter into minisystem view from developer view.

    :param logger:
    :param cli:
    :return:
    """
    isCmdSucc, testRet = exeCmdWithRetryAndReconnect(cli, 'show system general', logger)
    if not isCmdSucc:
        safeLogError(logger, 'Enter minisystem view failed.')
        return False

    if isInMinisystemView(testRet):
        return True
    elif isInDiagnoseView(testRet):
        isCmdSucc, cliRet = exeCmdWithRetry(cli, 'exit', logger)
        if not isCmdSucc:
            safeLogError(logger, 'Enter minisystem view failed.')
            return False

        if isInDeveloperView(cliRet):
            isCmdSucc, cliRet = exeCmdWithRetry(cli, 'minisystem', logger)
            if not isCmdSucc:
                safeLogError(logger, 'Enter minisystem view failed.')
                return False

            if isInMinisystemView(cliRet):
                return
            return False
    elif isInDeveloperView(testRet):
        isCmdSucc, cliRet = exeCmdWithRetry(cli, 'minisystem', logger)
        if not isCmdSucc:
            safeLogError(logger, 'Enter minisystem view failed.')
            return False

        if isInMinisystemView(cliRet):
            return

    cmd = "change user_mode current_mode user_mode=developer"
    isCmdSucc, cliRet = exeCmdWithRetryAndReconnect(cli, cmd, logger)
    if not isCmdSucc:
        safeLogError(logger, 'Enter minisystem view failed.')
        return False

    while 'y/n' in cliRet:
        isCmdSucc, cliRet = exeCmdWithRetry(cli, 'y', logger)
        if not isCmdSucc:
            safeLogError(logger, 'Enter minisystem view failed.')
            return False

    if isInDeveloperView(cliRet):
        isCmdSucc, cliRet = exeCmdWithRetry(cli, 'minisystem', logger)
        if not isCmdSucc:
            safeLogError(logger, 'Enter minisystem view failed.')
            return False

        if isInMinisystemView(cliRet):
            return True

    return False


def usrViewRequired(func):
    """Execute command in developer view.
    """
    @wraps(func)
    def inner(*args, **kwargs):
        cliCon = getCliArg(*args)
        if not cliCon:
            raise CliException('CLI.connection.invalid')

        logger = getLoggerArg(args)
        enterUserView(cliCon, logger)
        result = func(*args, **kwargs)
        return result
    return inner


def developerViewRequired(func):
    """Execute command in developer view.
    """
    @wraps(func)
    def inner(*args, **kwargs):
        cliCon = getCliArg(*args)
        if not cliCon:
            raise CliException('CLI.connection.invalid')

        logger = getLoggerArg(args)
        enterDeveloperView(cliCon, logger)
        result = func(*args, **kwargs)
        return result
    return inner


def minisystemViewRequired(func):
    """Execute command in minisystem view.
    """
    @wraps(func)
    def inner(*args, **kwargs):
        cliCon = getCliArg(*args)
        if not cliCon:
            raise CliException('CLI.connection.invalid')
        logger = getLoggerArg(*args)

        enterMinisystemView(cliCon, logger)
        result = func(*args, **kwargs)
        return result
    return inner


def diagnoseViewRequired(func):
    """Execute command in diagnose view.
    """
    @wraps(func)
    def inner(*args, **kwargs):
        cliCon = getCliArg(*args)
        if not cliCon:
            raise CliException('cli.connection.invalid')

        logger = getLoggerArg(*args)
        enterDiagnoseView(cliCon, logger)
        result = func(*args, **kwargs)
        return result
    return inner


def restoreUserView(func):
    """Restore CLI view to user view.
    """
    @wraps(func)
    def inner(*args, **kwargs):
        cliCon = getCliArg(*args)
        if not cliCon:
            raise CliException('cli.connection.invalid')
        logger = getLoggerArg(*args)
        try:
            result = func(*args, **kwargs)
        finally:
            enterUserView(cliCon, logger)
        return result
    return inner


@dependingDeveloperSwitchOn
def enterDeveloperView(cli, logger):
    """Enter into developer view.

    :param logger:
    :param cli:
    :return:
    """
    isCmdSucc, testRet = exeCmdWithRetryAndReconnect(cli, 'show system general', logger)
    if not isCmdSucc:
        safeLogError(logger, 'Enter developer failed, return.')
        return False

    if isInDeveloperView(testRet):
        return True

    elif isInMinisystemView(testRet) or isInDiagnoseView(testRet):
        isCmdSucc, testRet = exeCmdWithRetry(cli, 'exit', logger)
        if not isCmdSucc:
            safeLogError(logger, 'Executing exit command failed.')
            return False

        while 'y/n' in testRet:
            isCmdSucc, testRet = exeCmdWithRetry(cli, 'y', logger)
            if not isCmdSucc:
                safeLogError(logger, 'Executing y command failed.')
                return False

        if isInDeveloperView(testRet):
            return True
    else:
        isCmdSucc, cliRet = exeCmdWithRetryAndReconnect(cli, 'change user_mode current_mode user_mode=developer',
                                                        logger)
        if not isCmdSucc:
            safeLogError(logger, 'Enter developer failed, return.')
            return False

        while 'y/n' in cliRet:
            isCmdSucc, cliRet = exeCmdWithRetry(cli, 'y', logger)
            if not isCmdSucc:
                safeLogError(logger, 'Executing y command failed.')
                return False

        if isInDeveloperView(cliRet):
            return True
        return False


@dependingDeveloperSwitchOn
def enterDiagnoseView(cli, logger):
    """Enter into diagnose view from developer view.

    :param logger:
    :param cli:
    :return:
    """
    isCmdSucc, testRet = exeCmdWithRetryAndReconnect(cli, 'show system general', logger)
    if not isCmdSucc:
        safeLogError(logger, 'Enter developer failed, return.')
        return False

    if isInDiagnoseView(testRet):
        return True

    elif isInMinisystemView(testRet):
        isCmdSucc, cliRet = exeCmdWithRetry(cli, 'exit', logger)
        if not isCmdSucc:
            safeLogError(logger, 'Executing exit command failed.')
            return False

        while 'y/n' in cliRet:
            isCmdSucc, cliRet = exeCmdWithRetry(cli, 'y', logger)
            if not isCmdSucc:
                safeLogError(logger, 'Executing y command failed.')
                return False

        if isInDeveloperView(cliRet):
            isCmdSucc, cliRet = exeCmdWithRetry(cli, 'debug', logger)
            if not isCmdSucc:
                safeLogError(logger, 'Executing debug command failed.')
                return False

            if isInDiagnoseView(cliRet):
                return True
        else:
            safeLogError(logger, 'Error happened when entering diagnose view.')
            return False

    elif isInDeveloperView(testRet):
        isCmdSucc, cliRet = exeCmdWithRetry(cli, 'debug', logger)
        if not isCmdSucc:
            safeLogError(logger, 'Executing debug command failed.')
            return False
        if isInDiagnoseView(cliRet):
            return True

        safeLogError(logger, 'Error happened when entering diagnose view.')
        return False
    else:
        isCmdSucc, cliRet = exeCmdWithRetryAndReconnect(cli, 'change user_mode current_mode user_mode=developer',
                                                        logger)
        if not isCmdSucc:
            safeLogError(logger, 'Executing change user_mode current_mode user_mode=developer command failed.')
            return False

        while 'y/n' in cliRet:
            isCmdSucc, cliRet = exeCmdWithRetry(cli, 'y', logger)
            if not isCmdSucc:
                safeLogError(logger, 'Executing y command failed.')
                return False

        if isInDeveloperView(cliRet):
            isCmdSucc, cliRet = exeCmdWithRetry(cli, 'debug', logger)
            if not isCmdSucc:
                safeLogError(logger, 'Executing debug command failed.')
                return False

            if isInDiagnoseView(cliRet):
                return True

        return False


def getSpecifiedArg(func, argName, *args):
    """Get specified argument of a function calling according to specifying argument name.

    :param func: function being calling
    :param argName: argument name
    :param args: Arguments when calling function.
    :return:
    """
    if not args or not func:
        return None

    funcArgNames = func.func_code.co_varnames
    for arg in funcArgNames:
        if argName == arg:
            return args[funcArgNames.index(arg)]
    else:
        return None


def getCliArg(*args):
    """Get CLI connection argument when calling a function.

    :param args:
    :return:
    """
    if not args:
        return None
    for arg in args:
        if isinstance(arg, SshConnection):
            return arg
    else:
        return None


def getLoggerArg(*args):
    """Get logger argument when calling a function.

    :param args:
    :return:
    """
    if not args:
        return None
    for arg in args:
        if isinstance(arg, Logger):
            return arg
    else:
        return None


@minisystemViewRequired
def heartbeatToPeer(cliCon, loginPwd, logger):
    isHeartbeatSucc, heartbeatRet = exeCmdInMinisystemView(cliCon, 'sshtoremote', logger)
    if not isHeartbeatSucc:
        return False, heartbeatRet

    hearBeatCliRetList = []
    hearBeatCliRetList.append(heartbeatRet)

    if 'yes/no' in heartbeatRet:
        isCmdSucc, heartbeatRet = exeCmdWithRetry(cliCon, 'yes', logger)
        hearBeatCliRetList.append(heartbeatRet)
        if not isCmdSucc:
            return False, '\n'.join(hearBeatCliRetList)

    if 'password' in heartbeatRet.lower():
        isCmdSucc, heartbeatRet = exeCmdWithRetryNoLog(cliCon, loginPwd, logger)
        hearBeatCliRetList.append(heartbeatRet)
        if not isCmdSucc:
            return False, '\n'.join(hearBeatCliRetList)

        if isInUserView(heartbeatRet):
            return True, '\n'.join(hearBeatCliRetList)

    return False, '\n'.join(hearBeatCliRetList)


def heartbeatToCtrl(cliCon, nid, loginPwd, logger):
    heartbeatCmd = 'sshtoremoteExt %(nodeId)s' % {'nodeId': nid}
    isHeartbeatSucc, heartbeatRet = exeCmdInMinisystemView(cliCon, heartbeatCmd, logger)
    if not isHeartbeatSucc:
        return False, heartbeatRet

    hearBeatCliRetList = []
    hearBeatCliRetList.append(heartbeatRet)

    if 'yes/no' in heartbeatRet:
        isCmdSucc, heartbeatRet = exeCmdWithRetry(cliCon, 'yes', logger)
        hearBeatCliRetList.append(heartbeatRet)
        if not isCmdSucc:
            safeLogError(logger, 'Executing yes command failed.')
            return False, '\n'.join(hearBeatCliRetList)

    if 'password' in heartbeatRet.lower():
        isCmdSucc, heartbeatRet = exeCmdWithRetryNoLog(cliCon, loginPwd, logger)
        hearBeatCliRetList.append(heartbeatRet)
        if not isCmdSucc:
            safeLogError(logger, 'Executing login password command failed.')
            return False, '\n'.join(hearBeatCliRetList)

        if isInUserView(heartbeatRet):
            return True, '\n'.join(hearBeatCliRetList)

    return False, '\n'.join(hearBeatCliRetList)


def getLoginUser(javaEnv):
    return javaEnv.get("devInfo").getLoginUser().getUserName()


def getUserLevel(cli, javaEnv, logger):
    usrName = getLoginUser(javaEnv)
    isCmdSucc, cliRet = exeCmdInUsrView(cli, 'show user user_name=%s' % usrName, logger)
    if not isCmdSucc:
        return False, '', ''

    for line in cliRet.splitlines():
        line = line.strip()
        if not line:
            continue

        fields = line.split()
        if fields[0] == usrName:
            return True, fields[1].lower(), cliRet
    else:
        return True, '', cliRet


def is6U4CtrlDevice(cli, logger):
    isCmdSucc, cliRet = exeCmdInUsrView(cli, 'show enclosure', logger)
    if not isCmdSucc:
        return False, False

    if re.search('6U 4 Controllers Enclosure', cliRet, re.I):
        return True, True
    else:
        return False, False


def safeLogError(logger, errMsg):
    if not logger or not isinstance(logger, Logger):
        return
    else:
        logger.logError(errMsg)


def safeLogInfo(logger, infoMsg):
    if not logger or not isinstance(logger, Logger):
        return
    else:
        logger.logInfo(infoMsg)


def exeCmdInSpecifiedView(cli, cmd, logger, cliView, timeOut=3 * 60, reconnect=True, retry=True, retryTimes=3,
                          retryDelay=10):
    """在指定CLI视图（用户视图、研发视图、小系统视图、诊断视图）下执行CLI命令（超时时间3min），失败重连重试）。

    :param cli: CLI连接
    :param cmd: 执行的命令
    :param logger: 日志对象
    :param cliView: CLI视图
    :param timeOut: 超时时间
    :param reconnect: 是否重连
    :param retry: 是否重试
    :param retryTimes: 重试次数
    :param retryDelay: 重试时延
    :return: 命令是否执行成功、命令回显
    """
    if not cli:
        raise CliException('cli.connection.invalid')

    retryCnt = 0
    while retryCnt < retryTimes:
        # noinspection PyBroadException
        try:
            if not enterSpecifiedCliView(cli, cliView, logger):
                return False, ''

            cliRet = cli.execCmdWithTimout(cmd, timeOut)
        except:
            safeLogError(logger, str(traceback.format_exc()))
            if not retry:
                return False, ''

            retryCnt += 1
            time.sleep(retryCnt * retryDelay)

            if reconnect and not reConnectAndHandleView(cli, cliView, logger):
                safeLogError(logger, 'Reconnect and handle CLI view failed.')
                return False, ''
            elif not isSshConnected(cli):
                safeLogError(logger, 'SSH not connected.')
                return False, ''
        else:
            return True, cliRet
    else:
        return False, ''


exeCmdInUsrView = partial(exeCmdInSpecifiedView, cliView=CliView.CLI_USR_VIEW)
"""
在用户视图下执行CLI命令（超时时间3min），失败重连重试。
"""

exeCmdInDeveloperView = partial(exeCmdInSpecifiedView, cliView=CliView.CLI_DEVELOPER_VIEW)
"""
在研发视图下执行CLI命令（超时时间3min），失败重连重试。
"""

exeCmdInMinisystemView = partial(exeCmdInSpecifiedView, cliView=CliView.CLI_MINISYSTEM_VIEW)
"""
在小系统视图下执行CLI命令（超时时间3min），失败重连重试。
"""

exeCmdInDiagnoseView = partial(exeCmdInSpecifiedView, cliView=CliView.CLI_DIAGNOSE_VIEW)
"""
在诊断视图下执行CLI命令（超时时间3min），失败重连重试。
"""