# coding:utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2022. All rights reserved.
"""
日志文件处理模块：包括包解压，提取指定文件、重命令等，最后按控制器分组，并按时间升序排列
每一种日志文件对应一个处理类，皆继承至BaseProcess
LogExtract会自动扫描BaseProcess的所有子类，找出日志文件对应的处理类进行处理。
"""
import os
import shutil
import re
from cbb.frame.util.tar_util import decompress_tar_special_file, \
    decompress_tar_all_file
import traceback


class LogExtract(object):
    """
    从日志包中提取需要的文件，并按格式对文件重命名
    日志文件目录文件示例：
    0A_messages
    0A_his_tar_0000000003_20201028_155844_powerXX.tgz
        (cur_debug/mcelog;cur_debug/messages)
    0A_messages_0000000001_20201028_203140.tgz
    OA_mcelog
    OA_mcelog_0000000003_20200624_233449.tgz

    0B_messages
    0B_his_tar_0000000003_20201028_155844_powerXX.tgz
        (cur_debug/mcelog;cur_debug/messages)
    0B_messages_0000000001_20201028_203140.tgz
    OB_mcelog
    OB_mcelog_0000000003_20200624_233449.tgz

    :param log_path: 日志路径
    :param logger: 日志对象
    :return: {0A：{mcelog:[], messages:[]}, {0B: {mcelog:[], messages:[]}}}
    """
    def __init__(self, log_path, logger):
        self.log_path = log_path
        self.logger = logger
        self.dest_log_path = self.log_path + "\\dest\\"
        self.tmp_log_path = self.log_path + "\\temp\\"
        self.ret = {}

    def execute(self):
        self.logger.info("begin to log sorting.")
        list_files = os.listdir(self.log_path)

        shutil.rmtree(self.dest_log_path, True)
        if not os.path.exists(self.dest_log_path):
            os.mkdir(self.dest_log_path)

        for file_name in list_files:
            self.extract_log_file(file_name)
        self.logger.info("collected log before sort:{}".format(self.ret))
        return self.sort_by_time()

    def sort_by_time(self):
        sorted_log_files = {}
        for ctrl_id, log_record in self.ret.items():
            sorted_log_files[ctrl_id] = {}
            for log_type, file_list in log_record.items():
                sorted_log_files[ctrl_id].setdefault(log_type, sorted(file_list))
        self.logger.info("collected log after sort:{}".format(sorted_log_files))
        return sorted_log_files

    def extract_log_file(self, file_name):
        self.logger.info("begin to process file:{}".format(file_name))
        shutil.rmtree(self.tmp_log_path, True)
        for bp_cls in BaseProcess.__subclasses__():
            if bp_cls(self.logger, self.log_path, file_name, self.ret).do_process():
                return


class BaseProcess(object):
    def __init__(self, logger, log_path, file_name, result_dict):
        self.logger = logger
        self.log_path = log_path
        self.file_name = file_name
        self.dest_log_path = self.log_path + "\\dest\\"
        self.tmp_log_path = self.log_path + "\\temp\\"
        self.result_dict = result_dict

    def do_process(self):
        try:
            return self.process()
        except Exception:
            self.logger.error("un tgz file error. file name={}, trace={}".format(
                self.file_name, traceback.format_exc()))
            return False

    def process(self):
        raise NotImplementedError()

    def un_tgz_file(self, file_name, dest_path, extract_files=None):
        if not os.path.isdir(dest_path):
            os.mkdir(dest_path)
        if not extract_files:
            # 解压全部
            decompress_tar_all_file(os.path.join(self.log_path, file_name), dest_path, "r")
        else:
            # 解压指定的文件列表
            decompress_tar_special_file(os.path.join(self.log_path, file_name), dest_path, extract_files, "r")

    def record_file(self, ctrl_id, file_time, file_type, log_file):
        if os.path.exists(log_file):
            dest_path = self.dest_log_path + "{}_{}_{}".format(
                ctrl_id, file_type, file_time)
            shutil.copy(log_file, dest_path)
            self.save_result(ctrl_id, file_type, dest_path)

    def save_result(self, ctrl_id, file_type, file_path):
        if ctrl_id not in self.result_dict:
            self.result_dict[ctrl_id] = {}
        if file_type not in self.result_dict[ctrl_id]:
            self.result_dict[ctrl_id][file_type] = []
        self.result_dict[ctrl_id][file_type].append(file_path)


class HisTarProcess(BaseProcess):
    """
    对0A_his_tar_0000000001_20210513_133708.tgz日志文件的处理
    """
    def __init__(self, *args):
        super(HisTarProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_his_tar_\d+_(\d+_\d+)", self.file_name)
        if not match_obj:
            return False
        ctrl_id, file_time = match_obj.group(1), match_obj.group(2)
        self.un_tgz_file(self.file_name, self.tmp_log_path,
                         ["cur_debug/messages", "cur_debug/mcelog", "nvram/log_reset.txt"])
        message_log_file = self.tmp_log_path + "cur_debug\\messages"
        self.record_file(ctrl_id, file_time,
                         "messages", message_log_file)

        mce_log_file = self.tmp_log_path + "cur_debug\\mcelog"
        self.record_file(ctrl_id, file_time,
                         "mcelog", mce_log_file)

        reset_log_file = self.tmp_log_path + "nvram\\log_reset.txt"
        self.record_file(ctrl_id, file_time,
                         "reset", reset_log_file)
        return True


class HisRasDaemonProcess(BaseProcess):
    """
    对0A_rasdaemon_0000000001_20210513_133708.tgz日志文件的处理
    """
    def __init__(self, *args):
        super(HisRasDaemonProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_rasdaemon_\d+_(\d+_\d+)", self.file_name)
        if not match_obj:
            return False
        ctrl_id, file_time = match_obj.group(1), match_obj.group(2)
        self.un_tgz_file(self.file_name, self.tmp_log_path)
        for mce_name in os.listdir(self.tmp_log_path):
            mce_log_file = self.tmp_log_path + mce_name
            self.record_file(ctrl_id, file_time, "rasdaemon",
                             mce_log_file)
        return True


class HisMceLogProcess(BaseProcess):
    """
    对0A_mcelog_0000000001_20210513_133708.tgz日志文件的处理
    """
    def __init__(self, *args):
        super(HisMceLogProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_mcelog_\d+_(\d+_\d+)", self.file_name)
        if not match_obj:
            return False
        ctrl_id, file_time = match_obj.group(1), match_obj.group(2)
        self.un_tgz_file(self.file_name, self.tmp_log_path)
        for mce_name in os.listdir(self.tmp_log_path):
            mce_log_file = self.tmp_log_path + mce_name
            self.record_file(ctrl_id, file_time, "mcelog",
                             mce_log_file)
        return True


class HisMessagesProcess(BaseProcess):
    """
    对0A_messages_0000000008_20210522_221133.tgz日志文件的处理
    """
    def __init__(self, *args):
        super(HisMessagesProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_messages_\d+_(\d+_\d+)", self.file_name)
        if not match_obj:
            return False
        ctrl_id, file_time = match_obj.group(1), match_obj.group(2)
        self.un_tgz_file(self.file_name, self.tmp_log_path)
        for message_name in os.listdir(self.tmp_log_path):
            message_log_file = self.tmp_log_path + message_name
            self.record_file(ctrl_id, file_time, "messages",
                             message_log_file)
        return True


class CurMessagesProcess(BaseProcess):
    """
    对当前messages日志文件的处理
    """
    def __init__(self, *args):
        super(CurMessagesProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_messages$", self.file_name)
        if not match_obj:
            return False
        self.record_file(match_obj.group(1),
                         "zzzzzzzz",
                         "messages",
                         os.path.join(self.log_path, self.file_name))
        return True


class CurMceLogProcess(BaseProcess):
    """
    对当前mcelog日志文件的处理
    """
    def __init__(self, *args):
        super(CurMceLogProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_mcelog$", self.file_name)
        if not match_obj:
            return False

        self.record_file(match_obj.group(1),
                         "zzzzzzzz",
                         "mcelog",
                         os.path.join(self.log_path, self.file_name))
        return True


class CurRasDaemonProcess(BaseProcess):
    """
    对当前rasdaemon.log日志文件的处理
    """
    def __init__(self, *args):
        super(CurRasDaemonProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_rasdaemon\.log", self.file_name)
        if not match_obj:
            return False
        self.record_file(match_obj.group(1),
                         "zzzzzzzz",
                         "rasdaemon",
                         os.path.join(self.log_path, self.file_name))
        return True


class ImuProcess(BaseProcess):
    """
    对0A_imu_debug_log.tar.gz和0A_imu_debug_log_his.tar.gz日志文件的处理
    """
    def __init__(self, *args):
        super(ImuProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_imu_debug_log(_his)*", self.file_name)
        if not match_obj:
            return False
        ctrl_id = match_obj.group(1)
        file_tag = "his" if match_obj.group(2) else "cur"
        self.un_tgz_file(self.file_name, self.tmp_log_path,
                         ["var/log/pme/imu_debug_log"])
        message_log_file = self.tmp_log_path + r"var\log\pme\imu_debug_log"
        self.record_file(ctrl_id, file_tag, "imu", message_log_file)
        return True


class BiosUartPrintProcess(BaseProcess):
    """
    对0A_bios_uart_print.tar.gz日志文件的处理， Purely 2U设备无此日志文件
    """
    def __init__(self, *args):
        super(BiosUartPrintProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_bios_uart_print*", self.file_name)
        if not match_obj:
            return False
        ctrl_id = match_obj.group(1)
        self.un_tgz_file(self.file_name, self.tmp_log_path,
                         ["tmp/bios_uart_print"])
        bios_log_file = self.tmp_log_path + r"tmp\bios_uart_print"
        self.record_file(ctrl_id, "bios", "memory_info", bios_log_file)
        return True


class MemoryInfoProcess(BaseProcess):
    """
    对0A_memory_info日志文件的处理，针对purely 2U且安装了特定补丁的版本才有这个文件
    """
    def __init__(self, *args):
        super(MemoryInfoProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_memory_info*", self.file_name)
        if not match_obj:
            return False
        ctrl_id = match_obj.group(1)
        bios_log_file = os.path.join(self.log_path, self.file_name)
        self.record_file(ctrl_id, "patch", "memory_info", bios_log_file)
        return True


class CurSesLogProcess(BaseProcess):
    """
    对0A_SES_log.txt日志文件的处理
    """
    def __init__(self, *args):
        super(CurSesLogProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_ses_log\.txt*", self.file_name, re.IGNORECASE)
        if not match_obj:
            return False
        ctrl_id = match_obj.group(1)
        bios_log_file = os.path.join(self.log_path, self.file_name)
        self.record_file(ctrl_id, "zzzzzz", "ses_log", bios_log_file)
        return True


class HisSesLogProcess(BaseProcess):
    """
    对0A_SES_log_202103062562_0.tgz日志文件的处理
    """
    def __init__(self, *args):
        super(HisSesLogProcess, self).__init__(*args)

    def process(self):
        match_obj = re.match("([0-9][A-D])_ses_log_(\d+)", self.file_name, re.IGNORECASE)
        if not match_obj:
            return False
        ctrl_id, file_time = match_obj.group(1), match_obj.group(2)
        self.un_tgz_file(self.file_name, self.tmp_log_path)
        for ses_log_name in os.listdir(self.tmp_log_path):
            ses_log_file = self.tmp_log_path + ses_log_name
            self.record_file(ctrl_id, file_time, "ses_log",
                             ses_log_file)
        return True
