# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2012-. All rights reserved.

import codecs
import re


class MemoryInfoParser(object):
    def __init__(self, file_name):
        self.file_name = file_name

    def execute(self):
        """
        执行文件解析，得到所有内存条的基本信息（厂商、批次、型号）
        :return: [{vendor:xx, year:xx, week:xx, model:xx}, {}, ...]
        """
        all_memory_origin_info = self._split_memory_info()
        pattern = r"Manufacturer\s*:\s*([\w]+)[\s\S]*Asset Tag\s*:\s*(\d+)\s*Part Number\s*:\s*([\w-]+)\s*"
        all_memory_data = []
        for one_memory in all_memory_origin_info:
            match_obj = re.search(pattern, one_memory, re.IGNORECASE)
            if match_obj:
                year, week = self._get_year_and_week(match_obj.group(2))
                all_memory_data.append(dict(vendor=match_obj.group(1),
                                            year=year,
                                            week=week,
                                            model=match_obj.group(3)))
        return all_memory_data

    @staticmethod
    def _get_year_and_week(batch_no):
        """
        从批次信息中提取年份和周信息，年份默认以20开头
        :param batch_no: 批次信息，示例：1926-2019年第26周
        :return: year, week
        """
        year = "20" + batch_no[:2]
        week = batch_no[2:]
        return year, week

    def _split_memory_info(self):
        """
        将整个文件按关键字拆分成多个内存条的信息
        :return: [memory1， memory2, ...]
        """
        all_memory_infos = []
        with codecs.open(self.file_name, "r") as f_obj:
            start_flag = False
            for line in f_obj:
                if re.match("Memory Device", line):
                    start_flag = True
                    temp_data = []
                    continue
                if not start_flag:
                    continue
                if re.match(r"^\s*$", line):
                    all_memory_infos.append("\n".join(temp_data))
                    start_flag = False
                    continue
                temp_data.append(line)
        return all_memory_infos
