# -*- coding: UTF-8 -*-
from collections import namedtuple
import copy

import cliUtil
import common
from cbb.business.operate.expansion import common as exp_common
from cbb.business.operate.expansion import config
from cbb.frame.adapter.replace_adapter import compare_version
from cbb.frame.rest import commonRestUtil
from cbb.frame.util import common as cbbCommon
from cbb.frame.cli import cliUtil as cbbCliUtil

# 需屏蔽的错误码 - 新扩引擎 同时评估存储池和性能层
ENCLOSURE_REDUNDANT = "1"
# 硬盘框列表
FRAME_ID_LIST = "frameIdList"
# 硬盘数量列表
DISK_NUM_LIST = "diskNumList"
# 硬盘容量列表
DISK_CAP_LIST = "diskCapacityList"
EXP_POOL_AND_PERFORMANCE_ERR_CODE = "1073810131"

DiskCapacityInfo = namedtuple("DiskCapacityInfo",
                              ["param_0", "param_1", "param_2", "param_3",
                               "param_4", "param_5", "param_6", "param_7", "param_8"])


def execute(cli, lang, logger, py_java_env):
    """
    扩容评估硬盘容量规格检查
    @param cli: cli
    @param lang: lang
    @param logger: logger
    @param py_java_env: py_java_env
    @return:
    """

    cli_ret_all = []

    try:
        is_pass_though_mode_flag, res = cbbCommon.is_micro_pass_though_mode(cli, lang)
        cli_ret_all.append("System mode: {}".format(res))
        if is_pass_though_mode_flag:
            return cliUtil.RESULT_NOSUPPORT, "\n".join(cli_ret_all), ""

        exp_common.renew_expansion_config_data(logger)
        # 获取硬盘域信息
        flag, cli_ret, err_msg, disk_domain_list = \
            common.getDiskDomainInfo(cli, lang)
        cli_ret_all.append(cli_ret)
        if not flag:
            logger.logSysAbnormal()
            return cliUtil.RESULT_NOCHECK, "\n".join(cli_ret_all), err_msg
        logger.logInfo("diskDomainList is %s" % disk_domain_list)
        domain_eng_dict = get_domain_engine_info(cli, lang, disk_domain_list)
        # 深度拷贝 防止后续处理修改py_java_env中的原始数据
        exp_disk_list = copy.deepcopy(
            common.getExpDiskListFromContextFilter(py_java_env))
        exp_disk_list = get_real_capacity(exp_disk_list)
        if not exp_disk_list:
            return cliUtil.RESULT_NOSUPPORT, "", ""
        logger.logInfo("exp_disk_list:" + str(exp_disk_list))

        # 需要持久化的数据 和 扩容评估下发的参数
        # 按硬盘域循环调用扩容评估接口
        performance_layer_domain_dict = get_performance_layer_domain_info(cli, lang)
        execute_exp_info = ExecuteExpInfo(
            cli_ret_all, exp_disk_list, domain_eng_dict, py_java_env, logger, performance_layer_domain_dict)
        config_persist_data, err_msg, flag = execute_exp(execute_exp_info)
        # 检查通过，持久化配置数据
        if flag:
            # 扩容配置持久化数据
            logger.logInfo("check pass, start to persist config data: %s"
                           % config_persist_data)
            exp_common.save_expansion_config_data(config_persist_data, logger)
            exp_common.get_expansion_config_data(logger)
        return flag, "\n".join(cli_ret_all), err_msg

    except Exception as exception:
        logger.logException(exception)
        err_msg = common.getMsg(lang, "query.result.abnormal")
        return cliUtil.RESULT_NOCHECK, "".join(cli_ret_all), err_msg


def get_domain_engine_info(cli, lang, domain_list):
    domain_eng_dict = {}
    for domain in domain_list:
        domain_id = domain.get("ID")
        flag, ret, msg, domain_detail_dict = common.getDomainInfoById(
            cli, domain.get("ID"), lang
        )
        if flag is not True:
            continue
        ctrl_enc = domain_detail_dict.get("Controller enclosure")
        domain_eng_dict[domain_id] = ctrl_enc
    return domain_eng_dict


class ExecuteExpInfo:
    def __init__(self,
                 cli_ret_all, exp_disk_list, domain_eng_dict, py_java_env, logger, performance_layer_domain_dict):
        self.cli_ret_all = cli_ret_all
        self.exp_disk_list = exp_disk_list
        self.domain_eng_dict = domain_eng_dict
        self.py_java_env = py_java_env
        self.logger = logger
        self.performance_layer_domain_dict = performance_layer_domain_dict


def execute_exp(execute_exp_info):
    """
    执行评估
    :param execute_exp_info: 评估参数
    :return: 持久数据， 错误消息， 是否通过
    """
    flag = True
    err_msg_list = []
    pool_and_disk_domain_map = common.getpoolAndDiskDomainMap(execute_exp_info.py_java_env)
    (
        config_persist_data,
        exp_evaluate_param,
        exp_performance_layer_evaluate_param,
    ) = get_persist_data_and_eval_para(execute_exp_info.exp_disk_list,
                                       pool_and_disk_domain_map,
                                       execute_exp_info.py_java_env)
    execute_exp_info.logger.logInfo("exp performance layer evaluate param:{}, \nexp_evaluate_param:{}".format(
        exp_performance_layer_evaluate_param, exp_evaluate_param))

    # 框级冗余，同时扩性能层和存储池。
    is_enc_redundant = any(a_persist_data["isEnclosureRedundant"] == "1" for a_persist_data in config_persist_data)
    if is_enc_redundant and exp_evaluate_param and exp_performance_layer_evaluate_param:
        err_msg, flag = exp_enc_redundant_disk_domain_and_performance_together(
            err_msg_list, execute_exp_info, exp_evaluate_param, exp_performance_layer_evaluate_param)
        return config_persist_data, err_msg, flag

    if exp_performance_layer_evaluate_param:
        for performance_layer_id in exp_performance_layer_evaluate_param:
            evaluate_param = exp_performance_layer_evaluate_param.get(performance_layer_id)
            if any(a_persist_data["isEnclosureRedundant"] == "1" for a_persist_data in config_persist_data):
                evaluate_param["checkFrameLevelRule"] = "0"
            rest_flag, rest_msg, ret = \
                common.exp_performance_layer_evaluate_by_rest(
                    execute_exp_info.py_java_env, evaluate_param, execute_exp_info.logger)
            execute_exp_info.cli_ret_all.append(ret)
            if not rest_flag:
                flag = False
                err_msg_list += rest_msg
    for disk_domain_id in exp_evaluate_param:
        evaluate_param = exp_evaluate_param.get(disk_domain_id)
        shield_err_code_list = get_shield_err_code_list(
            evaluate_param, exp_performance_layer_evaluate_param, execute_exp_info.domain_eng_dict,
            execute_exp_info.performance_layer_domain_dict
        )
        rest_flag, rest_msg, ret = common.exp_evaluate_by_rest(
            execute_exp_info.py_java_env, evaluate_param, execute_exp_info.logger, shield_err_code_list
        )
        execute_exp_info.cli_ret_all.append(ret)
        if rest_flag is False:
            flag = False
            err_msg_list += rest_msg
    err_msg = "".join(err_msg_list)
    return config_persist_data, err_msg, flag


def exp_enc_redundant_disk_domain_and_performance_together(
        err_msg_list, execute_exp_info, exp_evaluate_param, exp_performance_layer_evaluate_param):
    flag = True
    exp_param = exp_evaluate_param.values()[0]
    exp_performance_layer_param = exp_performance_layer_evaluate_param.values()[0]
    exp_param["performanceLayerDiskType"] = exp_performance_layer_param.get("TIER0DISKTYPE")
    exp_param["performanceLayerDiskCapList"] = exp_performance_layer_param.get(DISK_CAP_LIST)
    exp_param["performanceLayerDiskNumList"] = exp_performance_layer_param.get(DISK_NUM_LIST)
    exp_param["performanceLayerId"] = exp_performance_layer_param.get("id")
    exp_param["performanceLayerFrameId"] = exp_performance_layer_param.get(FRAME_ID_LIST)
    exp_param["performanceLayerEngineIdList"] = exp_performance_layer_param.get("engineIdList")
    shield_err_code_list = get_shield_err_code_list(
        exp_param, exp_performance_layer_evaluate_param, execute_exp_info.domain_eng_dict,
        execute_exp_info.performance_layer_domain_dict
    )
    rest_flag, rest_msg, ret = common.exp_evaluate_by_rest(
        execute_exp_info.py_java_env, exp_param, execute_exp_info.logger, shield_err_code_list
    )
    execute_exp_info.cli_ret_all.append(ret)
    if rest_flag is False:
        flag = False
        err_msg_list += rest_msg
    for performance_layer_id in exp_performance_layer_evaluate_param:
        evaluate_param = exp_performance_layer_evaluate_param.get(performance_layer_id)
        evaluate_param["checkFrameLevelRule"] = "1"
        rest_flag, rest_msg, ret = \
            common.exp_performance_layer_evaluate_by_rest(
                execute_exp_info.py_java_env, evaluate_param, execute_exp_info.logger)
        execute_exp_info.cli_ret_all.append(ret)
        if not rest_flag:
            flag = False
            err_msg_list += rest_msg
    err_msg = "".join(err_msg_list)
    return err_msg, flag


def get_shield_err_code_list(pool_param, performance_param, domain_eng_dict, performance_layer_domain_dict):
    """
    设置屏蔽的错误码。
    :param pool_param: 存储池评估参数
    :param performance_param: 性能层评估参数
    :param domain_eng_dict: 硬盘域和引擎
    :param performance_layer_domain_dict 性能层关联的硬盘域
    :return: shield_err_code_list
    """
    shield_err_code_list = []
    if is_exp_pool_and_performance_layer_on_new_engine(
            pool_param, performance_param, domain_eng_dict, performance_layer_domain_dict):
        shield_err_code_list.append(EXP_POOL_AND_PERFORMANCE_ERR_CODE)
    return shield_err_code_list


def is_exp_pool_and_performance_layer_on_new_engine(
        pool_param, performance_param, domain_eng_dict, performance_layer_domain_dict
):
    """
    新扩引擎 同时评估存储池和性能层
    更新屏蔽场景：新扩性能层包含存储池的引擎场景，而不止完全相同
    :param pool_param: 存储池评估参数
    :param performance_param: 性能层评估参数
    :param domain_eng_dict: 硬盘域和引擎
    :param performance_layer_domain_dict: 性能层关联的硬盘域
    :return: 是否需要屏蔽错误码
    """
    for performance_layer_id in performance_param:
        disk_domain_id = pool_param.get("diskDomainId")
        associated_disk_domain_list = performance_layer_domain_dict.get(performance_layer_id, [])
        if disk_domain_id not in associated_disk_domain_list:
            continue
        diff_eng = [
            eng
            for eng in pool_param.get("engineIdList", [])
            if is_exp_on_new_engine(domain_eng_dict, eng, performance_param.get(performance_layer_id), pool_param)
        ]
        return (
                pool_param.get("engineIdList") == performance_param.get(performance_layer_id).get("engineIdList")
                or not diff_eng
        )
    return False


def get_performance_layer_domain_info(cli, lang):
    """
    获取性能层关联的硬盘域ID
    :param cli: ssh连接
    :param lang: 语言
    :return: 性能层关联的硬盘域
    """
    cmd = "show performance_layer general"
    flag, ret, _ = cbbCliUtil.execCmdInCliMode(cli, cmd, True, lang)
    if not flag:
        return {}

    performance_layer_cmd_list = []
    performance_layer_detail_cmd = 'show performance_layer general performance_layer_id={}'
    ret_list = cbbCliUtil.getHorizontalCliRet(ret)
    [performance_layer_cmd_list.append(performance_layer_detail_cmd.format(line.get("ID")))
     for line in ret_list]

    performance_layer_domain_dict = {}
    for cmd in performance_layer_cmd_list:
        flag, ret, _ = cbbCliUtil.execCmdInCliMode(cli, cmd, True, lang)
        if not flag:
            continue
        ret_list = cbbCliUtil.getVerticalCliRet(ret)
        for line in ret_list:
            performance_layer_domain_dict[line.get("ID")] = line.get(
                "Associated Disk Domain Id List", "").split(",")
    return performance_layer_domain_dict


def is_exp_on_new_engine(domain_eng_dict, eng, performance_param, pool_param):
    return (eng not in performance_param.get("engineIdList", [])
            and eng not in domain_eng_dict.get(pool_param.get("diskDomainId"), []))


def get_persist_data_and_eval_para(exp_disk_list, pool_and_disk_domain_map, py_java_env):
    """获取持久化数据和扩容评估参数

    :param exp_disk_list: 扩容配置数据
    :param pool_and_disk_domain_map: 存储池和硬盘映射关系
    :param py_java_env: py_java_env
    :return:
    """
    config_persist_data = []
    exp_evaluate_param = {}
    exp_performance_layer_evaluate_param = {}
    capacity_number_dict = {}
    for line in exp_disk_list:
        (disk_domain_id, disk_num, engine_id, performance_layer_model, pool_id) = get_exp_disk_domain_info(
            line, pool_and_disk_domain_map)
        (capacity_unit, disk_capacity, disk_enclosure_id, disk_model, disk_work_mode,
         disk_bom) = get_exp_disk_detail_info(line)
        disk_capacity_2_gb, disk_enclosure_name, need_enc_redundancy_check = get_exp_disk_capacity_detail_info(
            capacity_unit, disk_capacity, line)
        a_persist_data = save_persist_data_pool_info(
            disk_domain_id, disk_num, performance_layer_model, pool_id)
        save_persist_data_disk_info(a_persist_data, disk_capacity_2_gb,
                                    disk_model, disk_work_mode, engine_id)
        # 添加硬盘BOM编码
        a_persist_data["diskBom"] = disk_bom
        save_persist_data_enc_info(
            a_persist_data, config_persist_data, disk_enclosure_id, disk_enclosure_name, line)

        # 如果是性能层的盘，则不设置存储池评估参数
        if str(performance_layer_model) == "1":
            tmp_performance_param = dict()
            tmp_performance_param["disk_capacity_2_gb"] = disk_capacity_2_gb
            tmp_performance_param["disk_num"] = disk_num
            tmp_performance_param["engine_id"] = engine_id
            tmp_performance_param["pool_id"] = pool_id
            tmp_performance_param["disk_model"] = disk_model
            tmp_performance_param["disk_bom"] = disk_bom
            tmp_performance_param["diskEnclosureId"] = disk_enclosure_id
            tmp_performance_param["is_enc_redundant"] = \
                a_persist_data.get("isEnclosureRedundant") == ENCLOSURE_REDUNDANT
            a_performance_evaluate_param = exp_performance_layer_evaluate_param.get(pool_id, {})
            update_a_performance_layer_capacity_param(
                capacity_number_dict, a_performance_evaluate_param, tmp_performance_param, py_java_env)
            if a_persist_data.get("isEnclosureRedundant") == ENCLOSURE_REDUNDANT:
                a_performance_evaluate_param["is_enc_redundant"] = True
            exp_performance_layer_evaluate_param[pool_id] = a_performance_evaluate_param
            continue

        # 扩容评估下发的参数
        a_evaluate_param = init_evaluate_param(disk_domain_id, disk_work_mode, exp_evaluate_param, pool_id)

        disk_capacity_list_key, disk_num_list_key, disk_type_key = get_disk_tie_type(disk_model)

        disk_cap_info = DiskCapacityInfo(disk_capacity_2_gb, disk_capacity_list_key,
                                         disk_enclosure_id, disk_model, disk_num, disk_num_list_key,
                                         disk_type_key, need_enc_redundancy_check, disk_bom)
        update_evaluate_param_disk_number(
            a_evaluate_param, disk_cap_info)
        if engine_id not in a_evaluate_param.get("engineIdList"):
            a_evaluate_param.get("engineIdList").append(str(engine_id))
        exp_evaluate_param[disk_domain_id] = a_evaluate_param
    # 将硬盘数量从int转换为str
    update_disk_num(exp_evaluate_param)
    set_exp_performance_layer_eval_param(capacity_number_dict, exp_performance_layer_evaluate_param)
    return config_persist_data, exp_evaluate_param, exp_performance_layer_evaluate_param


def get_exp_disk_capacity_detail_info(capacity_unit, disk_capacity, line):
    need_enc_redundancy_check = str(
        line.get("needCheckEnclosureRedundant")) == "1"
    disk_capacity_2_gb = \
        common.changUnit2GBLabelCap(disk_capacity + capacity_unit)[1]
    disk_capacity_2_gb = str(int(disk_capacity_2_gb))
    disk_enclosure_name = str(line.get("diskEnclosureName"))
    return disk_capacity_2_gb, disk_enclosure_name, need_enc_redundancy_check


def get_exp_disk_detail_info(line):
    disk_capacity = str(line.get("diskCapacity"))
    capacity_unit = str(line.get("unit"))
    disk_model = str(line.get("diskModel"))
    disk_work_mode = line.get("diskWorkMode")
    disk_enclosure_id = str(line.get("diskEnclosureId"))
    disk_bom = str(line.get("diskBom"))
    data_param = (capacity_unit, disk_capacity, disk_enclosure_id, disk_model, disk_work_mode, disk_bom)
    return data_param


def update_a_performance_layer_capacity_param(
        capacity_number_dict, a_performance_evaluate_param, tmp_performance_param, py_java_env
):
    disk_capacity_2_gb = tmp_performance_param.get("disk_capacity_2_gb")
    disk_num = tmp_performance_param.get("disk_num")
    engine_id = tmp_performance_param.get("engine_id")
    pool_id = tmp_performance_param.get("pool_id")
    disk_model = tmp_performance_param.get("disk_model")
    disk_bom = tmp_performance_param.get("disk_bom")
    disk_enclosure_id = tmp_performance_param.get("diskEnclosureId")

    tmp_dict = capacity_number_dict.get(pool_id, {})
    # 框级冗余
    if tmp_performance_param.get("is_enc_redundant"):
        enc_id_cap = disk_enclosure_id + "_" + disk_capacity_2_gb
        if not tmp_dict.get(enc_id_cap):
            tmp_dict[enc_id_cap] = [disk_capacity_2_gb, int(disk_num)]
        else:
            tmp_dict.get(enc_id_cap)[1] += int(disk_num)
    else:
        tmp_dict[disk_capacity_2_gb] = tmp_dict.get(disk_capacity_2_gb, 0) + int(disk_num)
    capacity_number_dict[pool_id] = tmp_dict
    a_performance_evaluate_param["engineIdList"] = list(set(
        a_performance_evaluate_param.get("engineIdList", []) + [
            engine_id]
    ))
    # 615 及之后的需要传入disk type
    if is_615_or_later(py_java_env):
        a_performance_evaluate_param["TIER0DISKTYPE"] = commonRestUtil.get_disk_type_enum(disk_model, disk_bom)


def is_615_or_later(py_java_env):
    dev_version = str(py_java_env.get("devInfo").getProductVersion())
    return compare_version(dev_version, '6.1.5RC1') >= 0


def get_exp_disk_domain_info(line, pool_and_disk_domain_map):
    performance_layer_model = line.get("performanceLayerModel")
    pool_id = line.get("diskDomain")
    disk_domain_id = pool_and_disk_domain_map.get(pool_id)
    engine_id = str(line.get("eng"))
    disk_num = int(line.get("diskNum")) if str(line.get("diskNum")) != "0" else int(common.EMPTY_POOL_FLAG)
    param_tuple = (disk_domain_id, disk_num, engine_id, performance_layer_model, pool_id)
    return param_tuple


def update_evaluate_param_disk_number(a_evaluate_param, disk_cap_info):
    disk_capacity_2_gb, disk_capacity_list_key, disk_enclosure_id, disk_model, \
        disk_num, disk_num_list_key, disk_type_key, need_enc_redundancy_check, disk_bom = disk_cap_info
    if not need_enc_redundancy_check:
        if disk_capacity_2_gb not in a_evaluate_param.get(
                disk_capacity_list_key):
            a_evaluate_param.get(disk_num_list_key).append(disk_num)
            a_evaluate_param.get(disk_capacity_list_key).append(
                disk_capacity_2_gb)
            a_evaluate_param.get(disk_type_key).append(
                commonRestUtil.get_disk_type_enum(disk_model, disk_bom))
        else:
            # 相同容量点的硬盘数量需要相加
            i = a_evaluate_param.get(disk_capacity_list_key).index(
                disk_capacity_2_gb)
            a_evaluate_param.get(disk_num_list_key)[i] += disk_num
    else:
        info_need_add = True
        for i, _ in enumerate(a_evaluate_param.get(disk_capacity_list_key)):
            if a_evaluate_param[disk_capacity_list_key][i] == \
                    disk_capacity_2_gb and \
                    a_evaluate_param["diskEnclosureId"][i] == \
                    disk_enclosure_id:
                a_evaluate_param[disk_num_list_key][i] += disk_num
                info_need_add = False
                break
        if info_need_add:
            a_evaluate_param.get(disk_num_list_key).append(disk_num)
            a_evaluate_param.get(disk_capacity_list_key).append(
                disk_capacity_2_gb)
            a_evaluate_param.get("diskEnclosureId").append(
                disk_enclosure_id)
            a_evaluate_param.get(disk_type_key).append(
                commonRestUtil.get_disk_type_enum(disk_model, disk_bom))


def update_disk_num(exp_evaluate_param):
    for key in exp_evaluate_param:
        item = exp_evaluate_param.get(key)
        # ssd类型硬盘数量
        ssd_int_list = item.get("diskNumList")
        item["diskNumList"] = list(map(str, ssd_int_list))
        # ssd类型硬盘数量
        ssd_int_list = item.get("nlsasDiskNumList")
        item["nlsasDiskNumList"] = list(map(str, ssd_int_list))


def get_disk_tie_type(disk_model):
    if disk_model in config.HDD_DISK_TYPE:
        disk_num_list_key = "nlsasDiskNumList"
        disk_capacity_list_key = "nlsasdiskCapacityList"
        disk_type_key = "tier2DiskType"
    else:
        disk_num_list_key = "diskNumList"
        disk_capacity_list_key = "diskCapacityList"
        disk_type_key = "tier0DiskType"
    return disk_capacity_list_key, disk_num_list_key, disk_type_key


def save_persist_data_enc_info(a_persist_data, config_persist_data,
                               disk_enclosure_id, disk_enclosure_name, line):
    a_persist_data["diskEnclosureId"] = disk_enclosure_id
    a_persist_data["isEnclosureRedundant"] = str(
        line.get("needCheckEnclosureRedundant"))
    a_persist_data["diskEnclosureName"] = disk_enclosure_name
    config_persist_data.append(a_persist_data)


def save_persist_data_disk_info(a_persist_data, disk_capacity_2_gb, disk_model,
                                disk_work_mode, engine_id):
    a_persist_data["diskCapacity"] = disk_capacity_2_gb
    a_persist_data["engine"] = engine_id
    a_persist_data["diskModel"] = disk_model
    a_persist_data["diskWorkMode"] = disk_work_mode


def save_persist_data_pool_info(disk_domain_id, disk_num,
                                performance_layer_model, pool_id):
    # 需要持久化的数据
    a_persist_data = {}
    a_persist_data["performanceLayerModel"] = performance_layer_model
    a_persist_data["poolId"] = pool_id
    a_persist_data["domainId"] = disk_domain_id
    a_persist_data["diskNum"] = disk_num
    return a_persist_data


def init_evaluate_param(disk_domain_id, disk_work_mode, exp_evaluate_param,
                        pool_id):
    a_evaluate_param = exp_evaluate_param.get(disk_domain_id, None)
    if a_evaluate_param is None:
        a_evaluate_param = {}
        a_evaluate_param["poolId"] = pool_id
        a_evaluate_param["diskDomainId"] = disk_domain_id
        # SSD盘个数列表
        a_evaluate_param["diskNumList"] = []
        # SSD盘容量大小列表
        a_evaluate_param["diskCapacityList"] = []
        # HDD类型盘个数列表
        a_evaluate_param["nlsasDiskNumList"] = []
        # HDD类型盘容量大小列表
        a_evaluate_param["nlsasdiskCapacityList"] = []
        # SSD盘类型
        a_evaluate_param["tier0DiskType"] = []
        # HDD盘类型
        a_evaluate_param["tier2DiskType"] = []
        # 引擎ID列表
        a_evaluate_param["engineIdList"] = []
        a_evaluate_param["diskEnclosureId"] = []
        # 盘的工作模式
        a_evaluate_param["diskWorkMode"] = disk_work_mode
    return a_evaluate_param


def set_exp_performance_layer_eval_param(disk_cap_num_dict, eval_param):
    """
    设置性能层容量点和数量
    :param disk_cap_num_dict:
    :param eval_param:
    :return:
    """
    for domain_id, tmp_dict in disk_cap_num_dict.items():
        eval_param.get(domain_id)["id"] = domain_id
        # 框级冗余
        if eval_param.get(domain_id).get("is_enc_redundant"):
            for end_id_cap, cap_num in tmp_dict.items():
                frame_ids = eval_param.get(domain_id).get(FRAME_ID_LIST, [])
                frame_ids.append(end_id_cap.split("_")[0])
                eval_param.get(domain_id)[FRAME_ID_LIST] = frame_ids
                disk_num = eval_param.get(domain_id).get(DISK_NUM_LIST, [])
                disk_num.append(str(cap_num[1]))
                eval_param.get(domain_id)[DISK_NUM_LIST] = disk_num
                disk_cap = eval_param.get(domain_id).get(DISK_CAP_LIST, [])
                disk_cap.append(cap_num[0])
                eval_param.get(domain_id)[DISK_CAP_LIST] = disk_cap
            continue
        for cap, disk_num in tmp_dict.items():
            eval_param.get(domain_id)[DISK_NUM_LIST] = \
                eval_param.get(domain_id).get(DISK_NUM_LIST, []) + [str(disk_num)]
            eval_param.get(domain_id)[DISK_CAP_LIST] = \
                eval_param.get(domain_id).get(DISK_CAP_LIST, []) + [str(cap)]


def get_real_capacity(cfg_data):
    """将扩容配置中的电子标签容量折算为实际容量

    :param cfg_data: 扩容配置数据
    :return cfg_data: 折算后的配置数据
    """
    # 1GB电子标签容量折算为实际容量的系数, 算法为:
    # 1GB电子标签容量 = 1000*1000*1000B
    # 实际容量 = 1000*1000*1000/1024/1024/1024 = 0.9313225746154785 GB
    to_real_capacity = 0.9313225746154785
    for a_data in cfg_data:
        e_label_capacity = float(a_data.get("diskCapacity", "0"))
        real_capacity = e_label_capacity * to_real_capacity
        a_data["diskCapacity"] = real_capacity

    return cfg_data
