# -*- coding: UTF-8 -*-
import re
import common
import cliUtil
from risk_version_config import HIGH_IDLE_DISK_NUM_VERSION
from common_utils import get_err_msg
from cbb.frame.cli import cliUtil as cbb_cliUtil

PY_JAVA_ENV = py_java_env


def execute(cli):
    """
    检查高空闲率盘数
    :param cli: cli连接
    :return:
    """
    check_item = CheckHighIdleDiskNum(cli)
    flag = check_item.check()
    return flag, "\n".join(check_item.cli_ret_list), \
        check_item.err_msg


class CheckHighIdleDiskNum:
    def __init__(self, cli):
        self.cli = cli
        self.lang = common.getLang(PY_JAVA_ENV)
        self.logger = common.getLogger(PY_LOGGER, __file__)
        self.software_version = ""
        self.hot_patch_version = ""
        self.cli_ret_list = []
        self.err_msg = ""

    def check(self):
        try:
            # 检查是否为风险版本和补丁
            if not self.is_risk_version_and_patch():
                return True

            # 检查BOM编码
            return self.check_high_idle_disk()

        except common.UnCheckException as e:
            self.logger.logError(str(e))
            self.err_msg = e.errorMsg
            return cliUtil.RESULT_NOCHECK
        except Exception as e:
            self.logger.logError(str(e))
            self.err_msg = common.getMsg(self.lang, "query.result.abnormal")
            return cliUtil.RESULT_NOCHECK

    def is_risk_version_and_patch(self):
        """
        判断是否为风险版本
        :return:
        """
        flag, self.software_version, self.hot_patch_version, cli_ret = \
            cbb_cliUtil.get_system_version_with_ret(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            err_msg = common.getMsg(self.lang,
                                    "cannot.get.product.version.info")
            raise common.UnCheckException(err_msg, "")

        self.logger.logInfo("version:{}, patch:{}".format(
            self.software_version, self.hot_patch_version))
        # 不是风险版本，检查通过
        if self.software_version not in HIGH_IDLE_DISK_NUM_VERSION.keys():
            return False
        # 风险版本，不存在补丁,需要继续检查
        if not self.hot_patch_version:
            self.hot_patch_version = "--"
            return True

        patch_num = self.get_patch_num(self.hot_patch_version)
        if not patch_num:
            return True

        require_patch = HIGH_IDLE_DISK_NUM_VERSION.get(self.software_version)
        if not require_patch:
            return True
        require_patch_num = self.get_patch_num(require_patch)
        return patch_num < require_patch_num

    @staticmethod
    def get_patch_num(patch_version):
        """
        获取补丁版本的数字
        :return:
        """
        patch = re.search(r"SPH(\d+)", patch_version)
        if not patch:
            return 0
        return int(patch.group(1))

    def check_high_idle_disk(self):
        """
        检查高空闲率的盘数
        :return:
        """
        domain_ids = self.get_domain_id()
        self.logger.logInfo("domain_ids:{}".format(domain_ids))
        if not domain_ids:
            return True
        for domain_id in domain_ids:
            if self.has_too_many_high_idle_disk(domain_id):
                self.set_srr_msg()
                return False

        return True

    def get_domain_id(self):
        """
        获取硬盘域ID
        :return:
        """
        disk_domain_id_list = []
        cmd = "show disk_domain general |filterColumn include columnList=ID"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            err_msg = common.getMsg(self.lang,
                                    "cannot.get.disk.domain.info")
            raise common.UnCheckException(err_msg, "")
        if cliUtil.queryResultWithNoRecord(cli_ret):
            return disk_domain_id_list
        disk_domain_info_list = cliUtil.getHorizontalCliRet(cli_ret)

        for disk_domain_info in disk_domain_info_list:
            disk_domain_id = disk_domain_info.get("ID")
            disk_domain_id_list.append(disk_domain_id)
        return disk_domain_id_list

    def has_too_many_high_idle_disk(self, domain_id):
        """
        判断高空闲率的盘数量是否过多
        :return:
        """
        ctrl_id_list = self.get_ctrl_id_list()
        self.logger.logInfo("ctrl_id_list:{}".format(ctrl_id_list))

        for ctrl_id in ctrl_id_list:
            cmd = "show block_manager bitmap disk_pool_id={} space_type=data " \
                  "controller={}".format(domain_id, ctrl_id)
            flag, cli_ret, err_msg = cliUtil.excuteCmdInDeveloperMode(
                self.cli, cmd, True, self.lang)
            self.cli_ret_list.append(cli_ret)
            if flag is not True:
                self.logger.logInfo("The current controller does not "
                                    "return correct echo:{}".format(ctrl_id))
                continue
            dict_list = cliUtil.getHorizontalCliRet(cli_ret)
            self.logger.logInfo("dict_list:{}".format(dict_list))
            # 如果总盘数小于25，则高空闲率硬盘数不会过多，返回False
            if len(dict_list) <= 25:
                return False

            high_idle_disk_num = self.get_high_idle_disk_num(dict_list)
            if high_idle_disk_num >= 25:
                return True
            return False

        # 所有控制器都不支持该命令
        err_msg = common.getMsg(self.lang,
                                "cannot.get.diskindomain.info")
        raise common.UnCheckException(err_msg, "")

    def get_high_idle_disk_num(self, dict_list):
        """
        获取高空闲率盘数量
        :return:
        """
        # 获取所有的空闲率
        free_percent_list = []

        for dict_info in dict_list:
            free_percent = dict_info.get("Free Percent(%)", "").strip()

            if not free_percent or free_percent == '--':
                err_msg = common.getMsg(self.lang,
                                        "cannot.get.diskindomain.info")
                raise common.UnCheckException(err_msg, "")

            free_percent_list.append(float(free_percent))
        self.logger.logInfo("free_percent_list:{}".format(free_percent_list))

        # 最小空闲率
        min_free_percent = min(free_percent_list)
        # 获取高空闲率盘数
        high_idle_disk_num = 0
        for free_percent in free_percent_list:
            if free_percent - min_free_percent >= 7.0:
                high_idle_disk_num += 1
        self.logger.logInfo("high_idle_disk_num:{}".format(high_idle_disk_num))

        return high_idle_disk_num

    def get_ctrl_id_list(self):
        """
        获取控制器ID列表
        :return:
        """
        flag, controller_id_list, err_msg, cli_ret = \
            cliUtil.getControllerIdListWithRet(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            err_msg = common.getMsg(self.lang,
                                    "cannot.get.controller.info")
            raise common.UnCheckException(err_msg, "")

        return controller_id_list

    def set_srr_msg(self):
        """
        设置错误信息
        """
        if self.software_version == "6.0.0":
            self.err_msg = get_err_msg(
                self.lang, "storagepool.param.check.not.pass.upgrade.version",
                (self.software_version, "6.1.2"))
        else:
            self.err_msg = get_err_msg(
                self.lang, "software.io.delay.risk.patch.not.pass",
                (self.software_version, self.hot_patch_version,
                 HIGH_IDLE_DISK_NUM_VERSION.get(self.software_version)))
