# coding=UTF-8

import java.lang.Exception as JException

from cbb.frame.base.exception import CmdExecuteException
from cbb.frame.rest.restUtil import CommonRestService
from cbb.frame.rest.restDataConstants import DISK_TYPE_MAP
from cbb.frame.base.baseUtil import getPyResource

import cliUtil
import expandconfig
import common
from frameone.util import contextUtil


PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """
    RAID10存储层扩容检查

    检查方法：
    步骤1、以admin用户登录设备。
    步骤2、执行如下命令：show upgrade package，查询软件版本。
    步骤3、通过rest接口：https://${ip}:${port}/deviceManager/rest/${deviceId}/storagepool 批量查询存储池。  # noqa
    步骤4、查看存储池是否存在RAID10存储层（TIER*RAIDLV == 1）。
    步骤5、获取RAID10存储层的成员盘数（TIER*RAIDDISKNUM）。
    步骤6、获取RAID10存储层的对应的硬盘类型（TIER*DISKTYPE）。
    步骤7、获取硬盘容量截断检查结果。

    检查标准：
    1.如果步骤2软件版本不在范围内（V300R003C20SPC200到V300R006C60SPC100所有V3融合版本，V500R007C00到V500R007C60SPC300的所有V5融合版本），则检查通过，否则继续检查。      # noqa
    2.如果步骤3不存在存储池，则检查通过，否则继续检查。
    3.如果步骤4所有存储池都不存在RAID10存储层，则检查通过，否则继续检查。
    4.如果步骤5存储池所有RAID10存储层的成员盘都等于8，检查通过，否则继续检查。
    5.如果步骤6RAID10存储层对应的硬盘类型与扩入硬盘的类型都不相同，则检查通过，否则继续检查。       # noqa
    6.如果步骤7硬盘容量截断检查通过，则检查通过，否则不通过。

    :return: (检查结果，CLI回显，错误提示信息）
    """
    check_ret = StorageLayerCheck(cli, context=PY_JAVA_ENV).run()
    LOGGER.logInfo("Expand storagepool check raid10 tier finished.")
    return check_ret


class StorageLayerCheck(object):

    flag = True
    all_cli_ret_list = []
    err_msg = ""

    all_exp_disk_types = None

    def __init__(self, cli, context, logger=LOGGER, lang=LANG):
        self.cli = cli
        self.logger = logger
        self.lang = lang
        self.context = context

        self.exp_disk_list = expandconfig.ExpandConfig(
            self.context).getAllExpDiskList()
        self.get_all_exp_disk_types()

    def get_expansion_domain_ids(self):
        exp_disk_list = common.getExpDiskListFromContextFilter(PY_JAVA_ENV)
        return set(map(lambda x: x.get("diskDomain"), exp_disk_list))

    def get_all_cli_ret(self):
        return "\n".join(self.all_cli_ret_list)

    def get_all_exp_disk_types(self):
        self.all_exp_disk_types = set(
            map(lambda x: x.getDiskType(), self.exp_disk_list)
        )
        self.logger.logInfo("all_exp_disk_types:{}, exp_disk_list:{}".format(
            self.all_exp_disk_types, self.exp_disk_list))

    def check_version(self):
        """
        从V3R3C20SPC200到V500R007C60SPC300的所有V3/V5版本，需要继续检查
        """
        flag, ver, _, cli_ret, err = common.get_soft_and_patch_version(
            self.cli, self.logger, self.lang)
        self.all_cli_ret_list.append(cli_ret)
        if flag is not True:
            self.flag, self.err_msg = flag, err
            raise CmdExecuteException

        is_passed_ver = True
        if ("V300R003C20SPC200" <= ver <= "V300R006C60SPC100") or \
                ("V500R007C00" <= ver <= "V500R007C60SPC300"):
            is_passed_ver = False
        self.logger.logInfo("is_passed_ver:{}, ver:{}".format(
            is_passed_ver, ver))
        if is_passed_ver:
            self.flag, self.err_msg = True, ""
            raise CmdExecuteException

    def get_storage_pool_records(self):
        """
        获取存储池数据
        :return:存储池数据
        """
        context = contextUtil.getContext(self.context)
        rest = contextUtil.getRest(context)
        perf_url = "storagepool"

        self.all_cli_ret_list.append(rest.getBaseUri() + perf_url)

        storage_pool_records = CommonRestService.get4Big(
            rest,
            perf_url
        )
        self.all_cli_ret_list.append(str(storage_pool_records))
        exp_domain_ids = self.get_expansion_domain_ids()
        self.logger.logInfo("exp_domain_ids:{}".format(exp_domain_ids))
        return list(filter(
            lambda x: x.get("PARENTID") in exp_domain_ids,
            storage_pool_records
        ))

    def get_tier_info(self, num, info):
        return {
            "layer_id": num,
            "raid_lv": info.get("TIER{}RAIDLV".format(num)),
            "disk_type": DISK_TYPE_MAP.get(
                info.get("TIER{}DISKTYPE".format(num)), "-1"),
            "disk_num": info.get("TIER{}RAIDDISKNUM".format(num)),
            "name": getPyResource(self.lang, "tier.{}".format(num)),

        }

    def _is_raid10(self, tier_info):
        """是否为raid10"""
        return "1" == tier_info["raid_lv"]

    def _is_disk_num_equals_8(self, tier_info):
        """成员盘数是否为8"""
        return "8" == tier_info["disk_num"]

    def storage_pool_records_check(self, storage_pool_records):
        risk_pool_id_info_dict = {}
        for info in storage_pool_records:
            pool_id = info.get("ID")
            for i in range(3):
                tier_info = self.get_tier_info(i, info)
                self.logger.logInfo("pool_id:{}, tier_info:{}".format(
                    pool_id, tier_info))
                if not self._is_raid10(tier_info):
                    continue
                if not self._is_disk_num_equals_8(tier_info):
                    if tier_info["disk_type"] in self.all_exp_disk_types:
                        risk_pool_id_info_dict.setdefault(pool_id, [])
                        risk_pool_id_info_dict[pool_id].append(tier_info)

        self.logger.logInfo(
            "risk_pool_id_info_dict:{}".format(risk_pool_id_info_dict)
        )
        if not risk_pool_id_info_dict:
            self.flag, self.err_msg = True, ""
            raise CmdExecuteException

        return risk_pool_id_info_dict

    def _run_check(self):
        # 版本范围检查
        self.check_version()

        # 获取存储池信息
        storage_pool_records = self.get_storage_pool_records()

        # 存储池信息检查
        risk_pool_id_info_dict = self.storage_pool_records_check(
            storage_pool_records
        )

        # 获取容量截断检查结果, check_item_hardware_disk_capacity_cut
        flag, cli_ret, err_msg = self.context.get(
            "check_item_hardware_disk_capacity_cut.result",
            (False, "", "未获取检查结果")
        )
        self.logger.logInfo(
            "容量截断检查返回 flag:{}, cli_ret:{}, err_msg:{}".format(
                flag, cli_ret, err_msg)
        )

        self.all_cli_ret_list.append(cli_ret)
        if flag is not True:
            return (
                flag,
                self.get_all_cli_ret(),
                self.get_err_msg(risk_pool_id_info_dict)
            )
        return flag, self.get_all_cli_ret(), self.err_msg

    def get_err_msg(self, risk_pool_id_info_dict):
        lines = []
        for pool_id, tier_list in risk_pool_id_info_dict.items():
            lines.append(
                common.getMsg(
                    self.lang,
                    "dm_expand_storagepool_check_raid10_tier.error",
                    (pool_id, ", ".join(map(lambda x: x["name"], tier_list)))
                )
            )
        return "".join(lines)

    def run(self):
        try:
            return self._run_check()
        except CmdExecuteException:
            return self.flag, self.get_all_cli_ret(), self.err_msg
        except (Exception, JException) as e:
            self.logger.logException(e)
            return cliUtil.RESULT_NOCHECK, self.get_all_cli_ret(), ""
        finally:
            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)
            self.logger.logInfo(
                "Storage Layer Check end."
                "Enter cli mode from some model, ret: %s" % str(ret)
            )
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, self.logger)
