# -*- coding: UTF-8 -*-
import common
import cliUtil
import re

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """
    @summary: 【201806197733】SmartIO卡使用低64K地址DMA导致数据不一致。版本问题触发，接口卡，扣卡，电口卡在特定的版本下数据损坏的风险（概率性出现）。
    """
    allCliRet = ""
    
    try:
        # 检查系统是否为风险版本。
        flag, softwareVersion, cliRet = isRiskVersion(cli)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if flag == False:
            return (True, allCliRet, "")
        
        # 获取系统上存在风险的接口卡
        riskCardList, cliRet = getRiskCard(cli)
        allCliRet = common.joinLines(allCliRet, cliRet)
        if riskCardList:
            errCode = "exit.risk.interface.modul"
            errMsg = getErrMsg(errCode, (softwareVersion, "\n".join(riskCardList)))
            return (False, allCliRet, errMsg)
        
        return (True, allCliRet, "")

    except common.UnCheckException, unCheckException:
        logException()
        errMsg = unCheckException.errorMsg
        allCliRet = common.joinLines(allCliRet, unCheckException.cliRet)
        return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
    
    except:
        logException()
        errMsg = getErrMsg("cannot.get.system.info")
        return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)


def isRiskVersion(cli):
    """
    @summary: 检查系统软件版本是否是风险版本。若满足以下条件之一，则是风险版本。
    1) 系统软件版本为V500R007C00SPC100，无热补丁或者热补丁版本为V500R007C00SPH101且设备型号不为5300 V5和5500 V5 Elite。
    2) 系统软件版本为V300R001C21SPC100，无热补丁且设备型号为Dorado6000 V3或Dorado5000 V3。
    """
    allCliRet = ""
    
    productModel, cliRet = getProductModel(cli)
    allCliRet = common.joinLines(allCliRet, cliRet)
    
    softwareVersion, hotPatchVersion, cliRet = getProductVersion(cli)
    allCliRet = common.joinLines(allCliRet, cliRet)

    if softwareVersion == "V500R007C00SPC100" and \
       (isHotPatchVersionNull(hotPatchVersion) or hotPatchVersion == "V500R007C00SPH101") and \
       productModel not in ["5300 V5", "5500 V5 Elite"]:
        return True, softwareVersion, allCliRet
    
    if softwareVersion == "V300R001C21SPC100" and \
       isHotPatchVersionNull(hotPatchVersion) and \
       productModel in ["Dorado6000 V3", "Dorado5000 V3"]:
        return True, softwareVersion, allCliRet
    
    return False, softwareVersion, allCliRet


def isHotPatchVersionNull(hotPatchVersion):
    """
    @summary: 判断热补丁版本是否为空。
    """
    if not hotPatchVersion or (hotPatchVersion == "--"):
        return True
    
    return False


def getRiskCard(cli):
    """
    @summary: 获取系统上导致版本风险的接口卡。以下六种类型的接口卡在风险版本上存在数据损坏的风险。
    1) 4xGE的电口卡。
    2) FCoE_ISCSI模式的SmartIO扣卡。
    3) FC模式的SmartIO扣卡，且该SmartIO扣卡上的所有端口运行状态为Link Down。
    4) FCoE_ISCSI模式的SmartIO接口卡。
    5) FC模式的SmartIO接口卡，且该SmartIO接口卡上的所有端口运行状态为Link Down。
    6) IWARP模式的SmartIO接口卡，且该SmartIO接口卡上的所有端口运行状态为Link Down。
    """
    allCliRet = ""
    cardNameList = []
    
    # 存在电口卡，就存在风险，检查不通过
    flag, cliRet = isExistElectricPortCard(cli)
    allCliRet = common.joinLines(allCliRet, cliRet)
    if flag == True:
        cardName = getCardName("electricPortCard")
        cardNameList.append(cardName)
     
    # SmartIO接口卡
    smartIOCardIds, cliRet = getSmartIOCard(cli)
    allCliRet = common.joinLines(allCliRet, cliRet)
    
    # 获取FCoE模式的SmartIO扣卡
    smartIOBuckleCardList, cliRet = getFCoESmartIOBuckleCard(cli)
    allCliRet = common.joinLines(allCliRet, cliRet)
    if smartIOBuckleCardList:
        cardName = getCardName("smartIOBuckleCard")
        cardNameList.append(cardName)
        
    # 获取所有端口Link Down，FC模式的SmartIO扣卡
    linkDownFCSmartIOBuckleCardList = getLinkDownFCSmartIOBuckleCard(cliRet)
    if linkDownFCSmartIOBuckleCardList:
        cardName = getCardName("linkDownFCSmartIOBuckleCard")
        cardNameList.append(cardName)

    # 获取FCoE模式的SmartIO接口卡。
    FCoESmartIOCardList = getFCoESmartIOCard(cliRet, smartIOCardIds)
    if FCoESmartIOCardList:
        cardName = getCardName("FCoESmartIOCard")
        cardNameList.append(cardName)
    
    # 获取所有端口Link Down，FC模式的SmartIO接口卡。
    linkDownFCSmartIOCardList = getLinkDownFCSmartIOCard(cliRet, smartIOCardIds)
    if linkDownFCSmartIOCardList:
        cardName = getCardName("linkDownFCSmartIOCard")
        cardNameList.append(cardName)
    
    # 获取所有端口Link Down，IWARP模式的SmartIO接口卡。
    linkDownIWARPSmartIOCardList = getLinkDownIWARPSmartIOCard(cliRet, smartIOCardIds)
    if linkDownIWARPSmartIOCardList:
        cardName = getCardName("linkDownIWARPSmartIOCard")
        cardNameList.append(cardName)
        
    return cardNameList, allCliRet


def getCardName(card):
    """
    @summary: 获取接口卡名称。
    """
    cardNameDict = {
                       
        "electricPortCard":{
            "zh":u"4xGE电口卡",
            "en":"A 4xGE electrical interface module."
                            },
        "smartIOBuckleCard":{
            "zh":u"FCoE_ISCSI模式的SmartIO扣卡",
            "en":"A SmartIO buckle in the FCoE_ISCSI."
                            },
        "FCoESmartIOCard":{
            "zh":u"FCoE_ISCSI模式的SmartIO接口卡",
            "en":"A SmartIO interface module in the FCoE_ISCSI."
                            },
        "linkDownFCSmartIOCard":{
            "zh":u"FC模式的SmartIO接口卡，且该SmartIO接口卡上的所有端口运行状态为Link Down",
            "en":"A SmartIO interface module in the FC mode, and the running status of all ports on the SmartIO interface module is Link Down."
                            },
        "linkDownIWARPSmartIOCard":{
            "zh":u"IWARP模式的SmartIO接口卡，且该SmartIO接口卡上的所有端口运行状态为Link Down",
            "en":"A SmartIO interface module in the IWARP mode, and the running status of all ports on the SmartIO interface module is Link Down."
                            },
        "linkDownFCSmartIOBuckleCard":{
            "zh":u"FC模式的SmartIO扣卡，且该SmartIO扣卡上的所有端口运行状态为Link Down",
            "en":"A SmartIO buckle in the FC mode, and the running status of all ports on the SmartIO buckle is Link Down."
                            }
                   }
    
    return cardNameDict.get(card, {}).get(LANG)


def isExistElectricPortCard(cli):
    """
    @summary: 检查系统是否存在电口卡。
    """
    cmd = "show interface_module|filterRow column=Model predict=match value=4x(10)*GE\sElectrical\sInterface\sModule"
    cliRet = getInfoFromDevice(cli, cmd)
    
    if cliUtil.queryResultWithNoRecord(cliRet):
        return False, cliRet
    
    return True, cliRet


def getSmartIOCard(cli):
    """
    @summary: 获取SmartIO接口卡
    """
    cmd = "show interface_module|filterRow column=Model predict=equal_to value=4\sport\sSmartIO\sI/O\sModule"
    cliRet = getInfoFromDevice(cli, cmd)
    
    if cliUtil.queryResultWithNoRecord(cliRet):
        return [], cliRet
    
    idList = []
    
    infoDictList = cliUtil.getHorizontalCliRet(cliRet)
    for infoDict in infoDictList:
        id = infoDict.get("ID")
        if id:
           idList.append(id) 
    
    return idList, cliRet


def getMiddleString(allString, beginString, endString):
    """
    @summary: 获取两个关键字之前的字符串。
    """
    items = allString.split(beginString)
    if len(items) < 2:
        return ""
    
    tempCliRet = items[1]
    items = tempCliRet.split(endString)
    
    return items[0]


def getCard4Port(portId, cardIdList):
    """
    @summary: 获取端口属于cardIdList中哪一张接口卡。
    """
    for cardId in cardIdList:
        if cardId in portId:
            return cardId
    
    return ""


def getPortStatus4Card(cliRet, card):
    """
    @summary: 获取接口卡上的所有端口的运行状态。
    """
    portStatusList = []
         
    for line in cliRet.splitlines():
        items = line.split()
        if len(items) < 4:
            continue
        portId = items[0]
        if card not in portId:
            continue
        runningStatus = " ".join([items[2], items[3]])
        portStatusList.append(runningStatus)
        
    return portStatusList


def getSmartIOCard4Port(portString, smartIOCardIds):
    """
    @summary: 获取指定类型端口上的SmartIO接口卡。
    """
    # 某一类型端口下的接口卡即代表某一模式的接口卡
    modelSmartIOCardList = []
    
    for line in portString.splitlines():
        items = line.split()
        if not items:
            continue
        portId = items[0]
        smartIOCard = getCard4Port(portId, smartIOCardIds)
        if smartIOCard:
            modelSmartIOCardList.append(smartIOCard)
    
    return modelSmartIOCardList


def getFCSmartIOCard(cliRet, SmartIOCardIds):
    """
    @summary: 获取FC模式的SmartIO接口卡。
              show port general回显中FC port:下的端口所在的SmartIO卡即为FC模式的SmartIO接口卡。
    """    
    portString = getMiddleString(cliRet, "FC port:", "SAS port:")
    return getSmartIOCard4Port(portString, SmartIOCardIds)


def isAllPortLinkDown(portStatusList):
    """
    @summary: 判断所有端口的运行状态是否都是Link_Down。
    """
    if len(set(portStatusList)) == 1 and portStatusList[0] == "Link Down":
        return True
    
    return False


def getLinkDownFCSmartIOCard(cliRet, smartIOCardIds):
    """
    @summary: 获取所有端口Link Down，FC模式的SmartIO接口卡。
    """
    linkDownFCSmartIOCardList = []
    
    FCSmartIOCardList = getFCSmartIOCard(cliRet, smartIOCardIds)
    for FCSmartIOCard in FCSmartIOCardList:
        # 获取所有FC模式的SmartIO接口卡上的端口的运行状态
        portStatusList = getPortStatus4Card(cliRet, FCSmartIOCard)
        if isAllPortLinkDown(portStatusList):
            linkDownFCSmartIOCardList.append(FCSmartIOCard)
            
    return linkDownFCSmartIOCardList


def getFCoESmartIOCard(cliRet, smartIOCardIds):
    """
    @summary: 获取FCoE模式的SmartIO接口卡。
                                    若FCoE port下的端口ID以SmartIO接口卡ID为前缀，则表示该卡处于FCoE_ISCSI模式。
    """
    portString = getMiddleString(cliRet, "FCoE port:", "PCIE port:")
    return getSmartIOCard4Port(portString, smartIOCardIds)


def getIWARPSmartIOCard(cliRet, smartIOCardIds):
    """
    @summary: 获取IWARP模式的IWARP接口卡。
              show port general回显中IP Scale Out下的端口所在的SmartIO卡，即为为IWARP模式的SmartIO接口卡。
    """
    portString = getMiddleString(cliRet, "IP Scale Out:", "FC port:")
    return getSmartIOCard4Port(portString, smartIOCardIds)


def getLinkDownIWARPSmartIOCard(cliRet, smartIOCardIds):
    """
    @summary: 获取所有端口Link Down，IWARP模式的SmartIO接口卡。
    """
    linkDownIWARPSmartIOCardList = []
    
    IWARPSmartIOCardList = getIWARPSmartIOCard(cliRet, smartIOCardIds)
    for IWARPSmartIOCard in IWARPSmartIOCardList:
        # 获取所有IWARP模式的SmartIO接口卡上的端口的运行状态
        portStatusList = getPortStatus4Card(cliRet, IWARPSmartIOCard)
        if isAllPortLinkDown(portStatusList):
            linkDownIWARPSmartIOCardList.append(IWARPSmartIOCard)
            
    return linkDownIWARPSmartIOCardList


def getFCoESmartIOBuckleCard(cli):
    """
    @summary: 获取获取FCoE_ISCSI模式的SmartIO扣卡。
          若FCoE port和ETH port下的同一个端口ID以H加数字为后缀，则表示该端口所属的接口卡为FCoE_ISCSI模式的SmartIO扣卡。
    """
    cmd = "show port general"
    cliRet = getInfoFromDevice(cli, cmd)
    
    FCoEPortList = []
    FCoEPortString = getMiddleString(cliRet, "FCoE port:", "PCIE port:")
    for line in FCoEPortString.splitlines():
        items = line.split()
        if items and "." in items[0]:
            portId = items[0]
            FCoEPortList.append(portId)
                
    ethPortList = []
    ethPortString = getMiddleString(cliRet, "ETH port:", "FC port:")
    for line in ethPortString.splitlines():
        items = line.split()
        if items and "." in items[0]:
            portId = items[0]
            ethPortList.append(portId)
            
    # 既是FCoE类型又是ETH类型的端口
    portList = [port for port in ethPortList if port in FCoEPortList]
    
    smartIOBuckleCardList = []
    for port in portList:
        if re.match(".*\.H\d", port):
            smartIOBuckleCard = port[:port.rfind(".")]
            smartIOBuckleCardList.append(smartIOBuckleCard)
    
    return list(set(smartIOBuckleCardList)), cliRet


def getFCSmartIOBuckleCard(cliRet):
    """
    @summary: 获取获取FC模式的SmartIO扣卡。
          若FC port下存在端口ID以H加数字为后缀，则表示该端口所属的接口卡为FC模式的SmartIO扣卡。
    """
    FCSmartIOBuckleCardList = []
    
    FCPortString = getMiddleString(cliRet, "FC port:", "SAS port:")
    for line in FCPortString.splitlines():
        items = line.split()
        if items and "." in items[0]:
            portId = items[0]
            if re.match(".*\.H\d", portId):
                FCSmartIOBuckleCard = portId[:portId.rfind(".")]
                if FCSmartIOBuckleCard not in FCSmartIOBuckleCardList:
                    FCSmartIOBuckleCardList.append(FCSmartIOBuckleCard)
    
    return FCSmartIOBuckleCardList


def getLinkDownFCSmartIOBuckleCard(cliRet):
    """
    @summary: 获取所有端口Link Down，FC模式的SmartIO扣卡。
    """
    linkDownFCSmartIOBuckleCardList = []
    
    FCSmartIOBuckleCardList = getFCSmartIOBuckleCard(cliRet)
    for FCSmartIOBuckleCard in FCSmartIOBuckleCardList:
        # 获取所有FC模式的SmartIO扣卡上的端口的运行状态
        portStatusList = getPortStatus4Card(cliRet, FCSmartIOBuckleCard)
        if isAllPortLinkDown(portStatusList):
            linkDownFCSmartIOBuckleCardList.append(FCSmartIOBuckleCard)
    
    return list(set(linkDownFCSmartIOBuckleCardList))


def logException():
    """
    @summary: 打印异常栈到日志文件中。
    """
    import traceback
    LOGGER.logError(str(traceback.format_exc()))


def getErrMsg(errCode, args = ""):
    """
    @summary: 获取国际化错误消息。
    """
    return common.getMsg(LANG, errCode, args)


def getInfoFromDevice(cli, cmd):
    """
    @summary: 到设备上执行cmd，获取信息。
    """
    try:
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        if flag != True:
            raise common.UnCheckException(errMsg, cliRet)
        return cliRet
    except:
        logException()
        errMsg = getErrMsg("cannot.get.system.info")
        raise common.UnCheckException(errMsg, cliRet)
        

def getProductModel(cli):
    """
    @summary: 获取系统型号。
    """
    productModel = ""
    
    cmd = "show system general"
    cliRet = getInfoFromDevice(cli, cmd)
    
    for line in cliRet.splitlines():
        items = line.split(":")
        if len(items) < 2:
            continue

        if items[0].strip() == "Product Model":
            productModel = items[1].strip()
            break
        
    if not productModel:
        errMsg = getErrMsg("cannot.get.system.info")
        raise common.UnCheckException(errMsg, cliRet)
    
    return productModel, cliRet


def getProductVersion(cli):
    """
    @summary: 获取系统版本和热补丁版本。
    """
    flag, cliRet, errMsg, softwareVersion, hotPatchVersion = common.getVersion(cli, LANG)
    
    if flag != True or not softwareVersion:
        if not errMsg:
            errMsg = getErrMsg("cannot.get.system.info")
        raise common.UnCheckException(errMsg, cliRet)
        
    return softwareVersion, hotPatchVersion, cliRet
