# -*- coding: UTF-8 -*-

import traceback

import common
import cliUtil
from common import UnCheckException
from common_cache import not_support_nas_domain
import common_cache

PY_JAVA_ENV = py_java_env
LOGGER = common.getLogger(PY_LOGGER, __file__)
LANG = common.getLang(PY_JAVA_ENV)
LOCAL_SN = str(PY_JAVA_ENV.get("devInfo").getDeviceSerialNumber())
REMOTE_DEVICE_RET_HEADER = "\n\nON REMOTE DEVICE(SN:{})"
LOCAL_DEVICE_RET_HEADER = "ON LOCAL DEVICE(SN:{})\n"


def execute(cli):
    """
    租户Pair检查
    :param cli:
    :return:
    """
    item_check = ItemCheck(cli)
    flag, msg = item_check.execute_check()
    item_check.all_cli_ret.extend(item_check.remote_ret_list)
    return flag, "\n".join(item_check.all_cli_ret), msg


class ItemCheck:
    """
    租户Pair检查
    """

    def __init__(self, cli):
        self.cli = cli
        self.all_cli_ret = [LOCAL_DEVICE_RET_HEADER.format(LOCAL_SN)]
        self.remote_ret_list = list()
        self.domain_info_dict = dict()
        self.domain_remote_sn_info = dict()
        self.fs_local_to_remote_dict = dict()

    def execute_check(self):
        try:
            self.query_hyper_domain()
            LOGGER.logInfo("fs domain info:{}".format(self.domain_info_dict))
            if not self.domain_info_dict:
                return True, ""

            self.query_remote_device_info()
            LOGGER.logInfo("remote device info:{}".format(
                self.domain_remote_sn_info))
            flag, added_sn_list, msg = common.checkAddedRemoteDevSn(
                PY_JAVA_ENV, LANG
            )
            LOGGER.logInfo("added sn list is:{}".format(added_sn_list))
            if flag is not True:
                return cliUtil.RESULT_WARNING if common.is_opening_delivery_inspect(
                    PY_JAVA_ENV) else cliUtil.RESULT_NOCHECK, msg

            suggest_msg, error_msg, no_check_msg = self.check_consistency(
                added_sn_list
            )

            return self.get_check_result(error_msg, suggest_msg, no_check_msg)

        except UnCheckException as e:
            LOGGER.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            LOGGER.logError(traceback.format_exc())
            err_msg = "query.result.abnormal"
            return cliUtil.RESULT_NOCHECK, common.getMsg(LANG, err_msg)

    @staticmethod
    def get_check_result(error_msg, suggest_msg, no_check_msg):
        if error_msg:
            error_msg.extend(suggest_msg)
            error_msg.extend(no_check_msg)
            return False, "".join(error_msg)

        if suggest_msg:
            suggest_msg.extend(no_check_msg)
            suggest_msg.append(common.getMsg(LANG, "fs.type.diff.sugg"))
            return cliUtil.RESULT_WARNING, "".join(suggest_msg)

        if no_check_msg:
            return cliUtil.RESULT_WARNING if common.is_opening_delivery_inspect(
                PY_JAVA_ENV) else cliUtil.RESULT_NOCHECK, "".join(no_check_msg)

        return True, ""

    def check_consistency(self, added_sn_list):
        suggest_msg_list = []
        error_msg_list = []
        no_check_msg = []

        def check_capacity_consistency(fs_local_to_remote_dict):
            for local_fs_id, remote_fs_id in fs_local_to_remote_dict.items():
                capacity, block_size = self.query_filesystem(
                    local_fs_id, LOCAL_SN, self.all_cli_ret
                )
                remote_capacity, remote_block_size = self.query_filesystem(
                    remote_fs_id, remote_sn, self.remote_ret_list)
                if block_size != remote_block_size:
                    error_msg = common.getMsg(
                        LANG, "hypermetro.fs.some.info.consistency",
                        (local_fs_id, remote_fs_id,
                         "Block Size", block_size, remote_block_size)
                    )
                    suggest_msg_list.append(
                        error_msg
                    )

        for domain_id, remote_sn in self.domain_remote_sn_info.items():
            if remote_sn not in added_sn_list:
                no_check_msg.append(
                    common.getMsg(
                        LANG, "not.add.remote.device.again", remote_sn
                    )
                )
                continue

            fs_local_to_remote_dict = self.query_pair_filesystem_info(
                domain_id)
            self.remote_ret_list.append(
                REMOTE_DEVICE_RET_HEADER.format(remote_sn)
            )
            check_capacity_consistency(fs_local_to_remote_dict)

        return suggest_msg_list, error_msg_list, no_check_msg

    def query_filesystem(self, fs_id, dev_sn, ret_list):
        cmd = "show file_system general file_system_id={}".format(fs_id)
        flag, ret, msg = common.getObjFromFile(
            PY_JAVA_ENV, LOGGER, dev_sn, cmd, LANG
        )
        ret_list.append(ret)
        if not cliUtil.hasCliExecPrivilege(ret):
            return

        if flag is not True:
            raise UnCheckException(msg, ret)

        res_list = cliUtil.getVerticalCliRet(ret)
        for res_info in res_list:
            return (res_info.get("Capacity"), res_info.get("Block Size"))

    def query_hyper_domain(self):
        """
        1 双活仲裁服务器链路状态为Link Up；
        :return:
        """
        cmd = "show fs_hyper_metro_domain general"
        flag, ret, msg = common.getObjFromFile(
            PY_JAVA_ENV, LOGGER, LOCAL_SN, cmd, LANG
        )
        self.all_cli_ret.append(ret)

        if not_support_nas_domain(ret):
            return

        if flag is not True:
            raise UnCheckException(msg, ret)

        res_list = cliUtil.getHorizontalCliRet(ret)
        for res_info in res_list:
            self.domain_info_dict[res_info.get("ID")] = res_info.get(
                "Remote Device ID"
            )

    def query_remote_device_info(self):
        """
        1 双活仲裁服务器链路状态为Link Up；
        :return:
        """
        (
            flag,
            self.domain_remote_sn_info,
            msg,
            ret,
        ) = common.getRemoteDeviceInfo(
            PY_JAVA_ENV, LOCAL_SN, self.domain_info_dict, LOGGER, LANG
        )
        self.all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(msg, ret)

    def query_pair_filesystem_info(self, domain_id):
        """
        获取租户下文件系统数量
        :return:
        """
        fs_local_to_remote_dict = dict()
        flag, ret, msg, pair_list = common_cache.get_nas_pair_from_cache(
            py_java_env, LOGGER, LOCAL_SN, LANG)
        self.all_cli_ret.append(ret)
        if flag is not True:
            LOGGER.logInfo("Failed to get information about HyperMetro Pair")
            raise UnCheckException(msg)

        for res_info in pair_list:
            fs_local_to_remote_dict[res_info.get("Local ID")] = res_info.get(
                "Remote ID"
            )

        return fs_local_to_remote_dict
