# -*- coding: UTF-8 -*-
import cliUtil
import common
import common_cache
from com.huawei.ism.tool.obase.exception import ToolException
from common import UnCheckException

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """
    双活AD域DNS记录检查
    :param cli:
    :return:
    """
    cli_ret_list = []
    try:
        # 检查是否风险版本
        no_risk_flag, ret = check_not_risk_version(cli)
        cli_ret_list.append(ret)
        if no_risk_flag:
            return True, "\n".join(cli_ret_list), "Not a risk version."

        vstore_id_list, cli_ret = get_vstore_ids(cli)
        cli_ret_list.append(cli_ret)
        msg_list = []
        for vstore_id in vstore_id_list:
            enter_cli_ret = ""
            (err_port_dict, ret_list), exit_cli_ret = check_ddns_status(
                cli, LOGGER, LANG, enter_cli_ret, vstore_id
            )
            cli_ret_list.extend(ret_list)
            cli_ret_list.append(exit_cli_ret)
            if not err_port_dict:
                continue

            for err_port in err_port_dict:
                msg_list.append(
                    common.getMsg(
                        LANG,
                        "ad.domain.ddns.not.pass",
                        (err_port, ",".join(err_port_dict.get(err_port))),
                    )
                )
        if msg_list:
            return False, "\n".join(cli_ret_list), "".join(msg_list)
        return True, "\n".join(cli_ret_list), ""
    except UnCheckException as e:
        return (cliUtil.RESULT_NOCHECK, "\n".join(cli_ret_list), e.errorMsg)
    except ToolException as e:
        LOGGER.logError(str(e))
        return (
            cliUtil.RESULT_NOCHECK,
            "\n".join(cli_ret_list),
            common.getMsg(LANG, "query.result.abnormal"),
        )
    except Exception as e:
        LOGGER.logError(str(e))
        return (
            cliUtil.RESULT_NOCHECK,
            "\n".join(cli_ret_list),
            common.getMsg(LANG, "query.result.abnormal"),
        )
    finally:
        # 退出到cli模式
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode from some model ret is %s" % str(ret))
        # 退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)


def get_vstore_ids(cli):
    """
    获取租户ID列表
    :return:
    """
    vstore_id_list = []
    cmd = "show hyper_metro_vstore_pair general"
    flag, ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    # 当命令不支持时报通过
    if not cliUtil.hasCliExecPrivilege(ret):
        return vstore_id_list

    if flag is not True:
        raise UnCheckException(msg, ret)
    info_dict_list = cliUtil.getHorizontalCliRet(ret)
    for info_dict in info_dict_list:
        vstore_id = info_dict.get("Local Vstore ID", "")
        if vstore_id != "":
            vstore_id_list.append(vstore_id)
    return vstore_id_list, ret


@common.checkInVstoreModeWrap
def check_ddns_status(cli, logger, LANG, vstore_cli_ret,
                      vstore_id, *args, **kargs):
    """
    步骤5 进入租户后，执行命令：show domain ad，查询加入AD域状态；
    步骤6 进入租户后，执行命令：show logical_port general，查询逻辑IP信息；
    步骤7 根据步骤6中查询到的逻辑端口名（Logical Port Name）xxx执行命令：
    show logical_port general logical_port_name=xxx，查询逻辑IP的详细信息
    :param cli:
    :param logger:
    :param LANG:
    :param vstore_cli_ret:
    :param vstore_id:
    :param args:
    :param kargs:
    :return:
    """
    cli_ret_list = []
    err_port_dict = {}
    cli_ret_list.append(vstore_cli_ret)
    cmd = "show domain ad"
    flag, ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    cli_ret_list.append(ret)
    if flag is not True:
        raise UnCheckException(msg, ret)
    domain_ad_list = cliUtil.getVerticalCliRet(ret)
    for ad_info in domain_ad_list:
        if ad_info.get("Domain Status") != "Joined":
            return err_port_dict, cli_ret_list

    cmd = "show logical_port general|filterColumn include " \
          r"columnList=Logical\sPort\sName,DDNS\sStatus"
    flag, ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    cli_ret_list.append(ret)
    if flag is not True:
        raise UnCheckException(msg, ret)
    logic_port_list = cliUtil.getHorizontalCliRet(ret)
    for logic_port in logic_port_list:
        port_name = logic_port.get("Logical Port Name", "")
        if logic_port.get("DDNS Status") == "On":
            err_list = err_port_dict.get(vstore_id, [])
            err_list.append(port_name)
            err_port_dict[vstore_id] = err_list
    return err_port_dict, cli_ret_list


def check_not_risk_version(cli):
    """
    1. 如果版本是V300R006C20及其之后的版本检查通过；V500R007C10及其之后
    的版本检查通过；V300R006C00之前的版本检查通过；否则继续检查；
    2. 如果版本是V500R007C00SPC100且安装了V500R007C00SPH102及以后版本的
    补丁则检查通过；
        如果版本是V300R006C10SPC100且安装了V300R006C10SPH102补丁及以后版
        本的补丁则检查通过；否则继续检查；

    :param cli:
    :return: True: 无风险。 False 有风险
    """
    (
        flag,
        p_version,
        p_patch,
        ret,
        msg,
    ) = common_cache.get_version_and_patch_cache(PY_JAVA_ENV, cli, LOGGER)
    if p_version.startswith("V300R006") and p_version >= "V300R006C20":
        return True, ret
    if p_version.startswith("V500R007") and p_version >= "V500R007C10":
        return True, ret
    if (
        p_version.startswith("V300R006C10SPC100")
        and p_patch >= "V300R006C10SPH102"
    ):
        return True, ret
    if (
        p_version.startswith("V500R007C00SPC100")
        and p_patch >= "V500R007C00SPH102"
    ):
        return True, ret

    return False, ret
