# -*- coding: UTF-8 -*-
import re

from com.huawei.ism.tool.obase.exception import ToolException

import cliUtil
import common
import common_utils
import cli_util_cache
import common_cache

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)

ITEM_ID = "service_external_lun_bypass_cache"


def execute(cli):
    """
    service_external_lun_bypass_cache
    外部lun透读检测
    """
    all_cli_ret = []
    new_conn = None
    cur_progress = 0
    try:
        flag, ret, err_msg = cliUtil.enterDeveloperMode(cli, LANG)
        all_cli_ret.append(ret)
        cur_progress += 1
        common.refreshProcess(PY_JAVA_ENV, cur_progress, LOGGER)
        if flag is not True:
            return cliUtil.RESULT_NOCHECK, "\n".join(all_cli_ret), err_msg

        smart_cp_lun_ids_list, cur_progress = get_all_smart_cp_lun_ids(
            cli, all_cli_ret, cur_progress
        )
        if not smart_cp_lun_ids_list:
            return True, "\n".join(all_cli_ret), err_msg

        # 18000 建立SSH连接, 部分版本C61支持SVP外进debug
        flag, new_conn, err_msg = common.createDeviceCliContFor18000(
            cli, PY_JAVA_ENV, LOGGER, LANG
        )
        # 18000 需要重新进入developer
        if common.is18000(PY_JAVA_ENV, new_conn):
            flag, ret, err_msg = cliUtil.enterDeveloperMode(new_conn, LANG)
            all_cli_ret.append(ret)
            cur_progress += 1
            common.refreshProcess(PY_JAVA_ENV, cur_progress, LOGGER)
            if flag is not True:
                return cliUtil.RESULT_NOCHECK, "\n".join(all_cli_ret), err_msg

        flag, ret, err_msg = cliUtil.enter_debug_mode_from_developer(
            new_conn, LANG
        )
        all_cli_ret.append(ret)
        if flag is not True:
            return cliUtil.RESULT_NOCHECK, "\n".join(all_cli_ret), err_msg

        err_luns = get_dev_lun_info(
            new_conn, all_cli_ret, smart_cp_lun_ids_list, cur_progress
        )
        # 防止LUN太多，内存溢出。每个LUN的回文较大。
        all_cli_ret = common_cache.save_cli_ret_to_file(
            all_cli_ret, ITEM_ID, PY_JAVA_ENV, LOGGER
        )
        if err_luns:
            err_msg_key = "service.external.lun.bypass.cache.no.pass"
            return (
                False,
                all_cli_ret,
                common_utils.get_err_msg(
                    LANG, err_msg_key, ",".join(err_luns)
                ),
            )

        return True, all_cli_ret, ""
    except common.UnCheckException as e:
        LOGGER.logError(str(e))
        # 部分异常时需要拼接回文
        if e.cliRet not in all_cli_ret:
            all_cli_ret.append(e.cliRet)
        return cliUtil.RESULT_NOCHECK, "\n".join(all_cli_ret), e.errorMsg
    except (ToolException, Exception) as e:
        LOGGER.logError(str(e))
        return (
            cliUtil.RESULT_NOCHECK,
            "\n".join(all_cli_ret),
            common.getMsg(LANG, "query.result.abnormal"),
        )
    finally:
        if new_conn is not cli:
            common.closeDeviceCliContFor18000(
                new_conn, PY_JAVA_ENV, LOGGER, LANG
            )
        # 退出到cli模式
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode from some model ret is %s" % str(ret))

        # 退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)
        common.refreshProcess(PY_JAVA_ENV, 100, LOGGER)


def get_all_smart_cp_lun_ids(cli, all_cli_ret, cur_progress):
    """
    获取smart cache partition 下的所以LUN
    :param cli: ssh连接
    :param all_cli_ret: 回文
    :param cur_progress: 进度
    :return: 分区下LUN ID列表, 已刷新进度
    """
    # 获取smart cache partition
    ret, smart_cp_list = cli_util_cache.get_smart_cache_partition_cache(
        PY_JAVA_ENV, cli, LOGGER
    )
    all_cli_ret.append(ret)
    cur_progress += 1
    common.refreshProcess(PY_JAVA_ENV, cur_progress, LOGGER)
    if not smart_cp_list:
        return [], cur_progress

    smart_cp_id_list = [
        smart_cp_dict.get("ID") for smart_cp_dict in smart_cp_list
    ]

    part_progress = 10.0
    # 防止刷新太频繁，导致UI挂掉
    tmp_progress = 0
    per_progress = part_progress / len(smart_cp_id_list)
    all_check_luns = []
    for scp_id in smart_cp_id_list:
        ret, scp_detail_list = cli_util_cache.get_smart_cp_detail_cache(
            PY_JAVA_ENV, cli, LOGGER, scp_id
        )
        all_cli_ret.append(ret)
        lun_ids = [scp_detail.get("LUN ID") for scp_detail in scp_detail_list]
        all_check_luns.extend(lun_ids)
        tmp_progress += per_progress
        cur_progress += per_progress
        if tmp_progress >= 1:
            common.refreshProcess(PY_JAVA_ENV, cur_progress, LOGGER)
            tmp_progress = 0

    return list(set(all_check_luns)), cur_progress


def get_dev_lun_info(cli, all_cli_ret, lun_ids, cur_progress):
    """
    获取devlun showctrl [lunID]，获取Bypass cache字段值判断是否为yes
    :param cli: ssh连接
    :param all_cli_ret: 回文
    :param lun_ids: 待检查lun ids
    :param cur_progress: 当前进度
    :return: 不通过的lun
    """
    regx = re.compile(r"Bypass cache\s*:\s*yes")
    err_lun_ids = []
    per_lun_progress = (100.0 - cur_progress) / len(lun_ids)
    tmp_progress = 0
    for lun_id in lun_ids:
        cmd = "devlun showctrl {}".format(lun_id)
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
        all_cli_ret.append(ret)
        cur_progress += per_lun_progress
        tmp_progress += per_lun_progress
        if tmp_progress >= 1:
            common.refreshProcess(PY_JAVA_ENV, cur_progress, LOGGER)
            tmp_progress = 0

        if flag is not True:
            raise common.UnCheckException(err_msg, ret)

        match_res = regx.findall(ret)
        if match_res:
            err_lun_ids.append(lun_id)

    return err_lun_ids
