# -*- coding: UTF-8 -*-
import cliUtil
import common
import traceback
import config
import common_cache
from common import UnCheckException

import cli_util_cache as cache_util
import common_utils

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
ALL_CLI_RET_LIST = []
SUGGESTION_HOST_LIST = ["HP-UX"]
sepereterA = {"zh": u"，", "en": ","}.get(LANG)
sepereterB = {"zh": u"；", "en": ";"}.get(LANG)
sepereterC = {"zh": u"。", "en": "."}.get(LANG)
sepereterD = {"zh": u"：", "en": ":"}.get(LANG)


def execute(cli):
    """
    @summary: 双活配置一致性检查
        变更记录：2018.9.20方案变更，采用以双活LUN为基准判断映射的主机启动器信息是否一致。
        检查步骤：
        步骤1 以admin用户登录双活任一站点设备。
        步骤2 执行show hyper_metro_domain general 查询本端设备上的双活域ID（ID）和远端设备的ID（Remote Device ID）。
        步骤3 执行show remote_device general 查询出远端设备ID（ID）和SN（SN）。
        步骤4 执行show hyper_metro_pair general |filterRow column=Type predict=equal_to value=LUN |filterRow column=Domain\sID predict=equal_to value=? (步骤2中查询出任一条记录中的双活域domain id) 查询该双活域下双活pair的pair ID。
        步骤5 执行show hyper_metro_pair general pair_id=？ 查询pair的本端LUN ID（Local ID）和远端LUN ID（Remote ID）。
        步骤6 执行show host general 查询出所有的主机ID。
        步骤7 执行show host lun host_id=？ host_id为步骤6中查询到的所有host id。根据步骤5中查询出的本端双活LUNID去掉没有映射双活LUN的主机
        步骤8 执行show initiator host_id=？ initiator_type=FC 查询主机FC启动器信息： Multipath Type、Failover Mode和Special Mode Type、WWN、Running Status。
        步骤9 执行show initiator host_id=？ initiator_type=iSCSI查询主机ISCSI启动器信息： Multipath Type、Failover Mode和Special Mode Type、IQN、Running Status。
        步骤10 执行show ib_initiator general host_id=?  查询主机IB启动器信息： Multi Path Type、Failover Mode和Special Mode Type、WWN、Running Status。
        步骤11 分别登录其他双活域（步骤2中查询出双活域ID）的远端设备执行步骤4-10中的命令查询远端设备的主机及启动器的信息。
        步骤12 整理前序步骤中查询到的信息，得出所有双活两端LUN对应的启动器，启动器对应的主机、主机操作系统类型、启动器信息以及LUN对应的主机LUNID。
        检查标准：
    1. 步骤2中，如果查询不到双活域记录，则直接返回检查通过，否则继续检查同Domain的本端和远端设备的下列信息是否一致。
    2. 步骤12中，如果双活pair两端LUN对应的启动器WWN或IQN不一致，报双活LUN启动器配置不一致错误，检查结果为：不通过（如果版本是V300R006C00之后，Dorado V300R001C21和V500R007C00及之后的版本，检查结果为：建议优化），继续检查。
    3. 步骤12中，如果双活pair两端LUN对应的启动器所在主机的主机操作系统类型不一致，报双活LUN映射主机的操作系统类型不一致错误，继续检查。
    4. 步骤12中，如果双活pair两端LUN对应的启动器配置（Multipath Type、Failover Mode、Special Mode Type、Running Status）不一致，报双活LUN启动器配置不一致错误，继续检查。提取的回显中如果没有Failover Mode和Special Mode Type，则跳过不进行这两个字段比较（老版本V300R003C20之前、V300R005全系列、V300R006C00启动器配置回显中不存在Failover Mode和Special Mode Type）。
    5. 步骤12中，如果双活pair两端LUN在对应的启动器的Host下Host LUN ID不一致，如果不是VMware操作系统直接通过，否则请确认实际主机操作系统类型是否为VMware 6.5版本，如果是则按修复建议整改，否则请忽略。
    6. 步骤12中，如果双活pair两端LUN在对应启动器配置的运行状态为“Offline”， 则检查不通过。
    """
    global ALL_CLI_RET_LIST
    errMsgInitiator = ""
    suggestionMsg = ""
    errMsgConfig = ""
    errMsgHostLun = ""
    noCheckMsg = ""
    errMsgOper = ""
    errMsgIniOffline = ""
    errMsgUnMapped = ""
    item = "service_hypermetro_configuration_consistence_doradov6"
    try:

        localDevSn = PY_JAVA_ENV.get("devInfo").getDeviceSerialNumber()
        ALL_CLI_RET_LIST.append("ON LOCAL DEVICE(SN:%s)" % localDevSn)

        isRisk = checkIsRiskVersion(localDevSn)
        LOGGER.logInfo("isRisk : %s, " % isRisk)
        if isRisk is not True:
            return True, "\n".join(ALL_CLI_RET_LIST), ''

        # 检查是否有双活Domain域
        domainDict = getDomainInfo(localDevSn)
        LOGGER.logInfo("isRisk domainDict: %s, " % domainDict)
        if not domainDict:
            return True, "\n".join(ALL_CLI_RET_LIST), ''

        # 检查是否有双活pair
        pairList = getAllPairInfo(localDevSn)
        LOGGER.logInfo("isRisk pairList: %s, " % pairList)
        if not pairList:
            return True, "\n".join(ALL_CLI_RET_LIST), ''

        # 检查是否添加了远端设备
        addedSnList = getAddedRemoteDevSn(cli)
        LOGGER.logInfo("checkAddedRemoteDevSn addedSnList:%s" % addedSnList)

        domainRemoteDevDict = getRemoteDeviceInfo(localDevSn, domainDict)
        for domainId in domainRemoteDevDict:
            # 查询该domainId下是否存在双活pair
            hyperMetroPairIdList, pair_info_list = getHyperMetroPairIdList(
                localDevSn, domainId)
            if not hyperMetroPairIdList:
                continue

            remoteDevSn = domainRemoteDevDict[domainId]
            if remoteDevSn not in addedSnList:
                noCheckMsg += common.getMsg(LANG, "not.add.remote.device.again", remoteDevSn)
                continue
            (
                hostDictLocal, hostDictRemote, initiatorDictLocal,
                localLunDict, pairToLunDict, remoteLunDict
            ) = get_hyper_metro_pair_detai_info(
                ALL_CLI_RET_LIST, localDevSn, pair_info_list, remoteDevSn)

            initiatorDictRemote = updateHostInitiatorInfo(remoteDevSn, hostDictRemote)

            if len(domainRemoteDevDict) > 1:
                ALL_CLI_RET_LIST.append("\n\nON LOCAL DEVICE(SN:%s)" %
                                        localDevSn)

            (
                errMsgInitiator, suggestionMsg, errMsgOper, errMsgConfig,
                errMsgHostLun, errMsgIniOffline, errMsgUnMapped
            ) = checkLunMappedInitiatorConsistent(
                hostDictLocal, localLunDict, hostDictRemote, remoteLunDict,
                getCurrentVersion(localDevSn), localDevSn, remoteDevSn,
                initiatorDictLocal, initiatorDictRemote, hyperMetroPairIdList,
                pairToLunDict)
        allCliRet = common_cache.save_cli_ret_to_file(ALL_CLI_RET_LIST,
                                                      item,
                                                      PY_JAVA_ENV, LOGGER)
        if errMsgInitiator or errMsgOper or errMsgConfig or errMsgHostLun or errMsgIniOffline or errMsgUnMapped:
            return False, allCliRet, errMsgOper + errMsgInitiator + errMsgConfig + errMsgHostLun + errMsgIniOffline + \
                   errMsgUnMapped + suggestionMsg + noCheckMsg

        if suggestionMsg:
            return cliUtil.RESULT_WARNING, allCliRet, suggestionMsg + noCheckMsg

        if noCheckMsg:
            return common_utils.get_result_bureau(py_java_env, allCliRet, noCheckMsg)

        # 检查通过
        return True, allCliRet, ''

    except UnCheckException, unCheckException:
        LOGGER.logInfo(u"UnCheckException, errMsg: %s" % unCheckException.errorMsg)
        ret = common_cache.save_cli_ret_to_file(ALL_CLI_RET_LIST, item,
                                                PY_JAVA_ENV, LOGGER)
        if not unCheckException.flag:
            return cliUtil.RESULT_NOCHECK, ret, unCheckException.errorMsg

        return unCheckException.flag, ret, unCheckException.errorMsg
    except Exception:
        LOGGER.logError(traceback.format_exc())
        ret = common_cache.save_cli_ret_to_file(ALL_CLI_RET_LIST, item,
                                                PY_JAVA_ENV, LOGGER)
        error_key = "query.result.abnormal"
        return cliUtil.RESULT_NOCHECK, ret, common.getMsg(LANG, error_key)

    finally:
        # 退出到cli模式
        ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
        LOGGER.logInfo("enter cli mode from some model ret is %s" % str(ret))

        # 退出失败后为不影响后续检查项重新连接cli
        if not ret[0]:
            common.reConnectionCli(cli, LOGGER)


def get_hyper_metro_pair_detai_info(ALL_CLI_RET_LIST, localDevSn,
                                    pair_info_list, remoteDevSn):
    # 查询出本端LUN ID 和 远端LUN ID, devLunDict每个远端设备对应的Lun id 列表。
    (localLunDict, remoteLunDict, localLunToRemoteLunDict,
     pairToLunDict) = getHyperMetroPairLunInfo(
        localDevSn, pair_info_list)
    # 过滤出双活主机
    hy_host_list = []
    if cache_util.is_downloaded_config(PY_JAVA_ENV, localDevSn):
        LOGGER.logInfo("{0} use config.txt!".format(__file__))
        hy_host_list = cache_util.get_hyper_host_from_config(
            py_java_env,
            localDevSn, LOGGER,
            localLunDict.keys())
    LOGGER.logInfo("hyper host local:{0}{1}".format(
        localDevSn, str(hy_host_list))
    )
    # 根据主机组ID信息查询主机操作系统和主机ID
    hostDictLocal = updateHostInfo(localDevSn)
    LOGGER.logInfo("host Dict Local local:{0}{1}".format(
        localDevSn, str(hostDictLocal))
    )
    # 根据主机查询 LUN ID和Host LUN ID
    updateHostLunInfo(localDevSn, hostDictLocal, hy_host_list)
    # 根据双活LUN信息，只保留双活主机
    hostDictLocal = updateHyperMetroHost(hostDictLocal, localLunDict)
    LOGGER.logInfo("host Dict Local after update hy:{0}{1}".format(
        localDevSn, str(hostDictLocal))
    )
    # 根据主机ID查询启动器信息FC/ISCSI/IB
    initiatorDictLocal = updateHostInitiatorInfo(localDevSn, hostDictLocal)
    ALL_CLI_RET_LIST.append("\n\nON REMOTE DEVICE(SN:%s)" %
                            remoteDevSn)
    # 过滤出双活主机
    hy_host_list = []
    if cache_util.is_downloaded_config(PY_JAVA_ENV, remoteDevSn):
        LOGGER.logInfo("{0} use config.txt!".format(__file__))
        hy_host_list = cache_util.get_hyper_host_from_config(
            py_java_env,
            remoteDevSn, LOGGER,
            remoteLunDict.keys())
    LOGGER.logInfo("hyper host remote:{0}{1}".format(
        remoteDevSn, str(hy_host_list))
    )
    hostDictRemote = updateHostInfo(remoteDevSn)
    LOGGER.logInfo("host Dict Remote local:{0}{1}".format(
        remoteDevSn, str(hostDictRemote))
    )
    updateHostLunInfo(remoteDevSn, hostDictRemote, hy_host_list)
    updateHyperMetroHost(hostDictRemote, remoteLunDict)
    LOGGER.logInfo("host Dict Remote after update hy:{0}{1}".format(
        remoteDevSn, str(hostDictRemote))
    )
    return (hostDictLocal, hostDictRemote, initiatorDictLocal,
            localLunDict, pairToLunDict, remoteLunDict)


def updateHyperMetroHost(hostDict, lunDict):
    """
    过滤双活主机
    :param hostDict:
    :param lunDict:
    :return:
    """
    hyperLunList = lunDict.keys()
    noHyperHosts = []
    for hostId in hostDict:
        hostLunDict = hostDict[hostId].get("hostLunInfo", {})
        hostHyperLun = []
        hostLunList = hostLunDict.keys()
        for lunId in hostLunList:
            if lunId in hyperLunList:
                hostHyperLun.append(lunId)
            else:
                hostLunDict.pop(lunId)
        if not hostHyperLun:
            noHyperHosts.append(hostId)
    if noHyperHosts:
        for hostId in noHyperHosts:
            hostDict.pop(hostId)

    return hostDict


def checkIsRiskVersion(localDevSn):
    """
    dorado v3r1c00不支持双活
    :param localDevSn:
    :return:
    """
    currentVersion = getCurrentVersion(localDevSn)
    if currentVersion in config.DORADO_NO_HYPERMETRO_PRODUCTS:
        return False
    return True


def getCurrentVersion(localDevSn):
    global ALL_CLI_RET_LIST
    cmd = "show upgrade package"
    flag, cliRet, errMsg = common.getObjFromFile(py_java_env, LOGGER, localDevSn, cmd, LANG)
    ALL_CLI_RET_LIST.append(cliRet)
    softwareVersionIndex = cliRet.find("Software Version")
    hotPatchVersionIndex = cliRet.find("HotPatch Version")
    softwareVersionList = cliUtil.getHorizontalCliRet(cliRet[softwareVersionIndex:hotPatchVersionIndex])
    for version in softwareVersionList:
        LOGGER.logInfo("getCurrentVersion :%s" % str(version["Current Version"]))
        return str(version["Current Version"])


def getAllPairInfo(devSn):
    """
    获取主机信息
    :param devSn:
    :return:
    """
    global ALL_CLI_RET_LIST
    flag, ret, msg, pair_list = common_cache.get_san_pair_from_cache(
        py_java_env, LOGGER, devSn, LANG)
    ALL_CLI_RET_LIST.append(ret)
    if flag is not True:
        LOGGER.logInfo("Failed to get information about HyperMetro Pair")
        raise UnCheckException(msg)

    return pair_list


def updateHostLunInfo(devSn, hostDict, hyper_host_list=[]):
    """
    更新主机下的LUN信息
    :param devSn:
    :param hostDict:
    :return:
    """
    global ALL_CLI_RET_LIST
    hostLunIdInfoDict = {}
    for hostId in hostDict:
        if hyper_host_list and hostId not in hyper_host_list:
            continue
        cmd = "show host lun host_id=%s" % hostId
        flag, cliRet, errMsg = common.getObjFromFile(py_java_env, LOGGER, devSn, cmd, LANG)
        ALL_CLI_RET_LIST.append(cliRet)
        if flag is not True:
            LOGGER.logInfo("Failed to get lun errMsg:%s" % errMsg)
            raise UnCheckException(errMsg, cliRet)

        hostLunDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)

        hostLunDict = {}
        for hostLun in hostLunDictList:
            hostLunId = hostLun.get("Host LUN ID", '')
            lunId = hostLun.get("LUN ID", '')
            if not hostLunId or not lunId:
                continue
            hostLunDict[lunId] = {"hostLunId": hostLunId,
                                  "hostId": hostId}
            hostLunIdInfoDict[lunId] = {"hostLunId": hostLunId,
                                        "hostId": hostId}

        hostDict[hostId]["hostLunInfo"] = hostLunDict

    return hostLunIdInfoDict


def updateHostInitiatorInfo(devSn, hostDict):
    """
    获取启动器信息
    :param devSn:
    :param hostDict:
    :return:
    """
    initiatorDict = {}

    cmd = "show initiator initiator_type=FC"
    initiatorInfoDict = updateInitiatorInfo(devSn, cmd, "FC", "WWN",
                                            "Multipath Type", hostDict)
    initiatorDict.update(initiatorInfoDict)

    cmd = "show initiator initiator_type=iSCSI"
    initiatorInfoDict = updateInitiatorInfo(devSn, cmd, "ISCSI",
                                            "iSCSI IQN", "Multipath Type",
                                            hostDict,
                                            )
    initiatorDict.update(initiatorInfoDict)

    cmd = "show nvme_over_roce_initiator general"
    initiatorInfoDict = updateInitiatorInfo(devSn, cmd, "nvme",
                                            "NQN", "Multipath Type",
                                            hostDict,
                                            )
    initiatorDict.update(initiatorInfoDict)

    return initiatorDict


def updateInitiatorInfo(devSn, cmd, initiatorType, wwnType,
                        multiPathType, host_dict):
    """
    获取不同类型的启动器配置信息
    :param devSn:
    :param cmd:
    :param initiatorType:
    :param wwnType:
    :param multiPathType:
    :param host_dict:
    :return:
    """
    global ALL_CLI_RET_LIST
    flag, cliRet, errMsg = common.getObjFromFile(py_java_env, LOGGER, devSn,
                                                 cmd, LANG)
    ALL_CLI_RET_LIST.append(cliRet)
    if not cliUtil.hasCliExecPrivilege(cliRet):
        return {}

    if flag is not True:
        LOGGER.logInfo(
            "Failed to get initiator %s. errMsg:%s" % (
                initiatorType, errMsg))
        raise UnCheckException(errMsg, cliRet)

    initiatorDictList = cliUtil.getHorizontalCliRet(cliRet)
    initiatorDict = {}
    for tmpInitiatorDict in initiatorDictList:
        initiatorInfoDict = {}
        multiPath = tmpInitiatorDict.get(multiPathType, '')
        wwn = tmpInitiatorDict.get(wwnType, '')
        host_id = tmpInitiatorDict.get('Host ID', '')
        failoverMode = tmpInitiatorDict.get('Failover Mode', '')
        specialModeType = tmpInitiatorDict.get('Special Mode Type', '')
        runningStatus = tmpInitiatorDict.get('Running Status', '')
        if host_id == '--' or not wwn or host_id not in host_dict:
            continue
        hostInfoDict = host_dict.get(host_id, {})
        initiatorList = hostInfoDict.get("initiatorList", [])
        initiatorInfoDict['failoverMode'] = failoverMode
        initiatorInfoDict['specialModeType'] = specialModeType
        initiatorInfoDict['multiPath'] = multiPath
        initiatorInfoDict['runningStatus'] = runningStatus
        initiatorInfoDict['hostId'] = host_id
        initiatorList.append(wwn)
        hostInfoDict[wwn] = initiatorInfoDict
        initiatorDict[wwn] = initiatorInfoDict
        host_dict[host_id] = hostInfoDict
        hostInfoDict["initiatorList"] = initiatorList

    return initiatorDict


def updateHostInfo(devSn):
    """
    获取主机信息
    :param devSn:
    :return:
    """
    global ALL_CLI_RET_LIST
    hostDict = {}
    cmd = "show host general |filterColumn include columnList=Operating\sSystem,Access\sMode,ID"
    flag, cliRet, errMsg = common.getObjFromFile(py_java_env, LOGGER, devSn, cmd, LANG)
    ALL_CLI_RET_LIST.append(cliRet)
    if flag is not True:
        LOGGER.logInfo("Failed to get information host:%s" % errMsg)
        raise UnCheckException(errMsg, cliRet)

    hostDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)

    for tmpHostDict in hostDictList:
        operSystem = tmpHostDict.get("Operating System", '')
        accessModel = tmpHostDict.get("Access Mode", '')
        hostId = tmpHostDict.get("ID", '')
        if not operSystem or not hostId:
            continue

        hostDict[hostId] = {"operatingSystem": operSystem, "accessModel":accessModel}
    return hostDict


def getDomainInfo(devSn):
    """
    获取domain信息
    :param devSn:
    :return:
    """
    global ALL_CLI_RET_LIST
    domainDict = {}
    cmd = "show hyper_metro_domain general"
    flag, cliRet, errMsg = common.getObjFromFile(py_java_env, LOGGER, devSn,
                                                 cmd, LANG)
    ALL_CLI_RET_LIST.append(cliRet)
    if flag != True:
        LOGGER.logInfo("Failed to get information about HyperMetro domain. errMsg:%s" % errMsg)
        raise UnCheckException(errMsg, cliRet)

    hyperMetroDomainList = cliUtil.getHorizontalCliRet(cliRet)
    for domainInfo in hyperMetroDomainList:
        domainId = domainInfo.get("ID", '')
        remoteDeviceId = domainInfo.get("Remote Device ID", '')
        domainDict[domainId] = remoteDeviceId

    return domainDict


def getRemoteDeviceInfo(devSn, domainDict):
    """
    获取远端设备信息
    :param devSn:
    :param domainDict:
    :return:
    """
    global ALL_CLI_RET_LIST
    remoteDeviceDict = {}
    cmd = "show remote_device general"
    flag, cliRet, errMsg = common.getObjFromFile(py_java_env, LOGGER, devSn, cmd, LANG)
    ALL_CLI_RET_LIST.append(cliRet)
    if flag is not True:
        LOGGER.logInfo("Failed to get information about remote device. errMsg:%s" % errMsg)
        raise UnCheckException(errMsg, cliRet)

    remoteDeviceList = cliUtil.getHorizontalCliRet(cliRet)
    for remoteDev in remoteDeviceList:
        remoteDeviceId = remoteDev.get("ID", '')
        remoteDeviceSn = remoteDev.get("SN", '')
        for domainId in domainDict:
            if remoteDeviceId == domainDict[domainId]:
                remoteDeviceDict[domainId] = remoteDeviceSn

    return remoteDeviceDict


def getHyperMetroPairLunInfo(devSn, hyperMetroPairIdList):
    """
    @summary: 获取双活pair的本端LUN和远端LUN的ID信息
    @param hyperMetroPairIdList: 双活pair id list
    @return: localLunDict
            remoteLunDict
    """
    global ALL_CLI_RET_LIST
    localLunDict = {}
    remoteLunDict = {}
    localLunToRemoteLunDict = {}
    pairToLunDict = {}
    for pair_info in hyperMetroPairIdList:
        pair_id = pair_info.get("ID")
        local_id = pair_info.get("Local ID", '')
        remote_id = pair_info.get("Remote ID", '')
        # 收集每个devSn自己的 lun。
        localLunDict[local_id] = pair_id
        remoteLunDict[remote_id] = pair_id
        localLunToRemoteLunDict[local_id] = remote_id
        pairToLunDict[pair_id] = {
            "localLunId": local_id,
            "remoteLunId": remote_id
        }
    return localLunDict, remoteLunDict, localLunToRemoteLunDict, pairToLunDict


def getHyperMetroPairIdList(devSn, domainId):
    """
    @summary: 获取双活pair
    @return: hyperMetroPairIdList 双活pair ID列表
    """
    global ALL_CLI_RET_LIST
    flag, ret, msg, pair_list = common_cache.get_san_pair_from_cache(
        py_java_env, LOGGER, devSn, LANG)
    if ret not in ALL_CLI_RET_LIST:
        ALL_CLI_RET_LIST.append(ret)
    if flag is not True:
        LOGGER.logInfo("Failed to get information about HyperMetro Pair")
        raise UnCheckException(msg)
    pair_info_list = []
    pair_id_list = []
    for pair_info in pair_list:
        if pair_info.get("Domain ID") == domainId:
            pair_info_list.append(pair_info)
            pair_id_list.append(pair_info.get("ID"))

    return pair_id_list, pair_info_list


def getAddedRemoteDevSn(cli):
    """
    @summary: 获取双活远端设备
    """
    snList = PY_JAVA_ENV.get("devInfo").getRemoteSNs()
    selectDevs = PY_JAVA_ENV.get("selectDevs")
    LOGGER.logInfo("isRisk getAddedRemoteDevSn")
    # 如果未None表示添加设备时检查远端设备未成功。
    if snList is None:
        LOGGER.logInfo("isRisk snList is None")
        flag, _, errMsg, snList = common.getHyperMetroRemoteDeviceSn(cli, LANG)
        if snList is None:
            return []

    # 如果为空列表，表示没有远端双活设备
    if not snList:
        return []

    selectSn = [devNode.getDeviceSerialNumber() for devNode in selectDevs]
    addedSn = [devSn for devSn in snList if devSn in selectSn]

    # 如果存在远端设备，但一台都没有添加则报未检查。否则尽量去检查。
    if not addedSn:
        LOGGER.logInfo("isRisk addedSn is []")
        return []

    LOGGER.logInfo("isRisk addedSn is: %s" % addedSn)
    return addedSn


def getUnMappedLunInfo(allLunInitiatorDict, hyperMetroPairIdList, pairToLunDict, key):
    """
    没有在主机中找到的双活LUN就是未映射的双活LUN。
    :param allLunInitiatorDict:
    :param hyperMetroPairIdList:
    :param pairToLunDict:
    :param key:
    :return:
    """
    # 获取未映射的本端LUN
    unMappedLunList = []
    allMappedPairsList = allLunInitiatorDict.keys()
    for pairId in hyperMetroPairIdList:
        if pairId not in allMappedPairsList:
            unMappedLunList.append(pairToLunDict.get(pairId).get(key))

    return unMappedLunList


def checkLunMappedInitiatorConsistent(localHostDict, localLunPairDict, remoteHostDict, remoteLunPairDict,
                                      productVersion, localSn, remoteSn, initiatorDictLocal,
                                      initiatorDictRemote, hyperMetroPairIdList, pairToLunDict):
    """
    检查双活Pair(LUN)映射的启动器是否一致, 以pair的维度检查每个pair映射的主机启动器信息是否一致，
    相同启动器的配置是否一致，
    vmware 时host lun id是否一致
    :param localHostDict: 本端主机包含主机ID，启动器，类型等信息
    :param localLunPairDict: 本端同一个pair和LUN ID的对应关系
    :param remoteHostDict: 远端主机包含主机ID，启动器，类型等信息
    :param remoteLunPairDict: 远端同一个pair和LUN ID的对应关系
    :param productVersion: 版本信息，用于判断是否可能为直连组网的版本
    :param localSn:
    :param remoteSn:
    :param initiatorDictLocal: 本端所有启动器信息
    :param initiatorDictRemote: 远端所有启动器的信息
    :return: 检查的全部错误信息
    """
    global ALL_CLI_RET_LIST
    errMsgInitiator = ''
    suggestionErrMsg = ''
    errMsgOper = ''
    errMsgConfig = ''
    errMsgHostLun = ''
    errMsgIniOffline = ''
    errMsgUnMapped = ''
    errMsgDifLunId = ''
    errMsgIniConfigList = []
    errMsgIniOfflineList = []
    errMsgConfigList = []
    unMappedLunListLocal = []
    unMappedLunListRemote = []
    errMsgDifLunIdList = []
    allLunInitiatorDictLocal = getLunMappedInitiator(localHostDict, localLunPairDict)
    allLunInitiatorDictRemote = getLunMappedInitiator(remoteHostDict, remoteLunPairDict)
    LOGGER.logInfo("allLunInitiatorDictLocal:%s\n\n allLunInitiatorDictRemote:%s" % (allLunInitiatorDictLocal,
                                                                                     allLunInitiatorDictRemote))

    # 获取未映射的LUN
    unMappedLunListLocal = getUnMappedLunInfo(allLunInitiatorDictLocal, hyperMetroPairIdList, pairToLunDict,
                                              "localLunId")
    unMappedLunListRemote = getUnMappedLunInfo(allLunInitiatorDictRemote, hyperMetroPairIdList, pairToLunDict,
                                               "remoteLunId")
    error_ini_info = set()
    for pairId in allLunInitiatorDictLocal:
        pairInfoLocal = allLunInitiatorDictLocal.get(pairId, {})
        pairInfoRemote = allLunInitiatorDictRemote.get(pairId, {})
        iniListLocal = pairInfoLocal.get("iniList", [])
        iniListRemote = pairInfoRemote.get("iniList", [])
        localLunId = pairToLunDict.get(pairId, {}).get("localLunId", '')
        remoteLunId = pairToLunDict.get(pairId, {}).get("remoteLunId", '')
        localHostList = pairInfoLocal.get("hostList", [])
        remoteHostList = pairInfoRemote.get("hostList", [])

        # 不加启动器限制。
        # 双活pair本端和远端所有的主机访问模式都要一致，本端和本端，远端和远端。
        # 如果不一致，检查结果为不通过，建议修改Access Mode为一致。
        check_ini_list = [ini for ini in iniListLocal if ini in iniListRemote]
        check_host_access_mode(
            error_ini_info, check_ini_list, initiatorDictLocal, initiatorDictRemote, localHostDict, remoteHostDict)
        # 远端多出的启动器
        iniNotInLocal = [ini for ini in iniListRemote if ini not in iniListLocal]
        # 本端多出的启动器
        iniNotInRemote = [ini for ini in iniListLocal if ini not in iniListRemote]
        LOGGER.logInfo("pairId:%s, localLunId:%s,remoteLunId:%s, iniNotInLocal:%s, iniNotInRemote:%s" % (
            pairId, localLunId, remoteLunId, iniNotInLocal, iniNotInRemote))

        # 判断不一致的启动器信息并获取错误信息。
        suggestionErrMsgTemp, noPassErrMsgTemp = getErrMsgOfIniNotConsistence(iniNotInLocal,
                                                                              iniNotInRemote,
                                                                              productVersion,
                                                                              localLunId,
                                                                              localHostList,
                                                                              remoteSn,
                                                                              remoteLunId,
                                                                              remoteHostList,
                                                                              iniListRemote,
                                                                              iniListLocal,
                                                                              localHostDict,
                                                                              remoteHostDict)

        errMsgIniConfigListTemp = getErrMsgOfIniNotConsistenceShow(iniNotInLocal,
                                                                   iniNotInRemote,
                                                                   productVersion,
                                                                   localLunId,
                                                                   localHostList,
                                                                   remoteSn,
                                                                   remoteLunId,
                                                                   remoteHostList,
                                                                   iniListRemote,
                                                                   iniListLocal,
                                                                   localHostDict,
                                                                   remoteHostDict)
        if suggestionErrMsgTemp:
            ALL_CLI_RET_LIST.append(suggestionErrMsgTemp)

        if noPassErrMsgTemp:
            ALL_CLI_RET_LIST.append(noPassErrMsgTemp)
        if errMsgIniConfigListTemp:
            errMsgIniConfigList.extend(errMsgIniConfigListTemp)

        # 检查启动器配置信息是否一致
        if not check_ini_list:
            continue

        for wwn in check_ini_list:
            localHostId = initiatorDictLocal[wwn].get("hostId", '')
            remoteHostId = initiatorDictRemote[wwn].get("hostId", '')
            localOperSys = localHostDict.get(localHostId, {}).get("operatingSystem", '')
            remoteOperSys = remoteHostDict.get(remoteHostId, {}).get("operatingSystem", '')
            localRunningStatus = initiatorDictLocal.get(wwn, {}).get("runningStatus")
            remoteRunningStatus = initiatorDictRemote.get(wwn, {}).get("runningStatus")

            # 检查启动器配置是否一致。
            diffConfgList = checkInitiatorInfo(initiatorDictRemote, initiatorDictLocal, wwn, localHostId,
                                               remoteHostId)
            if diffConfgList:

                # 当HP-UX只有启动器运行状态不一致时才报建议优化。
                if localOperSys in SUGGESTION_HOST_LIST and localRunningStatus != remoteRunningStatus and len(
                        diffConfgList) == 1:
                    tmpMsg = common.getMsg(LANG, "hyMetro.consis.initiator.config.hpux",
                                           (localHostId, remoteSn, remoteHostId,))
                    if tmpMsg not in suggestionErrMsg:
                        suggestionErrMsg += tmpMsg
                else:
                    errMsgConfigList.extend(diffConfgList)

            LOGGER.logInfo("localRunningStatus:%s, remoteRunningStatus:%s" % (localRunningStatus,
                                                                              remoteRunningStatus))
            # 新增逻辑检查，检查是否两端启动器已离线，离线要报不通过。
            if localRunningStatus.lower() == "offline" and remoteRunningStatus.lower() == "offline":
                tmpMsg = common.getMsg(LANG, "hyMetro.consis.initiator.offline", (localHostId, remoteHostId, wwn,
                                                                                  remoteSn))
                if tmpMsg not in errMsgIniOfflineList:
                    errMsgIniOfflineList.append(tmpMsg)

            # 检查操作系统是否一致，如果一致检查vmware host_lun_id 是否一致。
            if localOperSys != remoteOperSys:
                tmpMsg = common.getMsg(LANG, "hyMetro.consis.hostOper", (localHostId,
                                                                         localOperSys,
                                                                         remoteHostId,
                                                                         remoteOperSys,
                                                                         remoteSn))
                if tmpMsg not in errMsgOper:
                    errMsgOper += tmpMsg

            # 如果是vmware主机， 同一个LUN两端的hostlunid必须一致。
            elif "vmware" in localOperSys.lower():
                hostLunIdLocal = localHostDict.get(localHostId, {}).get("hostLunInfo", {}).get(localLunId, {}).get(
                    "hostLunId")
                hostLunIdRemote = remoteHostDict.get(remoteHostId, {}).get("hostLunInfo", {}).get(remoteLunId, {}).get(
                    "hostLunId")
                if hostLunIdLocal != hostLunIdRemote:
                    tmpMsg = common.getMsg(LANG,
                                           "hyMetro.consis.hostlunid."
                                           "doradov6",
                                           (localHostId, remoteHostId,
                                            remoteSn))
                    tmpBriefMsg = common.getMsg(LANG,
                                                "hyMetro.consis.hostlunid."
                                                "brief",
                                                (localHostId, remoteHostId,
                                                 hostLunIdLocal,
                                                 hostLunIdRemote))
                    if tmpBriefMsg not in errMsgDifLunIdList:
                        errMsgDifLunIdList.append(tmpBriefMsg)
                    if tmpMsg not in errMsgHostLun:
                        errMsgHostLun += tmpMsg
    if errMsgIniConfigList:
        if isSuggestionProductVersion(productVersion):
            addMsg = common.getMsg(LANG, "please.contact.engineers.to.confirm.risk")
            suggestionErrMsg = common.getMsg(LANG, "hyMetro.consis.initiator.wwn.not.consistence.old.version.title",
                                             (addMsg, remoteSn, "".join(errMsgIniConfigList)))
        else:
            addMsg = ""
            errMsgInitiator = common.getMsg(LANG, "hyMetro.consis.initiator.wwn.not.consistence.old.version.title",
                                            (addMsg, remoteSn, "".join(errMsgIniConfigList)))

    if errMsgIniOfflineList:
        errMsgIniOffline = common.getMsg(LANG, "hyMetro.consis.initiator.offline.title",
                                         "".join(errMsgIniOfflineList))
    if error_ini_info:
        errMsgConfig += get_show_err_msg(localSn, remoteSn, error_ini_info)
    if errMsgConfigList:
        errMsgConfigList = list(set(errMsgConfigList))
        tmpMsg = common.getMsg(LANG, "hyMetro.consis.initiator.config",
                               (remoteSn, "".join(errMsgConfigList)))
        errMsgConfig += tmpMsg

    if unMappedLunListLocal:
        errMsgUnMapped += common.getMsg(LANG, "hyMetro.consis.unmapped.hypermetro.lun.local",
                                        sepereterA.join(unMappedLunListLocal))

    if unMappedLunListRemote:
        errMsgUnMapped += common.getMsg(LANG, "hyMetro.consis.unmapped.hypermetro.lun.remote", (remoteSn,
                                                                                                sepereterA.join(
                                                                                                    unMappedLunListRemote)))
    if errMsgDifLunIdList:
        errMsgDifLunId = common.getMsg(LANG, "hyMetro.consis.hostlunid.title",
                                       (localSn, remoteSn, "".join(errMsgDifLunIdList)))
        ALL_CLI_RET_LIST.append(errMsgDifLunId)
    return errMsgInitiator, suggestionErrMsg, errMsgOper, errMsgConfig, errMsgHostLun, errMsgIniOffline, errMsgUnMapped


def check_host_access_mode(error_ini_info, initiator_list, init_host_local,
                           ini_host_remote, local_host_dict, remote_host_dict):
    for ini in initiator_list:
        local_host_id = init_host_local.get(ini, {}).get("hostId")
        local_access_mode = local_host_dict.get(local_host_id, {}).get("accessModel", "")
        remote_host_id = ini_host_remote.get(ini, {}).get("hostId")
        remote_access_mode = remote_host_dict.get(remote_host_id, {}).get("accessModel", "")
        if local_access_mode != remote_access_mode:
            error_ini_info.add((local_host_id, local_access_mode, remote_host_id, remote_access_mode))


def get_show_err_msg(local_sn, remote_sn, error_info_set):
    if not error_info_set:
        return ""
    error_msg_list = [common.getMsg(LANG, "hyMetro.consis.acces.model.not.pass.title", (local_sn, remote_sn))]
    for error_info_tuble in  error_info_set:
        error_msg_list.append(common.getMsg(LANG, "hyMetro.consis.acces.model.not.pass", error_info_tuble))
    return "".join(error_msg_list)


def getErrMsgOfIniNotConsistence(iniNotInLocal, iniNotInRemote, productVersion, localLunId,
                                 localHostList, remoteSn, remoteLunId, remoteHostList,
                                 iniListRemote, iniListLocal,
                                 localHostDict, remoteHostDict):
    """
    获取启动器不一致的信息
    :param iniNotInLocal:
    :param iniNotInRemote:
    :param productVersion:
    :param localLunId:
    :param localHostList:
    :param remoteSn:
    :param remoteLunId:
    :param remoteHostList:
    :param iniListRemote:
    :param iniListLocal:
    :param localHostDict:
    :param remoteHostDict:
    :return:
    """
    suggestionErrMsg = ''
    noPassErrMsg = ''
    if iniNotInLocal or iniNotInRemote:
        tmpStr = "[ID %s" + sepereterD + "%s]"
        localErrHostStrList = []
        remoteErrHostStrList = []
        for hostId in localHostList:
            hostIniListLocal = localHostDict.get(hostId, {}).get("initiatorList", [])
            localErrHostStrList.append(tmpStr % (hostId, sepereterA.join(hostIniListLocal)))

        for hostId in remoteHostList:
            hostIniListRemote = remoteHostDict.get(hostId, {}).get("initiatorList", [])
            remoteErrHostStrList.append(tmpStr % (hostId, sepereterA.join(hostIniListRemote)))
        LOGGER.logInfo("localErrHostStrList:%s, remoteErrHostStrList:%s" % (localErrHostStrList,
                                                                            remoteErrHostStrList))
        if isSuggestionProductVersion(productVersion):
            addMsg = common.getMsg(LANG, "please.contact.engineers.to.confirm.risk")
            suggestionErrMsg = common.getMsg(LANG, "hyMetro.consis.initiator.wwn.not.consistence", (
                addMsg, remoteSn, localLunId, sepereterA.join(localHostList), sepereterA.join(localErrHostStrList),
                remoteLunId,
                sepereterA.join(remoteHostList), sepereterA.join(remoteErrHostStrList)))
        else:
            noPassErrMsg = common.getMsg(LANG, "hyMetro.consis.initiator.wwn.not.consistence.old.version", (
                remoteSn, localLunId, sepereterA.join(localHostList), sepereterA.join(localErrHostStrList), remoteLunId,
                sepereterA.join(remoteHostList), sepereterA.join(remoteErrHostStrList)))

    return suggestionErrMsg, noPassErrMsg


def getErrMsgOfIniNotConsistenceShow(iniNotInLocal, iniNotInRemote, productVersion, localLunId,
                                     localHostList, remoteSn, remoteLunId, remoteHostList,
                                     iniListRemote, iniListLocal,
                                     localHostDict, remoteHostDict):
    """
    获取启动器不一致的信息
    :param iniNotInLocal:
    :param iniNotInRemote:
    :param productVersion:
    :param localLunId:
    :param localHostList:
    :param remoteSn:
    :param remoteLunId:
    :param remoteHostList:
    :param iniListRemote:
    :param iniListLocal:
    :param localHostDict:
    :param remoteHostDict:
    :return:
    """
    errMsgList = []
    if iniNotInLocal:
        errMsgTempList = getErrorHostAndInitiatorInfo(iniNotInLocal, remoteHostDict)
        errMsgList.append(common.getMsg(LANG, "hyMetro.consis.initiator.wwn.not.consistence.old.version.local.configed",
                                        (remoteLunId, sepereterA.join(errMsgTempList), localLunId)))

    if iniNotInRemote:
        errMsgTempList = getErrorHostAndInitiatorInfo(iniNotInRemote, localHostDict)
        errMsgList.append(
            common.getMsg(LANG, "hyMetro.consis.initiator.wwn.not.consistence.old.version.remote.configed",
                          (localLunId, sepereterA.join(errMsgTempList), remoteLunId)))

    return errMsgList


def getErrorHostAndInitiatorInfo(iniList, HostListInfoDict):
    """
    获取报错的启动器和主机信息
    :param iniList:
    :param HostListInfoDict:
    :return:
    """
    errHost = {}
    errMsgList = []
    for ini in iniList:
        for hostId in HostListInfoDict:
            hostIniListLocal = HostListInfoDict.get(hostId, {}).get("initiatorList", [])
            if ini in hostIniListLocal:
                iniListTemp = errHost.get(hostId, [])
                iniListTemp.append(ini)
                errHost[hostId] = iniListTemp

    for hostId in errHost:
        iniListTemp = errHost.get(hostId, [])
        tmpStr = "[ID %s" + sepereterD + "%s]"
        errMsgList.append(tmpStr % (hostId, sepereterA.join(iniListTemp)))
    return errMsgList


def isSuggestionProductVersion(productVersion):
    """
    判断是否为新版本可能为直连组网
    :param productVersion:
    :return:
    """
    if (productVersion.startswith("V300R006") and
        productVersion > config.MAY_PARALLEL_NETWORKING_PRODUCTS_V3R6) or \
            ((productVersion.startswith("V300R001") or productVersion.startswith("V300R002")) and
             productVersion >= config.MAY_PARALLEL_NETWORKING_PRODUCTS_DORADO) or \
            (productVersion.startswith("V500R007") and
             productVersion >= config.MAY_PARALLEL_NETWORKING_PRODUCTS_V5R7):
        return True


def getLunMappedInitiator(hostDict, lunPairDict):
    """
    从主机中获取每个pair映射的启动器
    :param hostDict:
    :param lunPairDict:
    :return:
    """
    allLunInitiatorDict = {}
    for hostId in hostDict:
        hostLunDict = hostDict[hostId].get("hostLunInfo", {})
        hostIniList = hostDict[hostId].get("initiatorList", [])
        for lunId in hostLunDict:
            pairId = lunPairDict.get(lunId, '')
            tmpDict = allLunInitiatorDict.get(pairId, {})
            tmpHostList = tmpDict.get("hostList", [])
            tmpHostList.append(hostId)
            tmpIniList = tmpDict.get("iniList", [])
            tmpIniList.extend(hostIniList)

            allLunInitiatorDict[pairId] = {"lunId": lunId,
                                           "iniList": list(set(tmpIniList)),
                                           "hostList": list(set(tmpHostList))
                                           }
    return allLunInitiatorDict


def checkInitiatorInfo(remoteHostInfoDict, localHostInfoDict, iniWwn, localHostId, remoteHostId):
    """
    检查启动器配置信息是否一致
    :param remoteHostInfoDict:
    :param localHostInfoDict:
    :param iniWwn:
    :param localHostId:
    :param remoteHostId:
    :return:
    """
    diffConfgList = []
    keys = {"failoverMode": "Failover Mode", "multiPath": "Multipath Type",
            "specialModeType": "Special Mode Type",
            "runningStatus": "Running Status"}
    for key in keys:
        localKeyValue = localHostInfoDict.get(iniWwn, {}).get(key)
        remoteKeyValue = remoteHostInfoDict.get(iniWwn, {}).get(key)
        if localKeyValue != remoteKeyValue:
            errMsg = common.getMsg(LANG, "hyMetro.consis.local.and.remote.value",
                                   (keys[key], localHostId, localKeyValue, remoteHostId, remoteKeyValue))
            diffConfgList.append(errMsg)

    return diffConfgList
