# -*- coding: UTF-8 -*-
import re
import traceback

import cliUtil
import common
from common_utils import (
    get_err_msg,
    ana_form_by_space,
)

LANG = common.getLang(py_java_env)
LOGGER = common.getLogger(PY_LOGGER, __file__)
PY_JAVA_ENV = py_java_env
REG_RISK_STR = re.compile(
    r"Tier ID\s*:\s*(\d+)\s+DG ID\s*:\s*31\s+GROUP ID\s*:\s*\d+"
    r"\s*Disk Num\s*:\s*1\s+"
)


def execute(cli):
    """
    硬盘域DG检查
    :param cli:
    :return:
    """
    domain_dc_check = DomainDCCheck(cli, LANG, PY_JAVA_ENV, LOGGER)
    flag, msg = domain_dc_check.execute_check()
    return flag, "\n".join(domain_dc_check.all_ret_list), msg


class DomainDCCheck:
    def __init__(self, cli, lang, env, logger):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = []

    def execute_check(self):
        conn_cli = None
        try:
            if not self.is_risk_version():
                self.all_ret_list.append(
                    "Product Version:{}".format(
                        str(self.env.get("devInfo").getProductVersion())
                    ),
                )
                return cliUtil.RESULT_NOSUPPORT, ""
            # 如果是18000执行debug命令获取阵列链接
            (
                new_conn_flag,
                conn_cli,
                err_msg,
            ) = common.createDeviceCliContFor18000(
                self.cli, self.env, self.logger, self.lang
            )

            if new_conn_flag is not True:
                return cliUtil.RESULT_NOCHECK, err_msg

            err_msg_list = []
            check_pool_dg(
                conn_cli,
                self.env,
                self.logger,
                self.lang,
                "",
                self.all_ret_list,
                err_msg_list,
            )
            if err_msg_list:
                return False, "".join(err_msg_list)

            return True, ""
        except common.UnCheckException as e:
            self.logger.logError(str(e))
            return cliUtil.RESULT_NOCHECK, e.errorMsg
        except Exception:
            self.logger.logError(str(traceback.format_exc()))
            return (
                cliUtil.RESULT_NOCHECK,
                common.getMsg(self.lang, "query.result.abnormal"),
            )
        finally:
            if conn_cli and conn_cli is not self.cli:
                common.closeConnection(conn_cli, self.env, self.logger)

            ret = cliUtil.enterCliModeFromSomeModel(self.cli, self.lang)
            self.logger.logInfo(
                "enter cli mode from some model ret is {}".format(str(ret))
            )
            # 退出失败后为不影响后续检查项重新连接cli
            if not ret[0]:
                common.reConnectionCli(self.cli, self.logger)

    def is_risk_version(self):
        """
        V500R007C00SPC100及之后版本、V300R006C10SPC100及之后版本
        :return:
        """
        p_version = str(self.env.get("devInfo").getProductVersion())
        return any(
            (
                (
                    p_version.startswith("V500R007")
                    and p_version >= "V500R007C00SPC100"
                ),
                (
                    p_version.startswith("V300R006")
                    and p_version >= "V300R006C10SPC100"
                ),
            ),
        )


@common.checkAllEngineInClusterWarp
def check_pool_dg(
    cli, env, logger, lang, heart_beat_cli_ret, all_cli_ret_list, err_msg_list,
):
    """
    步骤4的diskId对应步骤5的LDID，步骤6的ID对应步骤5的SDID
    :param cli: ssh连接
    :param env: 上下文
    :param logger: 日志
    :param lang: 语言
    :param heart_beat_cli_ret: 心跳回文
    :param all_cli_ret_list: 回文列表
    :param err_msg_list: 错误消息
    :return: 是否继续
    """
    all_cli_ret_list.append(heart_beat_cli_ret)
    flag, ret, msg = cliUtil.enterDeveloperMode(cli, lang)
    all_cli_ret_list.append(ret)
    if flag is not True:
        raise common.UnCheckException(msg, ret)

    (
        flag,
        msg,
        ret,
        node_id,
        engine_nodes,
        engine_id,
        node_num,
    ) = common.getEngineCtrlNodeInfo(cli, logger, lang)

    res_list = get_domain_work_ctrl(cli, lang, all_cli_ret_list, logger)
    logger.logInfo("work in this node res_list:{}".format(res_list))
    if not res_list:
        return common.CHECK_FLAG_CONTINUE

    check_dg(
        res_list, cli, lang, all_cli_ret_list, node_id, err_msg_list, logger
    )


def check_dg(
    pool_list, cli, lang, all_ret_list, node_id, err_msg_list, logger
):
    """
    检查硬盘域的DG信息
    :param pool_list: 硬盘域信息
    :param cli: ssh连接
    :param lang: 语言
    :param all_ret_list: 回文列表
    :param node_id: 节点
    :param err_msg_list: 错误提示
    :return:
    """
    cmd_str = "spa showpooldg -p {}"
    for res in pool_list:
        pool_id = res.get("poolid")
        cmd = cmd_str.format(pool_id)
        flag, ret, msg = cliUtil.excuteCmdInDebugModel(cli, cmd, lang)
        all_ret_list.append(ret)
        if flag is not True:
            raise common.UnCheckException(msg, ret)
        tmp_ret_str = "  ".join(ret.splitlines())
        reg_ret = REG_RISK_STR.findall(tmp_ret_str)
        if not reg_ret:
            continue
        logger.logInfo("reg res is :{}".format(reg_ret))
        get_msg(
            lang,
            err_msg_list,
            node_id,
            list(filter(lambda tier: reg_ret.count(tier) == 1, set(reg_ret))),
            pool_id,
        )


def get_msg(lang, err_msg_list, node_id, tier_list, pool_id):
    """
    组装错误消息
    :param lang: 语言
    :param err_msg_list: 错误消息
    :param node_id: 节点ID
    :param tier_list: 风险Tier
    :param pool_id: 存储池ID
    :return:
    """
    for tier in tier_list:
        err_msg_list.append(
            get_err_msg(
                lang,
                "disk.domain.check.disk.group.not.pass",
                (node_id, tier, pool_id),
            )
        )


def get_domain_work_ctrl(cli, lang, all_cli_ret_list, logger):
    """
    查询工作于本控制器的硬盘域ID
    :param cli: ssh连接
    :param lang: 语言
    :param all_cli_ret_list: 回文
    :param logger: 日志
    :return: 工作于本控制器的硬盘域信息
    """
    cmd = "pmgr showpool"
    flag, ret, msg = cliUtil.excuteCmdInDebugModel(cli, cmd, lang)
    all_cli_ret_list.append(ret)
    if flag is not True:
        raise common.UnCheckException(msg, ret)
    is_work_node = "Isworknode"
    pool_id_str = "poolid"
    ret_list = ana_form_by_space(
        ret, [is_work_node, pool_id_str], logger, split_key="|"
    )
    return list(
        filter(
            lambda ret_dict: ret_dict.get(is_work_node, "") == "YES", ret_list
        )
    )
