# -*- coding: UTF-8 -*-
import cliUtil
import common
import traceback
import funcUtil
from cliUtil import get_lun_id_list_with_cache
import common_utils
import common_cache

from cbb.frame.base.config import (
    LUN_CACHE_MIRROR_PATCH_CHECK_LATER_VERSION
)
from common_utils import is_support_read_only_user_enter_debug

# noinspection PyUnresolvedReferences
LANG = common.getLang(py_java_env)
# noinspection PyUnresolvedReferences
LOGGER = common.getLogger(PY_LOGGER, __file__)
# noinspection PyUnresolvedReferences
PY_JAVA_ENV = py_java_env
CMD_EXE_ERR_MSG = common.getMsg(LANG, 'system.status.abnormal')

VOLUMETOLUNDICT = {}

@funcUtil.countTime(LOGGER)
def execute(cli):
    isCmdSucc, isSwitchOn = funcUtil.isDeveloperSwitchOn(cli, LOGGER)
    if not isCmdSucc:
        return cliUtil.RESULT_NOCHECK, '', CMD_EXE_ERR_MSG

    try:
        return doCheck(cli)
    except:
        LOGGER.logError(str(traceback.format_exc()))
        return cliUtil.RESULT_NOCHECK, '', common.getMsg(LANG, "query.result.abnormal")
    finally:
        if not isSwitchOn:
            funcUtil.closeDeveloperSwitch(cli, LOGGER)


@funcUtil.restoreUserView
def doCheck(cli):
    """

    :param cli:
    :return:
    """
    allCliRet = ""
    allErrMsg = ""

    (
        flag, p_version, p_patch, ret, msg
    ) = common_cache.get_version_and_patch_cache(PY_JAVA_ENV, cli, LOGGER)
    allCliRet = joinLines(allCliRet, ret)

    if flag is not True:
        allErrMsg = joinLines(allErrMsg, msg)
        return (cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg)

    if need_install_patch(p_version, p_patch):
        target_patch = LUN_CACHE_MIRROR_PATCH_CHECK_LATER_VERSION.get(
            p_version
        )
        return False, allCliRet, common.getMsg(
            LANG, "lun.no.mirror.no.patch.not.pass", (
                p_version, p_patch, target_patch
            )
        )

    isQrySucc, ctrlIdList, errMsg, cliRet = cliUtil.getControllerIdListWithRet(cli, LANG)
    if isQrySucc != True:
        LOGGER.logError("Query controlled ID list failed, check terminated!")
        return cliUtil.RESULT_NOCHECK, cliRet, errMsg

    allCliRet = joinLines(allCliRet, cliRet)
    controllerNum = len(ctrlIdList)
    # 单控不存在lun镜像问题，检查通过
    if controllerNum == 1:
        LOGGER.logWarning('Single controller running, check PASSED!')
        return True, allCliRet, allErrMsg

    isQrySucc, lvlOrRet, errMsg = cliUtil.getUserPrivilege(cli, LANG)
    if not isQrySucc:
        allCliRet = joinLines(allCliRet, lvlOrRet)
        return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
    if lvlOrRet.lower() != 'super_admin' and not \
            is_support_read_only_user_enter_debug(p_version, p_patch):
        errMsg = common.getMsg(LANG, "loginUser.name.level.must.be.super.admin")
        return cliUtil.RESULT_NOCHECK, allCliRet, errMsg

    # >>获取LUN ID List
    flag, cliRet, errMsg, lunIdList = get_lun_id_list_with_cache(cli,
                                                                 LANG,
                                                                 PY_JAVA_ENV,
                                                                 LOGGER)
    allCliRet = joinLines(allCliRet, cliRet)
    if flag != True:
        LOGGER.logError('Query LUN ID list failed, check terminated!')
        return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg

    lunNum = len(lunIdList)
    if lunNum == 0:
        LOGGER.logInfo('No lun configured,check passed!')
        return True, allCliRet, allErrMsg

    # >>获取快照信息
    isQrySucc, cliRet, errMsg, snapshotIdList = getSnapshotIdList(cli)
    allCliRet = joinLines(allCliRet, cliRet)
    if not isQrySucc:
        allErrMsg = joinLines(allErrMsg, errMsg)
        return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg

    if common.is18000(PY_JAVA_ENV, cli):
        currentContrIp = PY_JAVA_ENV.get("devInfo").getIp()
        isCliBuildSucc, cliRet, errMsg, cliConToCtrl, connIp = \
            common.getConnectionByControllerIp(PY_JAVA_ENV, cli, LOGGER)
        allCliRet = joinLines(allCliRet, cliRet)
        if not isCliBuildSucc:
            allErrMsg = joinLines(allErrMsg, common.getMsg(LANG, "build.ssh.connection.to.controller.failed"))
            return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
        else:
            chkRet = chkAllNotMirrorLuns(cliConToCtrl, allCliRet, allErrMsg, lunIdList, snapshotIdList)
            if currentContrIp != connIp:
                common.closeSSH(cliConToCtrl)
            return chkRet

    return chkAllNotMirrorLuns(cli, allCliRet, allErrMsg, lunIdList, snapshotIdList)


def need_install_patch(p_version, p_patch):
    """
    安装对应的补丁版本后再执行巡检操作
    :param p_version: 版本
    :param p_patch: 补丁
    :return: True: 需要安装。 False: 继续检查
    """
    return all(
        [
            LUN_CACHE_MIRROR_PATCH_CHECK_LATER_VERSION.get(p_version),
            not common_utils.is_special_patch(p_patch),
            p_patch
            < LUN_CACHE_MIRROR_PATCH_CHECK_LATER_VERSION.get(p_version),
        ]
    )


def chkAllNotMirrorLuns(cliConToCtrl, allCliRet, allErrMsg, lunIdList, snapshotIdList):
    # >>检查登录控制器不镜像的LUN和快照
    isQrySucc, qryRet, errMsg, notMirrorObjIdList = checkLoginCtrlNotMirrorObj(cliConToCtrl)
    allCliRet = joinLines(allCliRet, qryRet)
    if not isQrySucc:
        allErrMsg = joinLines(allErrMsg, errMsg)

    # >>检查归属其他控制器的单控工作模式的LUN
    priKey = PY_JAVA_ENV.get("devInfo").getPriKey()
    if priKey:
        # 不支持PublicKey鉴权方式进行心跳控制器
        errMsg = common.getMsg(LANG, "no.support.publickey.forensics")
        return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)

    singleWorkModeObjIdList, heartbeatFailNids, checkFailNidList, chkRet = checkOtherNid(cliConToCtrl)
    allCliRet = joinLines(allCliRet, chkRet)
    notMirrorErrMsg = getErrMsg(notMirrorObjIdList, lunIdList, snapshotIdList, singleWorkModeObjIdList)
    if heartbeatFailNids:
        allErrMsg = joinLines(allErrMsg,
                              common.getMsg(LANG, "heart.beat.to.node.failed",
                                            ",".join(heartbeatFailNids)))
    if checkFailNidList:
        allErrMsg = joinLines(allErrMsg, common.getMsg(LANG, "heart.beat.to.node.to.check.failed",
                                                       ",".join(checkFailNidList)))
    if notMirrorErrMsg:
        allErrMsg = joinLines(allErrMsg, notMirrorErrMsg)
        return False, allCliRet, allErrMsg

    if heartbeatFailNids or checkFailNidList:
        return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg

    return True, allCliRet, ''


def getEngineCtrlNumDict(contrIdList):
    """
    @summary: 获取引擎下的控制器数量
    """
    engNumDict = {}

    for contrId in contrIdList:
        engineId = contrId[:1]
        engNumDict[engineId] = engNumDict.get(engineId, 0) + 1

    return engNumDict


def getAllEngineReachableIP(mgtPortDictList):
    """
    @summary: Get reachable IP address of every engine.
    @param mgtPortDictList: Management port IP address information dictionary.
    @return: {engineId: reachable_IP_address}
    """
    engineIpDict = {}
    for managementPortInfoDict in mgtPortDictList:
        ip = managementPortInfoDict.get("IPv4 Address")
        portRunningStatus = managementPortInfoDict.get("Running Status")
        if portRunningStatus == "Link Up" and ip not in ["", "--"]:
            managementPortId = managementPortInfoDict.get("ID")
            engineId = managementPortId.split(".")[0][-1:]
            if (engineId not in engineIpDict) and ("DSW" not in managementPortId):
                engineIpDict.setdefault(engineId, ip)

    return engineIpDict


def getSnapshotIdList(cli):
    """
    @summary: 获取所有的快照id
    @param cli: CLI connection.
    @return: (isQrySucc, cliRet, errMsg, snapshotIdList)
    """
    snapshotIdList = []

    cmd = "show snapshot general |filterColumn include columnList=ID"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, LANG)
    if flag == False:
        LOGGER.logInfo("Failed to get information about snapshot.")
        return False, cliRet, errMsg, snapshotIdList
    elif flag == cliUtil.RESULT_NOSUPPORT or flag == cliUtil.RESULT_NOCHECK:
        return True, cliRet, errMsg, snapshotIdList
    elif cliUtil.queryResultWithNoRecord(cliRet):
        return True, cliRet, errMsg, []

    snapshotInfoDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
    if len(snapshotInfoDictList) == 0:
        LOGGER.logWarning("Snapshot number is 0, CLI result parsing may have failed.")
        return cliUtil.RESULT_NOCHECK, cliRet, errMsg, snapshotIdList  # 修改备注：解析回显失败导致列表长度为0

    for snapshotInfoDict in snapshotInfoDictList:
        snapshotId = snapshotInfoDict.get("ID")
        snapshotIdList.append(snapshotId)

    LOGGER.logInfo('Snapshot number is:' + unicode(len(snapshotIdList)))

    return True, cliRet, errMsg, snapshotIdList


def transform_volume_to_lun_or_snapshot(cli, volume_id):
    """
    将卷ID转换为LUN ID/快照ID。
    当需要使用debug下查询到的卷ID查找cli下命令使用的LUN ID对象时，
    需要使用命令 volume showctrl [volume_id]通过卷ID查找LUN ID，再进行后续操作。
    :param cli: ssh连接
    :param volume_id: 卷ID
    :return: lun_id
    """
    global VOLUMETOLUNDICT
    lun_id = VOLUMETOLUNDICT.get(volume_id, None)
    if lun_id is not None:
        LOGGER.logInfo('The lunId is {} of volume_id({}).'.format(
            lun_id, volume_id)
        )
        return lun_id, ''

    cmd = "volume showctrl {}".format(volume_id)
    flag, ret, _ = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
    if flag is not True:
        LOGGER.logSysAbnormal()
        return "", ret

    for line in ret.splitlines():
        if ":" not in line:
            continue

        fileds = line.split(":")
        if len(fileds) <= 1:
            continue

        key = fileds[0]
        if key.strip().lower() == "lun id":
            lun_id = str(int(fileds[1].strip(), 16))
            VOLUMETOLUNDICT[volume_id] = lun_id
            return lun_id, ret

    return "", ret


# noinspection PyBroadException,PyBroadException
def checkLoginCtrlNotMirrorObj(cli):
    """

    :param cli:
    :return:isQrySucc, qryRet, errMsg, notMirrorObjIdList
    """
    cmd = "cache show obj v switch nomirrorlun"
    flag, cliRet, errMsg = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
    if flag != True:
        LOGGER.logSysAbnormal()
        return cliUtil.RESULT_NOCHECK, cliRet, errMsg, []  # 修改备注：excuteCmdInDebugModel执行命令失败是未检查

    ret_list = [cliRet]
    notMirrorObjIdList = []
    for line in cliRet.splitlines():
        fields = line.split()
        if len(fields) <= 1:
            continue
        if fields[1].strip().isdigit():
            objId = fields[1].strip()
            lun_id, ret = transform_volume_to_lun_or_snapshot(cli, objId)
            ret_list.append(ret)
            notMirrorObjIdList.append(lun_id)

    return True, "\n".join(ret_list), '', notMirrorObjIdList


def getOtherNormalNidList(cli):
    cmd = "sys showcls"
    flag, cliRet, errMsg = cliUtil.excuteCmdInDebugModel(cli, cmd, LANG)
    if flag != True:
        LOGGER.logSysAbnormal()
        return cliUtil.RESULT_NOCHECK, cliRet, errMsg, [], 0  # 修改备注：excuteCmdInDebugModel执行命令失败是未检查

    otherNidList = []
    localNid = None
    nodeNum = 0
    for line in cliRet.splitlines():
        if 'local node id' in line:
            localNid = line.split(':')[1].strip()
            continue
        if 'node cfg' in line:
            nodeNum = int(line.split(':')[1].strip())
            continue

        fields = line.split()
        if fields[0].isdigit() and len(fields) >= 3 and fields[2] == 'normal' and localNid != fields[0]:
            otherNidList.append(fields[0])
            continue

    return True, cliRet, '', otherNidList, nodeNum


def checkOtherNid(cli):
    """
    检查所有引擎，除当前登录的节点的cache show obj v switch singlectrl，获取归属控制器为当前节点且工作模式为单控的普通lun ID和快照lun ID。
    :param cli:
    :return:
    """
    heartbeatFailNids = []
    cliRetList = []
    checkFailNidList = []
    notMirrorObjIdList = []

    flag, errMsg, cliRet, currentCtrlNodeId, engineCtrlNodeMappingDict, currentEngine, nodeNum = common.getEngineCtrlNodeInfo(
        cli, LOGGER, LANG)
    cliRetList.append(cliRet)

    # 当前引擎，不检查当前登录节点。
    curEngineNodeList = engineCtrlNodeMappingDict.get(currentEngine)
    curEngineNodeList.remove(currentCtrlNodeId)
    cliRet = checkCurrentEngineNodes(curEngineNodeList, cli, heartbeatFailNids, checkFailNidList, notMirrorObjIdList)
    cliRetList.append(cliRet)

    # 其他引擎
    for engineId in engineCtrlNodeMappingDict:
        if engineId == currentEngine:
            continue
        curEngineNodeList = engineCtrlNodeMappingDict.get(engineId)
        flag, cliRet, errMsg, jumpConnCli, closeHeartBeatTimes = common.enginesJump(cli, engineId, currentEngine, currentCtrlNodeId, nodeNum, PY_JAVA_ENV, LOGGER, LANG, engineCtrlNodeMappingDict)
        cliRet = checkCurrentEngineNodes(curEngineNodeList, jumpConnCli, heartbeatFailNids, checkFailNidList,
                                         notMirrorObjIdList)
        cliRetList.append(cliRet)
        common.enginesBack(jumpConnCli, PY_JAVA_ENV, LOGGER, LANG, closeHeartBeatTimes)

    return notMirrorObjIdList, heartbeatFailNids, checkFailNidList, '\n'.join(cliRetList)


def checkCurrentEngineNodes(curEngineNodeList, cli, heartbeatFailNids, checkFailNidList, notMirrorObjIdList):
    """
    当前引擎下检查指定的节点。
    :param curEngineNodeList:
    :param cli:
    :param heartbeatFailNids:
    :param checkFailNidList:
    :param notMirrorObjIdList:
    :return:
    """
    allCliRet = ''
    # 当前引擎
    for contrNodeId in curEngineNodeList:
        __, __, __, cur_node_id, __, __, __ = common.getEngineCtrlNodeInfo(
            cli, LOGGER, LANG
        )
        try:
            if cur_node_id != contrNodeId:
                flag, cliRet, errMsg = common.heartBeatToOtherCtrl(
                    cli, contrNodeId, PY_JAVA_ENV, LOGGER, LANG
                )
                allCliRet = joinLines(allCliRet, cliRet)
                if flag is not True:
                    heartbeatFailNids.append(contrNodeId)
            cliRet = checkSingleCtrlInCurrentNode(
                cli, contrNodeId, checkFailNidList, notMirrorObjIdList
            )
            allCliRet = joinLines(allCliRet, cliRet)
        except:
            heartbeatFailNids.append(contrNodeId)
            LOGGER.logError(traceback.format_exc())
        finally:
            if cur_node_id != contrNodeId:
                cliUtil.exitHeartbeatCli(cli, LANG)
                LOGGER.logInfo("exit heart beat success.")
            # 退出到cli模式
            ret = cliUtil.enterCliModeFromSomeModel(cli, LANG)
            LOGGER.logInfo("enter cli mode from some model ret is %s" % str(ret))

    return allCliRet


def checkSingleCtrlInCurrentNode(cli, nid, checkFailNidList, notMirrorObjIdList):
    """
    当前节点检查cache show obj v switch singlectrl，获取归属控制器为当前节点且工作模式为单控的普通lun ID和快照lun ID。
    :param cli:
    :param nid:
    :param checkFailNidList:
    :param notMirrorObjIdList:
    :return:
    """
    ret_list = []
    try:
        dbgCmd = 'cache show obj v switch singlectrl'
        dbgCmdSucc, cliRet, dbgErrMsg = cliUtil.excuteCmdInDebugModel(cli, dbgCmd, LANG)
        if dbgCmdSucc != True:
            checkFailNidList.append(nid)
        ret_list.append(cliRet)
        for line in cliRet.splitlines():
            fields = line.split()
            if len(fields) <= 1:
                continue
            if fields[1].isdigit():
                obj_id = fields[1]
                # volume showctrl [volume ID]，获取到对应的lun ID和快照ID。
                lun_id, ret = transform_volume_to_lun_or_snapshot(cli, obj_id)
                if ret:
                    ret_list.append(ret)
                notMirrorObjIdList.append(lun_id)
    except:
        LOGGER.logError(traceback.format_exc())
        checkFailNidList.append(nid)
    return "\n".join(ret_list)


def sortByInt(sortList):
    """
    @summary: 对元素类型为int的列表排序
    """
    try:
        sortList.sort(cmp=None, key=int, reverse=False)
    except:
        LOGGER.logError("list sort by int failed, maybe the list item is not int type.")
        LOGGER.logError(str(traceback.format_exc()))


def getErrMsg(notMirrorObjIdList, lunIdList, snapshotIdList, allSingleWorkingModeIdList):
    notMirrorLunIdList = []
    notMirrorSnapshotIdList = []
    for objId in notMirrorObjIdList:
        if objId in lunIdList:
            notMirrorLunIdList.append(objId)
        if objId in snapshotIdList:
            notMirrorSnapshotIdList.append(objId)

    for objId in allSingleWorkingModeIdList:
        if objId in lunIdList:
            notMirrorLunIdList.append(objId)
        if objId in snapshotIdList:
            notMirrorSnapshotIdList.append(objId)

    if not notMirrorLunIdList and not notMirrorSnapshotIdList:
        return ''

    errMsgList = []
    if notMirrorLunIdList:
        sortByInt(notMirrorLunIdList)
        lunErrMsg = common.getMsg(LANG, "not.mirror.lun.list", ",".join(notMirrorLunIdList))
        errMsgList.append(lunErrMsg)

    if notMirrorSnapshotIdList:
        sortByInt(notMirrorSnapshotIdList)
        snapshotErrMsg = common.getMsg(LANG, "not.mirror.snapshot.list", ",".join(notMirrorSnapshotIdList))
        errMsgList.append(snapshotErrMsg)

    return '\n'.join(errMsgList)


def joinLines(oldLines, newLines):
    if not oldLines:
        return newLines
    return '\n'.join([oldLines, newLines]) if newLines else oldLines


