# -*- coding: UTF-8 -*-

import cliUtil
import common
import common_cache
import common_utils
import risk_version_config

from common import UnCheckException

PY_JAVA_ENV = py_java_env
LANG = common.getLang(PY_JAVA_ENV)
LOGGER = common.getLogger(PY_LOGGER, __file__)


def execute(cli):
    """
    逻辑端口回漂和CIFS业务检查
    """
    flag, ret_list, err_msg = check_cifs(cli)
    return flag, "\n".join(ret_list), err_msg


def check_cifs(cli):
    """
    检查逻辑端口回漂和CIFS业务流程
    :param cli: ssh连接
    :return: 检查结果
    """
    all_cli_ret = []
    try:
        if not is_risk_version(cli, all_cli_ret):
            return True, all_cli_ret, ""

        flag, ret, err_msg = cliUtil.enterDeveloperMode(cli, LANG)
        all_cli_ret.append(ret)
        if flag is not True:
            return cliUtil.RESULT_NOCHECK, all_cli_ret, err_msg

        share_cifs_ids, ret = common_cache.get_share_cifs_cache(
            PY_JAVA_ENV, cli, LOGGER
        )
        all_cli_ret.append(ret)
        if not share_cifs_ids:
            return True, all_cli_ret, ""

        cifs_detail_list, ret_list = common_cache.get_share_cifs_cache_detail(
            PY_JAVA_ENV, cli, LOGGER, share_cifs_ids
        )
        all_cli_ret.extend(ret_list)

        error_cifs = check_cifs_enabled(cifs_detail_list)
        if error_cifs:
            err_msg = common_utils.get_err_msg(
                LANG, "software.cifs.continuity.not.pass", ",".join(error_cifs)
            )
            return False, all_cli_ret, err_msg
        return True, all_cli_ret, ""
    except UnCheckException as unCheckException:
        all_cli_ret.append(unCheckException.cliRet),
        return (
            cliUtil.RESULT_NOCHECK,
            all_cli_ret,
            unCheckException.errorMsg,
        )
    finally:
        cliUtil.enterCliModeFromSomeModel(cli, LANG)


def is_risk_version(cli, ret_list):
    """
    检查是否风险版本
    :param cli: ssh连接
    :param ret_list: 回文列表
    :return: True 是风险版本，False 非风险版本
    """
    (
        flag,
        p_version,
        p_patch,
        ret,
        err_msg,
    ) = common_cache.get_version_and_patch_cache(PY_JAVA_ENV, cli, LOGGER)
    ret_list.append(ret)
    if flag is not True:
        raise UnCheckException(err_msg, "")

    # Kunpeng版本独立比较有风险
    p_version_mem = str(PY_JAVA_ENV.get("devInfo").getProductVersion())
    if p_version_mem == risk_version_config.SHARE_CIFS_RISK_VERSION_KUNPENG:
        return True

    # 其他风险版本判断
    if p_version not in risk_version_config.SHARE_CIFS_RISK_VERSION:
        return False

    if common_utils.is_special_patch(p_patch):
        return True

    # 无补丁场景
    if not risk_version_config.SHARE_CIFS_RISK_VERSION_DICT.get(p_version):
        return True

    # 如果已打对应补丁则无风险
    if p_patch >= risk_version_config.SHARE_CIFS_RISK_VERSION_DICT.get(
        p_version
    ):
        return False

    return True


def check_cifs_enabled(cifs_detail_list):
    """
    Continue Available Enabled字段为Disable，则检查通过
    :param cifs_detail_list: cifs的详细信息列表
    :return: 不为disable的 share id 列表
    """
    error_cifs = []
    for cifs_detail in cifs_detail_list:
        if cifs_detail.get("Continue Available Enabled") != "Disable":
            error_cifs.append(cifs_detail.get("Share ID"))

    return error_cifs
