# coding:utf-8
"""
@version: Toolkit V200R006C00
@time: 2019/12/11
@file: common_cache.py
@function:
@modify:
"""
import codecs
import re
import os
import traceback

import cliUtil
import common


import com.huawei.ism.tool.obase.connection.SftpTransporter as SftpCls

from event_log_util import collect_running_data
from cbb.frame.cli.cli_with_cache import execute_cmd_in_cli_mode_with_cache
from cbb.frame.cli import running_data_analyze_util as ana_util
from cbb.frame.base import product
from cbb.frame.rest import restUtil

# 限制回文字符长度5M
LIMIT_SIZE = 1024*1024*5

# 如果是list，限制长度超过2K，对应一般命令最小内存2M左右
LIMIT_LIST_SIZE = 2000

CMD_FS_DOMAIN = "show fs_hyper_metro_domain general"
CMD_SAN_PAIR = "show hyper_metro_pair general |filterRow column=Type predict=equal_to value=LUN|filterColumn include columnList=Local\sID,Remote\sID,ID,Domain\sID"
CMD_NAS_PAIR = "show hyper_metro_pair general |filterRow column=Type predict=equal_to value=FS|filterColumn include columnList=Local\sID,Remote\sID,ID,Domain\sID"
CMD_SAN_PAIR_CG = "show hyper_metro_pair general |filterRow column=Type predict=equal_to value=LUN|filterColumn" \
                  " include columnList=Local\sID,Remote\sID,ID,Domain\sID,Consistency\sGroup\sID"

MEMBER_DISK_TITLE_STR = "Member Disk Info"
FREE_DISK_TITLE_STR = "Free Disk And Cache Disk Info"
MEMBER_DISK_REGX = re.compile("{}\s-+".format(MEMBER_DISK_TITLE_STR))
FREE_DISK_REGX = re.compile("{}\s-+".format(FREE_DISK_TITLE_STR))
DISK_INFO_FROM_CONFIG = "disk info from config {}"
REMOTE_REPLICATION_INFO_FROM_CONFIG = "remote replication info from config {}"

def parse_upgrade_package_cache(env, cli, logger):
    """
    执行show upgrade packge命令，将Software Version与HotPatch Version的
    回显存放到同一个字典列表中，并增加Version Type
    键来区分Version类型,并返回CLI回显
    :param env: 上下文
    :param cli: cli对象
    :param logger: 日志打印
    :return:
    """
    lang = common.getLang(env)
    cmd = "show upgrade package"
    p_version_list = []
    p_patch_list = []
    flag, cli_ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                            logger)
    if flag is not True:
        return ((flag, cli_ret, msg), p_version_list, p_patch_list)

    version_index = cli_ret.find("Software Version")
    patch_index = cli_ret.find("HotPatch Version")
    p_version_list = cliUtil.getHorizontalCliRet(
        cli_ret[version_index:patch_index])
    p_patch_list = cliUtil.getHorizontalCliRet(
        cli_ret[patch_index:])

    if len(p_version_list) == 0 or len(p_patch_list) == 0:
        msg += common.getMsg(lang, "cannot.get.upgrade.package.info")
        return ((False, cli_ret, msg), p_version_list, p_patch_list)

    for softwareVersion in p_version_list:
        if softwareVersion["Current Version"] == "--":
            flag = False
            msg += common.getMsg(lang, "cannot.get.contrller.version.info",
                                 softwareVersion.get("Name"))

    return ((flag, cli_ret, msg), p_version_list, p_patch_list)


def get_version_and_patch_cache(env, cli, logger):
    """
    获取当前SPC版本信息,补丁版本信息
    :param env: 上下文py_java_env
    :param cli: 连接
    :param logger: 日志打印
    :return: flag, p_version, p_patch, ret, msg
            flag: 命令执行是否成功
            p_version: 软件版本号
            p_patch：补丁版本
            ret: cli回文
            msg: 失败时错误消息
    """
    lang = common.getLang(env)
    (flag, ret, msg), ver_list, patch_list = \
        parse_upgrade_package_cache(env, cli, logger)
    if flag is not True:
        logger.logInfo(
            "get fail! Version:%s, Patch%s" % (ver_list, patch_list))
        return flag, "", "", ret, msg

    flag, p_version, msg = common.getCurrentVersion(ver_list, lang)
    if flag is not True:
        logger.logInfo("get version error. errMsg: %s" % msg)
        return flag, "", "", ret, msg

    flag, p_patch, msg = common.getHotPatchVersion(patch_list, lang)
    if flag is not True:
        logger.logInfo("get Patch Version error. errMsg: %s" % msg)
        return flag, "", "", ret, msg

    return True, p_version, p_patch, ret, ""


def get_product_version_from_system(env, cli, logger):
    """
    Kunpeng系列获取版本信息需要用show system general获取
    :param env: 上下文py_java_env
    :param cli: 连接
    :param logger: 日志打印
    :return: flag, p_version, p_patch, ret, msg
            flag: 命令执行是否成功
            p_version: 软件版本号
            p_patch：补丁版本
            ret: cli回文
            msg: 失败时错误消息
    """
    cmd = "show system general"
    flag, cli_ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                            logger)
    if flag is not True:
        raise common.UnCheckException(msg, cli_ret)

    p_version = ''
    record_list = cliUtil.getVerticalCliRet(cli_ret)
    for record in record_list:
        p_version = record.get("Product Version", '')
    return p_version, cli_ret


def get_ctrl_ids_cache(env, cli, logger):
    """
    获取控制器ID信息
    :param env: 上下文py_java_env
    :param cli: 连接
    :param logger: 日志打印
    :return:
    """
    cmd = "show controller general|filterColumn include " \
          "columnList=Controller"
    flag, cli_ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                            logger)
    if flag is not True:
        raise common.UnCheckException(msg, cli_ret)

    ctrl_id_list = []
    record_list = cliUtil.getVerticalCliRet(cli_ret)
    for record in record_list:
        ctrl_id_list.append(record.get("Controller", ''))
    return ctrl_id_list, cli_ret


def get_ctrl_ip_list_cache(cli, env, logger):
    """
    @summary: 获取引擎对应下的控制器IP
    :param cli: 连接
    :param env: 上下文py_java_env
    :param logger: 日志打印
    :return: flag, cli_ret, msg, ip_list_dict
            flag: 命令执行是否成功
            cli_ret: cli回文
            msg: 失败时错误消息
            ip_list_dict: 每个引擎的ip列表
    """
    lang = common.getLang(env)
    ip_list_dict = {}

    cmd = "show upgrade package"
    flag, cli_ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)

    begin_index = cli_ret.find("Software Version")
    end_index = cli_ret.find("HotPatch Version")

    info_dict_list = []
    if begin_index != -1 and end_index != -1:
        info_dict_list = cliUtil.getHorizontalCliRet(
            cli_ret[begin_index: end_index])

    for info_dict in info_dict_list:
        ip = info_dict.get("IP", "")
        ctrl_name = info_dict.get("Name", "")
        if ip != "" and ctrl_name != "":
            engine = ctrl_name[:1]
            ipList = ip_list_dict.get(engine, [])
            ipList.append(ip)
            ip_list_dict.update({engine: ipList})

    # 控制器未配置管理IP，无法连接控制器。请参考预警公告手动检查。
    if len(ip_list_dict) == 0:
        msg = common.getMsg(lang, "contr.not.config.ip")
        return (False, cli_ret, msg, ip_list_dict)

    return True, cli_ret, msg, ip_list_dict


def get_manager_ip_list_cache(cli, env, logger):
    """
    @summary: 获取引擎对应下的控制器IP
    :param cli: 连接
    :param env: 上下文py_java_env
    :param logger: 日志打印
    :return: flag, cli_ret, msg, ip_list_dict
            flag: 命令执行是否成功
            cli_ret: cli回文
            msg: 失败时错误消息
            ip_list_dict: 每个引擎的ip列表
    """
    lang = common.getLang(env)
    cmd = "show port general logic_type=Management_Port " \
          "physical_type=ETH running_status=link_up"
    flag, cli_ret, msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    temp_list = cliUtil.getHorizontalCliRet(cli_ret)
    ip_list = []
    for info_dict in temp_list:
        port_id = info_dict.get("ID")
        if port_id.startswith("CTE") or port_id.startswith("ENG"):
            ip_address = info_dict.get("IPv4 Address")
            if ip_address == "--":
                continue

            ip_list.append(ip_address)

    return ip_list, cli_ret


def get_product_model_with_cliret_cache(env, cli, logger):
    """
    @summary: 获取产品类型并返回CLI回显
    :param cli: 连接
    :param env: 上下文py_java_env
    :param logger: 日志打印
    @return: (flag, ret, errMsg, productModel)
        flag:
            True: 获取成功
            False: 获取失败
        ret: cli回显
        errMsg: 错误消息
        productModel: 产品型号
 """
    lang = common.getLang(env)
    flag, cli_ret, msg, product_model, product_version = \
        get_oem_prodcut_model_version_cache(env, cli, logger, lang)

    if flag is True:
        return True, cli_ret, "", product_model

    cmd = "show system general"
    flag, cli_ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                            logger)
    if flag is not True:
        return flag, cli_ret, msg, ""

    cli_ret_list = cli_ret.splitlines()
    product_model = ""
    for line in cli_ret_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if field_name == "Product Model":
            product_model = field_value

        if len(product_model) != 0 and product_model != "--":
            return True, cli_ret, "", product_model

    return False, cli_ret, cliUtil.getMsg(lang,
                                          "cannot.get.product.model.info"), ""


def get_oem_prodcut_model_version_cache(env, cli, logger, lang):
    """
    获取深度OEM设备的华为型号和版本
    :param cli: 连接
    :param env: 上下文py_java_env
    :param logger: 日志打印
    :param lang: 中英文
    :return: (flag, ret, errMsg, productModel，productVersion)
        flag:
            True: 获取成功
            False: 获取失败
        ret: cli回显
        errMsg: 错误消息
        productModel: 产品型号
        productVersion：软件版本

    """
    flag, cli_ret, msg, info_dict = \
        get_oem_sys_info_cache(env, cli, logger, lang)
    if flag is not True:
        return flag, cli_ret, msg, "", ""

    product_model = info_dict.get("Internal Product Model")
    product_version = info_dict.get("Product Version")

    if product_model and product_version:
        return True, cli_ret, "", product_model.strip(), \
               product_version.strip()

    return False, cli_ret, cliUtil.getMsg(lang, "query.result.abnormal",
                                          resource=cliUtil.resource.
                                          MESSAGES_DICT), "", ""


def get_oem_sys_info_cache(env, cli, logger, lang):
    """
      获取深度OEM设备的华为型号等基本信息
    :param cli: 连接
    :param env: 上下文py_java_env
    :param logger: 日志打印
    :param lang: 中英文
    :return: (flag, ret, errMsg, list)
        flag:
            True: 获取成功
            False: 获取失败
        ret: cli回显
        errMsg: 错误消息
        list: 基本信息
    """
    dev_node = env.get('devInfo')
    product_model = restUtil.CommonRest.getOriginProductModel(dev_node)
    cmd = "show system general|filterColumn exclude columnList=Product\\sModel"
    flag, cli_ret, msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                            logger)
    if flag is not True:
        return flag, cli_ret, msg, {}

    dict_list = cliUtil.getVerticalCliRet(cli_ret)
    if len(dict_list) == 0:
        return (False, cli_ret,
                cliUtil.getMsg(lang, "query.result.abnormal",
                               resource=cliUtil.resource.MESSAGES_DICT), {})
    if product_model:
        dict_list[0]["Internal Product Model"] = product_model

    return True, cli_ret, "", dict_list[0]


def is_use_running_data(p_version):
    """
    原因：因维护版本FC驱动限制部分版本收集config.txt。
    结论：本次优化版本范围： Dorado V6、V500R007C60(包含kunpeng和非kunpeng)
        V300R006C61及其之后
    :param p_version:
    :return:
    """
    if not p_version:
        return False

    if product.is_dorado_series_version(p_version):
        return True

    if p_version.startswith("V500R007") and p_version >= "V500R007C60":
        return True

    if p_version.startswith("V300R006") and p_version >= "V300R006C61":
        return True

    return False


def filter_hyper_host(host_lun_info_list, hyper_lun_dict, host_dict):
    for host_id in host_dict:
        tmp_hyper_lun_list = []
        tmp_lun_list = host_lun_info_list.get(host_id, [])
        for lun_id in tmp_lun_list:
            if lun_id in hyper_lun_dict:
                tmp_hyper_lun_list.append(lun_id)
                break
        if not tmp_hyper_lun_list:
            host_dict.pop(host_id)


def update_fc_ini(fc_ini_dict, host_info_dict):
    for host_id in fc_ini_dict:
        if host_id not in host_info_dict:
            continue

        tmp_ini_list = fc_ini_dict.get(host_id, [])
        tmp_dict = host_info_dict.get(host_id, {})
        initiatorList = tmp_dict.get("initiatorList", [])
        initiatorList.extend(tmp_ini_list)
        tmp_dict["initiatorList"] = initiatorList
        host_info_dict[host_id] = tmp_dict


def ana_running_data(cli, sn, p_version, lang, conn, logger, obj_py):
    """
    下载并将config解析成字典
    :param cli: ssh连接
    :param sn: 阵列SN
    :param p_version: 版本
    :param lang: 中英文
    :param conn: 数据库连接
    :param logger: 日志
    :param obj_py: 用于保存特点信息的上下文
    :return:
    """
    try:
        script_path = os.path.split(os.path.realpath(__file__))[0]
        rule_path = os.path.join(script_path,
                                 "running_data_analyze_rule.xml")
        local_path = os.path.abspath(".\\temp")
        sftp = SftpCls(cli)
        flag, res_path, err_msg = collect_running_data(
            sftp, cli, lang, local_path, sn, logger
        )
        sftp.close()
        logger.logInfo(
            "flag is:{0},download path is:{1}, msg:{2}".format(
                flag, res_path, err_msg
            )
        )
        if flag is not True:
            logger.logError("download config.txt error!")
            return False
        ana_util.execute_analyze(res_path,
                                 p_version,
                                 rule_path,
                                 conn,
                                 logger)
        save_disk_info_to_env(sn, obj_py, res_path)
        logger.logInfo("save rm info to env")
        save_remote_replication_info_to_env(sn, obj_py, res_path)
        os.remove(res_path)
        return True
    except Exception as e:
        logger.logInfo("delete file exception:{0}".format(str(e)))
        return False
    except:
        logger.logError("unexpected exception:%s" % str(traceback.format_exc()))
        return False


def save_disk_info_to_env(sn, obj_py, file_path):
    """
    解析disk info 放到内存中，因硬盘信息非标准格式，单独解析。
    :param sn: 设备SN
    :param obj_py: env
    :param file_path: config路径
    :return:
    """
    disk_info_list = get_disk_info_from_config(file_path)
    tmp_info_dict = obj_py.get(DISK_INFO_FROM_CONFIG.format(sn))
    if not tmp_info_dict:
        tmp_info_dict = {}
    tmp_info_dict["disk_info"] = disk_info_list
    obj_py.put(DISK_INFO_FROM_CONFIG.format(sn), tmp_info_dict)


def is_disk_info_start(line):
    return (
        MEMBER_DISK_TITLE_STR in line and MEMBER_DISK_REGX.match(line)
    ) or (FREE_DISK_TITLE_STR in line and FREE_DISK_REGX.match(line))


def get_disk_info_from_config(file_path):
    """
    从config中解析 disk 信息
    :param file_path:
    :return:
    """
    start_flag = False
    disk_info_end_line_num = 0
    file_end_flag = 10
    disk_info_dict_list = []
    tmp_disk_info = {}
    with codecs.open(file_path, "r") as read_file:
        file_end_space_line = 0
        while file_end_space_line < file_end_flag:
            line = read_file.readline()
            if not line.strip():
                file_end_space_line += 1
            else:
                file_end_space_line = 0

            if is_disk_info_start(line):
                start_flag = True
                tmp_disk_info = {}
                continue

            if not start_flag:
                continue

            (
                disk_info_end_line_num,
                start_flag,
                file_end_space_line,
            ) = set_disk_info(
                line,
                disk_info_end_line_num,
                tmp_disk_info,
                disk_info_dict_list,
                start_flag,
                file_end_space_line,
            )
    return disk_info_dict_list


def set_disk_info(
    line,
    disk_info_end_line_num,
    tmp_disk_info,
    disk_info_dict_list,
    start_flag,
    file_end_space_line,
):
    if not line.strip():
        disk_info_end_line_num += 1
        if tmp_disk_info:
            disk_info_dict_list.append(tmp_disk_info.copy())
        tmp_disk_info = {}
    else:
        file_end_space_line = 0

    if disk_info_end_line_num >= 2:
        start_flag = False
        tmp_disk_info = {}
        disk_info_end_line_num = 0
        return disk_info_end_line_num, start_flag, file_end_space_line

    tmp_key_list = line.split(":")
    if len(tmp_key_list) < 2:
        return disk_info_end_line_num, start_flag, file_end_space_line

    disk_info_end_line_num = 0
    tmp_disk_info[tmp_key_list[0].strip()] = "".join(tmp_key_list[1:]).strip()

    return disk_info_end_line_num, start_flag, file_end_space_line


def save_remote_replication_info_to_env(sn, obj_py, file_path):
    """
    解析pair info 放到内存中。
    :param sn: 设备SN
    :param obj_py: env
    :param file_path: config路径
    :return:
    """
    rm_info_list = get_remote_replication_info_from_config(file_path)
    tmp_info_dict = obj_py.get(REMOTE_REPLICATION_INFO_FROM_CONFIG.format(sn))
    if not tmp_info_dict:
        tmp_info_dict = {}
    tmp_info_dict["remote_replication_info"] = rm_info_list
    obj_py.put(REMOTE_REPLICATION_INFO_FROM_CONFIG.format(sn), tmp_info_dict)


RM_INFO_COLUMN = [
    "Pair ID", "dsType", "Role", "Master Lun ID", "Slave Lun ID", "Primary Resource Capacity",
    "Remote Replication Type", "Master Fs ID", "Slave Fs ID"
]


def get_remote_replication_info_from_config(file_path):
    """
        从config中解析 rm 信息
        :param file_path:
        :return:
    """
    rm_info_list = [{}]
    with codecs.open(file_path, "r") as read_file:
        is_rm_start = False
        rm_index = 0
        for line in read_file:
            if "Max Asynchronous Remote Replication Number:" in line:
                line = line.strip()
                tmp_key_list = line.split(":")
                rm_number = tmp_key_list[1].strip()
                continue
            if " Remote Replication Information" in line and not is_rm_start:
                is_rm_start = True
                continue
            if "LUN Migration--------------------------------" in line:
                break
            if not is_rm_start:
                continue
            rm_index, rm_info_list = parse_remote_replication_info_from_config(rm_info_list, rm_index, line)
            if rm_number == rm_index:
                break
    return rm_info_list


def parse_remote_replication_info_from_config(rm_info_list, rm_index, line):
    if line.strip() == "":
        return rm_index, rm_info_list
    if "Remote Replication Information" in line:
        rm_info_list.append({})
        return rm_index + 1, rm_info_list
    tmp_key_list = line.split(":")
    if "Primary" in tmp_key_list[0].strip() and len(tmp_key_list) > 1:
        rm_info_list[rm_index].update({"Primary Resource Capacity": tmp_key_list[1].strip()})
        return rm_index, rm_info_list
    if tmp_key_list[0].strip() in RM_INFO_COLUMN and len(tmp_key_list) > 1:
        if tmp_key_list[0].strip() == "Master Lun ID":
            rm_info_list[rm_index].update({"dsType": "LUN"})
        if tmp_key_list[0].strip() == "Master Fs ID":
            rm_info_list[rm_index].update({"dsType": "FS"})
        rm_info_list[rm_index].update({tmp_key_list[0].strip(): tmp_key_list[1].strip()})
    return rm_index, rm_info_list


def save_cli_ret_to_file(ret, item_id, env, logger):
    """
    如果大于2M则写文件，目的不能脚本侧在返回回文之前就内存溢出
    :param ret: 回文
    :param item_id: 检查项ID
    :param env: env
    :param logger: logger
    :return: 保存路径
    """

    tmp_str = ret
    is_list_too_big = False
    if isinstance(ret, list):
        if len(ret) < LIMIT_LIST_SIZE:
            tmp_str = "\n".join(ret)
        else:
            is_list_too_big = True

    # 脚本转储
    if not is_list_too_big and len(tmp_str) < LIMIT_SIZE:
        return tmp_str

    logger.logInfo("dump info ret:{0}".format(len(tmp_str)))
    # 获取保存路径
    dev_node = env.get("devInfo")
    ip = dev_node.getIp()
    sn = dev_node.getDeviceSerialNumber()
    base_path = os.path.abspath(common.DIR_RELATIVE_CMD)
    pid = str(env.get('pid'))
    tmp_path = os.path.join(base_path, "tmp")
    tmp_path = os.path.join(tmp_path, "Record")
    tmp_path = os.path.join(tmp_path, pid)
    dump_path = os.path.join(tmp_path,
                             "{ip}_{sn}".format(
                                 ip=ip.replace(":", "_"), sn=sn))
    if not os.path.exists(dump_path):
        os.makedirs(dump_path)

    file_name = "{itemId}{tail}".format(itemId=item_id, tail=".dump")
    file_path = os.path.join(dump_path, file_name)
    with codecs.open(file_path, "w", encoding="utf-8") as f:
        if isinstance(tmp_str, list):
            for line in tmp_str:
                f.write("\n" + line)
        else:
            f.write(tmp_str)

    logger.logInfo("create dump by script info:{0}".format(file_path))
    return "[ONLY FOR INSPECT CLI ECHO PATH]:{0}".format(file_path)


def get_lower_obj_in_lower_list(obj, obj_list):
    """
    返回对象在list的key值
    如：Oracle_VM -> Oracle VM
    1:Linux;    2:Windows;
    3:Solaris;  4:HP-UX;
    5:AIX;      6:XenServer;
    7:Mac_OS;  8:VIS6000;
    9:VMware_ESX;  10:Windows_Server_2012;
    11:Oracle_VM; 12:OpenVMS;
    :param obj:
    :param obj_list:
    :return:
    """
    for tmp_obj in obj_list:
        if obj.lower().replace("_", " ") == tmp_obj.lower().replace(
                "_", " "):
            return tmp_obj

    return None


def is_4u_or_6u_engine_type(engine_type):
    """
    是否为4U或6U引擎
    :param engine_type: 引擎type
    :return:
    """
    low_engine_type = engine_type.lower()
    return '4u' in low_engine_type or '6u' in low_engine_type


def is_management_local_port(port):
    """
    是否为本地管理端口
    :param port: 本地端口
    :return:
    """
    return 'MGMT' in port


def get_remote_device_info_cache(env, cli, logger):
    """
    获取当前设备所有远端信息
    :param env:
    :param cli:
    :param logger:
    :return:
    """
    remote_device_dict = {}
    cmd = "show remote_device general"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                                logger)
    if flag is not True:
        raise common.UnCheckException(err_msg, cli_ret)
    remote_device_list = cliUtil.getHorizontalCliRet(cli_ret)
    for remote_dev in remote_device_list:
        r_id = remote_dev.get("ID", '')
        r_sn = remote_dev.get("SN", '')
        remote_device_dict[r_id] = r_sn
    return remote_device_dict, cli_ret


def get_share_cifs_cache(env, cli, logger):
    """
    获取share cifs信息
    :param env: 上下文
    :param cli: ssh连接
    :param logger: 日志
    :return: cifs id列表
    """
    share_ids = []
    cmd = "show share cifs"
    flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(env, cli, cmd,
                                                                logger)
    # 当命令不支持时，返回空列表
    if not cliUtil.hasCliExecPrivilege(cli_ret):
        return share_ids, cli_ret

    if flag is not True:
        raise common.UnCheckException(err_msg, cli_ret)

    share_cifs_dict_list = cliUtil.getHorizontalCliRet(cli_ret)
    for share_cifs_dict in share_cifs_dict_list:
        share_id = share_cifs_dict.get("Share ID", '')
        if not share_id:
            continue
        share_ids.append(share_id)
    return share_ids, cli_ret


def get_share_cifs_cache_detail(env, cli, logger, cifs_ids):
    """
    获取share cifs信息
    :param env: 上下文
    :param cli: ssh连接
    :param logger: 日志
    :return: cifs id列表
    """
    cifs_detail_list = []
    ret_list = []
    base_cmd = "show share cifs share_id={}"
    for cifs_id in cifs_ids:
        cmd = base_cmd.format(cifs_id)
        flag, cli_ret, err_msg = execute_cmd_in_cli_mode_with_cache(
            env, cli, cmd, logger
        )
        ret_list.append(cli_ret)
        if flag is not True:
            raise common.UnCheckException(err_msg, "\n".join(ret_list))

        cifs_detail_list.extend(cliUtil.getVerticalCliRet(cli_ret))

    return cifs_detail_list, ret_list


def get_filesystem_domain_info_cache(env, cli, logger, cmd=CMD_FS_DOMAIN):
    """
    获取fs双活信息
    :param env: 上下文
    :param cli: ssh连接
    :param logger: 日志
    :param cmd: 默认命令
    """
    fs_domain_dict = {}

    flag, ret, msg = execute_cmd_in_cli_mode_with_cache(
        env, cli, cmd, logger
    )

    if not_support_nas_domain(ret):
        return fs_domain_dict, ret

    if flag is not True:
        raise common.UnCheckException(msg, ret)

    return cliUtil.getHorizontalCliRet(ret), ret


def not_support_nas_domain(ret):
    """
    判断不支持NAS双活
    :param ret:
    :return: True：不支持，False 支持
    """
    if not cliUtil.hasCliExecPrivilege(ret):
        return True
    return "the current device does not support nas" in str(ret).lower()


def get_san_pair_from_cache(env, logger, sn, lang):
    flag, ret, msg = common.getObjFromFile(env, logger, sn, CMD_SAN_PAIR, lang)
    return flag, ret, msg, cliUtil.getHorizontalNostandardCliRet(ret)


def get_nas_pair_from_cache(env, logger, sn, lang):
    flag, ret, msg = common.getObjFromFile(env, logger, sn, CMD_NAS_PAIR, lang)
    return flag, ret, msg, cliUtil.getHorizontalNostandardCliRet(ret)


def get_san_pair_from_cache_cg(env, logger, sn, lang):
    flag, ret, msg = common.getObjFromFile(env, logger, sn, CMD_SAN_PAIR_CG, lang)
    return flag, ret, msg


def get_pair_to_dev_sn(env, sn, logger, lang):
    """
    @summary: 获取双活pair
    @return: 双活pair ID 对应设备SN字典
    """
    pair_sn_dict = {}
    flag, domain_dict, msg = common.getDomainInfo(
        env, sn, logger, lang
    )
    if not domain_dict:
        return flag, {}, msg

    flag, remote_device_dict, msg, ret = common.getRemoteDeviceInfo(
        env, sn, domain_dict, logger, lang)
    if not remote_device_dict:
        return flag, {}, msg

    flag, ret, msg, pair_list = get_san_pair_from_cache(
        env, logger, sn, lang)
    for pair_info in pair_list:
        pairId = pair_info.get("ID", '')
        domain_id = pair_info.get("Domain ID")
        if not pairId:
            continue
        pair_sn_dict[pairId] = remote_device_dict.get(domain_id)
    return True, pair_sn_dict, ''
