# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.


import urllib
import resource_dpa
import cliUtil
from common_utils import get_dpa_rest_record
from java.lang import Exception as JException

# 接口返回列表最多个数
MAX_CONTENT_COUNT = 15


def get_dpa_msg(lang, msg, args=""):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @return: 经过国际化处理后的消息
    '''
    return cliUtil.getMsg(lang, msg, args, resource_dpa.DPA_MESSAGES_DICT)


def parse_quote(url, char_code='utf-8'):
    """
    获取dpa巡检rest返回值(仅限python2使用,python3方法为urllib.parse.quote)
    :param url:解析目标
    :param char_code:字符编码
    :return:
    """
    return urllib.quote_plus(url, char_code)


def get_node_list(dpa_rest):
    """
    获取主机节点列表
    """
    index = 0
    node_query_uri = "/clusters/nodes?count={}&index={}"
    record = dpa_rest_pagination_query(dpa_rest, node_query_uri, index)
    dev_node_list = record.get("responseData", {}).get("data", [{}])
    total_num = record.get("responseData", {}).get("totalNum")
    if not total_num:
        return None, record
    while total_num > index + MAX_CONTENT_COUNT:
        index += MAX_CONTENT_COUNT
        tem_record = dpa_rest_pagination_query(dpa_rest, node_query_uri, index)
        record = "{}\n{}".format(record, tem_record)
        dev_node_list.extend(tem_record.get("responseData", {}).get("data", [{}]))
    return dev_node_list, record


def dpa_rest_pagination_query(dpa_rest, url, index, content_count=MAX_CONTENT_COUNT):
    """
    rest接口分页查询
    :param dpa_rest:
    :param url:
    :param index: 查询结果开始的下标
    :param content_count: 该页获取的结果数量
    :return:
    """
    return get_dpa_rest_record(dpa_rest, url.format(content_count, index))


def get_backup_job_list(dpa_rest):
    """
    获取备份任务列表
    :param dpa_rest:
    :return:
    """
    index = 0
    node_query_uri = "/backups/backup_jobs?count={}&groupId=&index={}"
    record = dpa_rest_pagination_query(dpa_rest, node_query_uri, index)
    job_list = record.get("responseData", {}).get("data", [{}])
    total_num = record.get("responseData", {}).get("totalNum", 0)
    if not total_num:
        return job_list, record
    while total_num > index + MAX_CONTENT_COUNT:
        # total_number大于index+count值，未查完信息，继续执行
        index += MAX_CONTENT_COUNT
        tem_record = record = dpa_rest_pagination_query(dpa_rest, node_query_uri, index)
        record = "{}\n{}".format(record, tem_record)
        job_list.extend(tem_record.get("responseData", {}).get("data", [{}]))
    return job_list, record


def get_time_stamp(dpa_rest, py_java_env, logger):
    """
    获取备份任务列表
    :param dpa_rest:
    :param py_java_env:
    :param logger:
    :return:
    """
    dev_node_list, record = get_node_list(dpa_rest)
    dev_ip_list = [dev_node.get("nodeIp") for dev_node in dev_node_list]
    for dev_node in py_java_env.get('selectDevs'):
        # 找到任一节点IP，登录对应主机查询时间戳
        if str(dev_node.getIp()) in dev_ip_list:
            connection = None
            try:
                connection = py_java_env.get("sshManager").getSshConnection(dev_node)
                time_stamp, date_ret = query_date_info(connection)
                record = "{}\n{}".format(record, date_ret)
                return time_stamp, record
            except JException as exp:
                logger.logNoPass("Create SSH {} connection catch exception:{}".format(dev_node.getIp(), exp))
            finally:
                py_java_env.get("sshManager").releaseConnection(connection)
    return None, record


def query_date_info(connection):
    """
    查询主机本地时间
    """
    date_ret = connection.execCmd('date +%s%3N')
    date_ret_line = date_ret.encode("utf8").splitlines()
    if len(date_ret_line) > 2:
        return date_ret.encode("utf8").splitlines()[1], date_ret
    return None, date_ret
