# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from memory_inspect.utils import constants

import re

MESSAGE_TIME_PATTERN = "[\[\s]*(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})"
BSP_MESSAGE_PATTERN = "[\[\s]*(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\]\[(\s*\w+\.*\w*)\]"
ERROR_DQ_REGEX = "[\[\s]*(\d{4}-\d{2}-\d{2} \d{2}:\d{2}).*,(( DQ\d+)+),"
ERROR_MEMORY_ECC_LINE_FLAG = "MemoryEcc"
ERROR_MEMORY_ECC = "[\[\s]*(\d{4}-\d{2}-\d{2} \d{2}:\d{2}).*\[DIMM(\d+) MemoryEcc add (\d+), rank(\d+), dev(\d+), bank(\d+), row=(\w+), col=(\w+), Reg \w+.*"

"""
系统盘检查规则定义：
由于message文件可能非常多，这里为了优化性能，不是直接用正则匹配关键字
先用find_str查看是否存在特定的关键字，满足条件后再正则匹配
由于错误的条数占比非常小，所以90%以上直接通过find_str就跳过了，不需要使用正则，该操作可提升8倍性能。
另外，还有一些小的优化：
    1、规则在定义时特地全部采用小写，避免在循环中使用lower()
    2、之前使用的re.search，改为re.match
"""


def bsp_accept_func(time_str):
    """
    Call bsp to power 关键字前面[x]里面的值小于300（表示系统重启起来5分钟），则本次打印不计入判断条件内
    :param time_str: 启动时间
    :return:
    """
    return float(time_str) < 300


SYS_DISK_ERROR_RULES = [
    {
        constants.BSP_POWER_OFF_RULE: {
            "find_str": "call bsp to power off port",
            "match_regex": BSP_MESSAGE_PATTERN,
            "accept_func": bsp_accept_func
        }
    },
    {
        constants.BSP_POWER_ON_RULE: {
            "find_str": "call bsp to power on port",
            "match_regex": BSP_MESSAGE_PATTERN,
            "accept_func": bsp_accept_func
        }
    },
    {
        constants.CTRL_BORD_BOOT_DISK: {
            "find_str": "0x4000cf05",
            "match_regex": MESSAGE_TIME_PATTERN
        }
    },
    {
        constants.HOT_PLUG_INTERRUPT: {
            "find_str": "hotplug interrupt from sata module is coming",
            "match_regex": MESSAGE_TIME_PATTERN
        }
    },
    {
        constants.MASK_ACT_ERR_RULE: {
            "find_str": "exception emask",
            "match_regex": MESSAGE_TIME_PATTERN + ".*exception emask.*sact.*serr.*action"
        }
    },
    {
        constants.EXT_FS_RULE: {
            "find_str": "ext4-fs error",
            "match_regex": MESSAGE_TIME_PATTERN
        }
    },
    {
        constants.READ_ONLY_FS_RULE: {
            "find_str": "read-only file system",
            "match_regex": MESSAGE_TIME_PATTERN
        }
    },
    {
        constants.HARD_RESETTING_LINK: {
            "find_str": "hard resetting link",
            "match_regex": MESSAGE_TIME_PATTERN
        }
    },
]


class MessageParser(object):
    """
    message日志解析，涉及2个文件
    """

    def __init__(self, logger):
        self.logger = logger
        self.ctrl_root_disk_info = []
        self.memory_info = []

    def parse(self, message_files):
        """
        从message文件中解析dq信息
        :param message_files: 一个控制器下所有的message文件
        :return: dq_info:{'2021-02-01 14:50': {time: '2021-02-01 14:50', dq: [DQ2,DQ5], multi_dq: True},
                         '2021-02-01 14:51':  {time: '2021-02-01 14:51', dq: [DQ2], multi_dq: False}} }
        """
        dq_info = {}
        self.memory_info = []
        for file_name in message_files:
            with open(file_name, "r") as log_file:
                for number, line_str in enumerate(log_file):
                    self.pick_dq_info(line_str, dq_info)
                    self.pick_memory_ecc_info(number, line_str)
        return dq_info.values()

    def pick_memory_ecc_info(self, number, line_str):
        if ERROR_MEMORY_ECC_LINE_FLAG not in line_str:
            return
        match_res = re.compile(ERROR_MEMORY_ECC).findall(line_str)
        # 数据格式[('2021-08-10 23:39', '6', '2329', '1', '10', '3', '0x77ff', '0x100')]
        if not match_res or len(match_res[0]) != 8:
            return
        res = match_res[0]
        self.memory_info.append({
            "time": res[0],
            "slot": res[1],  # dimm
            "rank": res[3],
            "device": res[4],  # dev
            "bank": res[5],
            "row": res[6],
            "col": res[7],
            "ordinal": "%s[%s]" % (res[0], str(number).zfill(6)),
            "err_type": "Message Memory Error"
        })

    def parse_err_disk_info(self, message_files):
        """
        从message文件中解析系统盘信息
        :param message_files: 一个控制器下所有的message文件
        :return: disk_info:{'2021-02-01 14:50': {time: '2021-02-01 14:50', dq: [DQ2,DQ5], multi_dq: True},
                         '2021-02-01 14:51':  {time: '2021-02-01 14:51', dq: [DQ2], multi_dq: False}} }
        """
        self.ctrl_root_disk_info = []
        for file_name in message_files:
            with open(file_name, "r") as log_file:
                for number, line_str in enumerate(log_file):
                    self.pick_ctrl_root_disk_info(line_str)
        return self.ctrl_root_disk_info

    @staticmethod
    def pick_dq_info(line, dq_info):
        dq_match = re.match(ERROR_DQ_REGEX, line)
        if not dq_match:
            return {}

        # time 只精确到分钟
        time = dq_match.group(1)
        dq = dq_match.group(2)
        if time not in dq_info:
            dq_info[time] = dict(dq=set(), multi_dq=False, time=time)
        dq_info[time]["dq"].add(dq)
        if len(dq.split()) > 1:
            dq_info[time]["multi_dq"] = True

    def pick_ctrl_root_disk_info(self, line):
        """
        匹配控制器 系统盘异常的规则
        ctrl_root_disk_info: [{"time":"2021-08-30 21:53:53", "otherInfo":"xxxx", "rule_num": "rule_0001"},
        {"time":"2021-08-30 21:53:53", "otherInfo":"xxxx","rule_num": "rule_0002"}],]
        :param line:
        :return:
        """
        line = line.lower()
        for rule_info in SYS_DISK_ERROR_RULES:
            for rule_name, rule in rule_info.items():
                if rule.get("find_str") not in line:
                    continue
                res_match = re.match(rule.get("match_regex"), line)
                if not res_match:
                    continue
                res_match_time = res_match.group(1)
                if rule.get("accept_func") and rule.get("accept_func")(res_match.group(2)):
                    continue
                self.ctrl_root_disk_info.append(
                    {
                        "time": res_match_time,
                        "err_type": rule_name,
                        "matched_data": line,
                        "origin_info": line,
                    }
                )
