# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2023. All rights reserved.

import os
from cbb.common.excel import excelParse


class QueryCtrlBomInfo:
    """
    查询控制器BOM编码信息
    """

    def __init__(self, logger):
        self.excel_file = get_excel_file_path(logger)
        self.excel_file_sheet_name = "ctrl_bom"
        # 起始行数
        self.begin_row = 10
        self.excel_instance = excelParse.Excel(self.excel_file, logger)
        # BOM编码列表，对应第一列（Part Number）
        self.bom_code_list = self.get_execl_info(0)
        # BOM编码支持的型号，对应第二列（Product）
        self.product_model_list = self.get_execl_info(1)
        # 单控内存，对应第8列（单控内存）
        self.single_ctrl_memory_list = self.get_execl_info(7)
        # BOM类型（控制器或控制框），对应第三列（类型）
        self.type_list = self.get_execl_info(2)
        # 背板形态（SAS/NVMe）,对应第六列（SAS/NVMe）
        self.back_model_list = self.get_execl_info(5)
        # 软件代次（C/C+），对应第四列（软件代次）
        self.software_generation_list = self.get_execl_info(3)
        # CPU信息（1P/2P/3P/4P）,对应第七列（CPU数）
        self.cpu_info_list = self.get_execl_info(6)
        # 需求版本，对应第五列（版本+内核版本）
        self.required_version_list = self.get_execl_info(4)
        # 当前bom对应的配件是混闪还是全闪
        self.device_type = self.get_execl_info(8)
        self.bom_info_dict = dict()

    def get_bom_info(self):
        for bom_index, bom_code in enumerate(self.bom_code_list):
            if bom_code in self.bom_info_dict:
                self.bom_info_dict[bom_code][0] += ";{}".format(self.product_model_list[bom_index])
                continue
            bom_info_list = list()
            bom_info_list.append(self.product_model_list[bom_index])
            bom_info_list.append(self.single_ctrl_memory_list[bom_index])
            bom_info_list.append(self.type_list[bom_index])
            bom_info_list.append(self.back_model_list[bom_index])
            bom_info_list.append(self.software_generation_list[bom_index])
            bom_info_list.append(self.cpu_info_list[bom_index])
            bom_info_list.append(self.required_version_list[bom_index])
            bom_info_list.append(self.device_type[bom_index])
            self.bom_info_dict[bom_code] = bom_info_list
        return self.bom_info_dict

    def get_execl_info(self, column):
        flag, info_list = self.excel_instance.getCellInfosOneColumn(self.excel_file_sheet_name, self.begin_row, column)
        return info_list


class QueryInterfaceBomInfo:
    """
    查询接口卡BOM编码信息
    """

    def __init__(self, logger):
        self.excel_file = get_excel_file_path(logger)
        self.excel_file_sheet_name = "interface_bom"
        # bom编码起始行数为5
        self.bom_begin_row = 4
        # 型号版本起始行数为3
        self.product_begin_row = 2
        self.excel_instance = excelParse.Excel(self.excel_file, logger)
        # BOM编码
        self.bom_code_list = self.get_execl_info(self.bom_begin_row, 0)
        # 要求的版本
        self.product_and_version_list = self.get_required_product_and_version()
        # 要求的软件代次
        self.software_generation_list = self.get_execl_info(self.bom_begin_row, 2)
        self.bom_info_dict = dict()

    def get_bom_info(self):
        """
        数据格式：{"bom_code": [(型号, 软件代次, 版本), (型号, 软件代次, 版本), 。。。]，
                 "bom_code": [(型号, 软件代次, 版本), (型号, 软件代次, 版本), 。。。]， 。。。}
        """
        for bom_code in self.bom_code_list:
            bom_index = self.bom_code_list.index(bom_code)
            self.bom_info_dict[bom_code] = self.get_product_and_version(bom_index)
        return self.bom_info_dict

    def get_product_and_version(self, bom_index):
        product_and_version_info_list = list()
        # 产品信息比bom编码多2行
        version_index = bom_index + 2
        for product_info in self.product_and_version_list:
            required_version = product_info[version_index]
            # 不支持
            if not required_version or required_version.upper() == "NO":
                continue
            required_product_model = product_info[0]
            required_software_generation = self.software_generation_list[bom_index]
            required_enc_software_generation = product_info[1]
            product_and_version_info_list.append(
                (required_product_model, required_software_generation, required_version,
                 required_enc_software_generation))
        return product_and_version_info_list

    def get_required_product_and_version(self):
        product_info_list = list()
        # 起始列数为3
        start_column = 3
        # 循环结束标志
        finish_flag = False
        while not finish_flag:
            product_info = self.get_execl_info(self.product_begin_row, start_column)
            if not product_info:
                finish_flag = True
                continue
            product_info_list.append(product_info)
            start_column += 1
        return product_info_list

    def get_execl_info(self, row, column):
        flag, info_list = self.excel_instance.getCellInfosOneColumn(self.excel_file_sheet_name, row, column)
        return info_list


class QueryDiskEnclosureBomInfo:
    """
    查询硬盘框BOM编码信息
    """

    def __init__(self, logger):
        self.logger = logger
        self.excel_file = get_excel_file_path(logger)
        self.excel_file_sheet_name = "disk_enc_bom"
        # bom编码起始行数
        self.bom_begin_row = 8
        self.excel_instance = excelParse.Excel(self.excel_file, logger)
        # BOM编码
        self.bom_code_list = self.get_execl_info(self.bom_begin_row, 0)
        self.enc_type_list = self.get_execl_info(self.bom_begin_row, 1)
        self.hardware_type_list = self.get_execl_info(self.bom_begin_row, 2)
        logger.logInfo("QueryDiskEnclosureBomInfo bom code list:{}".format(self.bom_code_list))
        self.bom_info_dict = dict()

    def get_bom_info(self):
        """
        数据格式：{"bom_code": [硬盘框型号, 软件代次]}
        """
        for bom_index, bom_code in enumerate(self.bom_code_list):
            self.bom_info_dict[bom_code] = [self.enc_type_list[bom_index], self.hardware_type_list[bom_index]]
        self.logger.logInfo("QueryDiskEnclosureBomInfo bom_info_dict:{}".format(self.bom_info_dict))
        return self.bom_info_dict

    def get_execl_info(self, row, column):
        flag, info_list = self.excel_instance.getCellInfosOneColumn(self.excel_file_sheet_name, row, column)
        return info_list


def get_excel_file_path(logger):
    """
    获取文件路径
    @param logger: 日志
    @return: 文件路径
    """
    current_file_path = os.path.split(os.path.realpath(__file__))[0]
    logger.logInfo("pyFilePath={}".format(current_file_path))
    excel_file = os.path.abspath(os.path.join(current_file_path, "../..", "expinfo\\hardware_bom_info.xls"))
    logger.logInfo("excelFile={}".format(excel_file))
    return excel_file


def remove_error_input(bom_list):
    if bom_list[-1] == "":
        bom_list.pop()
    return bom_list
