#!/bin/bash


# --------------------------  VARIABLE  ------------------------------------- #

DIR=`pwd`

CHECK_CMD_FILE=$DIR/check_item.xml
CHECK_RESULT_FILE=$DIR/check_result.xml

MegaDIR=/opt/MegaRAID/MegaCli
MegaLog=$DIR/megalog
ProcessDoneLog=$DIR/ProcessDoneLog
ROLE="vtl"
SIR_Option=`grep SIROption $IS_CONF | awk -F'enabled="' '{print $2}' | cut -d '"' -f1`
HOSTNAME=`hostname`

#----------------------------modprobe ipmimod---------------------------------#

modprobe ipmi_si
modprobe ipmi_devintf
modprobe ipmi_msghandler


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: InitFile
# Description: Initialization Result Files
# Parameter:
# Return:
# **************************************************************************** #

InitFile ()
{
	FileName="$*"
	[ -f $FileName ] && rm -rf $FileName 
	touch $FileName && chmod 640 $FileName
}

# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckAndExecCommand
# Description: Entrance of this scripts
# Parameter:
# Return:
# **************************************************************************** #


VTL_CheckAndExecCommand ()
{
	while [ "$2" != "" ]
	do
		CHECK_CMD=$2
	
		case $CHECK_CMD in
		baseinfo)
			VTL_CheckBaseInfo $CHECK_CMD;;
		0)
			VTL_CheckVTLStatus $CHECK_CMD;;
		1)
			VTL_CheckTomcatStatus $CHECK_CMD;;
		2)
			VTL_CheckKdumpStatus $CHECK_CMD;;
		3)
			VTL_CheckFCStatus $CHECK_CMD;;
		4)
			VTL_CheckMemIdel $CHECK_CMD;;
		5)
			VTL_CheckPowerSupply $CHECK_CMD;;
		6)
			VTL_CheckHostname $CHECK_CMD;;
		7)
			VTL_CheckHBADriver $CHECK_CMD;;
		8)
			VTL_CPULoadCheck $CHECK_CMD;;
		9)
			VTL_EthLinkCheck $CHECK_CMD;;
		10)
			VTL_HardDiskCheck $CHECK_CMD;;
		11)
			VTL_CheckVTL $CHECK_CMD;;
		12)
			VTL_CheckNBU $CHECK_CMD;;
		13)
			VTL_AmbientTemp $CHECK_CMD;;
		14)
			VTL_CheckFAN $CHECK_CMD;;
		15)
			VTL_CheckDatabase $CHECK_CMD;;
		16)
			VTL_CheckInVault $CHECK_CMD;;
		17)
			VTL_CheckHA $CHECK_CMD;;
		18)
			VTL_CheckSIRAssociate $CHECK_CMD;;
		19)
			VTL_CheckSIRIPMI $CHECK_CMD;;
		20)
			VTL_CheckStorage $CHECK_CMD;;
		21)
			VTL_SIRmonitor $CHECK_CMD;;
		22)
			VTL_CheckVDAttribute $CHECK_CMD;;
		23)
			VTL_CheckLog $CHECK_CMD;;
		24)
			VTL_CheckOfflineTape $CHECK_CMD;;
		esac
		shift
	done

	cat $CHECK_RESULT_FILE

	exit 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: WriteResultHeaderToXml
# Description: Write Fixed Format Header To The Result File
# Parameter:
# Return:
# **************************************************************************** #

WriteResultHeaderToXml()
{
	echo "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" >>$CHECK_RESULT_FILE
	echo "<device>" >>$CHECK_RESULT_FILE
	echo "	<cmditem id=\""$1"\">"  >>$CHECK_RESULT_FILE	
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: WriteErrorItemResultToXml
# Description: Write Fixed Error Format To The Result File
# Parameter:
# Return:
# **************************************************************************** #

WriteErrorItemResultToXml()
{
    echo "		<item id="\"$1\"">" >>$CHECK_RESULT_FILE
    echo "			<errorvalue type=\"$2\">$3</errorvalue>" >>$CHECK_RESULT_FILE
    echo "			<judge>"1"</judge>" >>$CHECK_RESULT_FILE
    echo "		</item>" >>$CHECK_RESULT_FILE
}

# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: WriteNormalItemResultToXml
# Description: Write Fixed Normal Format To The Result File
# Parameter:            
# Return:
# **************************************************************************** #

WriteNormalItemResultToXml()
{
    echo "		<item id=\""$1"\">" >>$CHECK_RESULT_FILE
    echo "			<commonvalue type=\"$2\">$3</commonvalue>" >>$CHECK_RESULT_FILE
    echo "			<judge>"0"</judge>" >>$CHECK_RESULT_FILE
    echo "		</item>" >>$CHECK_RESULT_FILE
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: WriteNormalItemResultToXml
# Description: Write Fixed Format End To The Result File
# Parameter:            
# Return:
# **************************************************************************** #

WriteResultEndToXml()
{	
    echo "	</cmditem>" >>$CHECK_RESULT_FILE
    echo "</device>" >>$CHECK_RESULT_FILE
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_TestNumberIsPowerOfTwo
# Description: Test if the number is power of 2
# Parameter:The number
# Return:0/1
# **************************************************************************** #

VTL_TestNumberIsPowerOfTwo ()
{
	local i=$1
	if [[ $i -eq 1 ]]
	then
		return 0
	fi
	while [ "$i" != "1.00" ]
	do
		i=`awk 'BEGIN{printf "%.2f",'$i'/2}'`
		j=`echo $i | cut -d'.' -f2`
		if [[ $j -eq 50 ]]
		then
			return 1
		fi
	done
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_modeTransfer
# Description: transfer mode id to string
# Parameter:mode id
# Return:mode string
# **************************************************************************** #

VTL_modeTransfer ()
{
    if [[ $1 -eq 1 ]]
    then
        echo p2p
    else
        echo loop
    fi
}



# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckSIRProportion
# Description: Check SIR Index/Data Proportion
# Parameter:
# Return:
# **************************************************************************** #

VTL_CheckSIRProportion ()
{
	Disk_Resource=$DIR/disk_resource
	InitFile $Disk_Resource

	index_f_all=0
	index_l_all=0
	fold_f_all=0
	fold_l_all=0
	data_f_all=0
	data_l_all=0
	sed -n '/<DiskResources>/,/\/DiskResources/'p $IS_CONF >> $Disk_Resource
	local index_f=`sed -n '/Index/,/\/LocalDataSet/'p  $Disk_Resource | grep PDevSection | awk -F'firstSector="' '{print $2}' | cut -d'"' -f1`
	for i in $index_f
	do
		index_f_all=`awk 'BEGIN{printf "%d",'$index_f_all'+'$i'}'`
	done
	local index_l=`sed -n '/Index/,/\/LocalDataSet/'p  $Disk_Resource | grep PDevSection | awk -F'lastSector="' '{print $2}' | cut -d'"' -f1`
	for i in $index_l
	do
		index_l_all=`awk 'BEGIN{printf "%d",'$index_l_all'+'$i'}'`
	done
	local fold_f=`sed -n '/Folder/,/\/LocalDataSet/'p  $Disk_Resource | grep PDevSection | awk -F'firstSector="' '{print $2}' | cut -d'"' -f1`
	for i in $fold_f
	do
		fold_f_all=`awk 'BEGIN{printf "%d",'$fold_f_all'+'$i'}'`
	done
	local fold_l=`sed -n '/Folder/,/\/LocalDataSet/'p  $Disk_Resource | grep PDevSection | awk -F'lastSector="' '{print $2}' | cut -d'"' -f1`
	for i in $fold_l
	do
		fold_l_all=`awk 'BEGIN{printf "%d",'$fold_l_all'+'$i'}'`
	done
	local data_f=`sed -n '/SIR_Data/,/\/LocalDataSet/'p  $Disk_Resource | grep PDevSection | awk -F'firstSector="' '{print $2}' | cut -d'"' -f1`
	for i in $data_f
	do
		data_f_all=`awk 'BEGIN{printf "%d",'$data_f_all'+'$i'}'`
	done
	local data_l=`sed -n '/SIR_Data/,/\/LocalDataSet/'p  $Disk_Resource | grep PDevSection | awk -F'lastSector="' '{print $2}' | cut -d'"' -f1`
	for i in $data_l
	do
		data_l_all=`awk 'BEGIN{printf "%d",'$data_l_all'+'$i'}'`
	done
	local Index_Capacity=`awk 'BEGIN{printf "%d",'$index_l_all'+'$fold_l_all'-'$index_f_all'-'$fold_f_all'}'`
	local Data_Capacity=`awk 'BEGIN{printf "%d",'${data_l_all}'-'${data_f_all}'}'`
	
	SIR_Proportion=`awk 'BEGIN{printf "%d",'$Index_Capacity'/'$Data_Capacity'*100}'`
	rm -f $Disk_Resource
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_Discover_Architecture
# Description: Discovery the Architecture for this host
# Parameter:void
# Return:Architecture name
# **************************************************************************** #

VTL_Discover_Architecture ()
{
	local version=`rpm -qa | egrep "vtl|sir" | cut -d'-' -f2`
	Node_1_Name=`hostname`
        Node_1_IP=`ifconfig eth0 | grep "inet addr" | sed 's/^.*addr://g;s/Bcast.*$//g' | sed 's/ //g'`

	if [ "$version" == "5.20" ]
	then
		Architecture=AllInOne
		NumberOfAllNodes=1
		return 0
	fi

	if [ "$ROLE" == "vtl" ]
	then
		local vtlpartner=`grep "<SANBridge name" $IS_CONF | awk -F'groupPartner="' '{print $2}' | awk -F'"' '{print $1}'`
		if [ -z "$vtlpartner" ]
		then
			Architecture=SOLO
			NumberOfAllNodes=1
			return 0
		else
			Node_1_IP=`sed -n '/FailoverPrimary/,/\/FailoverPrimary/'p $IS_CONF | grep clientConnectionIP | awk -F'clientConnectionIP="' '{print $2}' | cut -d'"' -f1`
			Node_2_Name=`sed -n '/FailoverPrimary/,/\/FailoverPrimary/'p $IS_CONF | grep secondaryServer | awk -F'secondaryServer="' '{print $2}' | cut -d '"' -f1`
			Node_2_IP=`grep groupPartner $IS_CONF | awk -F'groupPartner="' '{print $2}' | cut -d '"' -f1`
			NumberOfSIRNode=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | wc -l`
			case $NumberOfSIRNode in
			0)
				Architecture=HA
				NumberOfAllNodes=2;;
			1)
				Architecture=2+1
				NumberOfAllNodes=3
				Node_3_Name=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | awk -F'name="' '{print $2}' | cut -d '"' -f1`
				Node_3_IP=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | awk -F'ip="' '{print $2}' | cut -d '"' -f1`;;
			2)
				Architecture=2+2
				NumberOfAllNodes=4
                                Node_3_Name=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | sed -n 1p | awk -F'name="' '{print $2}' | cut -d '"' -f1`
                                Node_3_IP=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | sed -n 1p |awk -F'ip="' '{print $2}' | cut -d '"' -f1`
                                Node_4_Name=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | sed -n 2p | awk -F'name="' '{print $2}' | cut -d '"' -f1`
                                Node_4_IP=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | sed -n 2p | awk -F'ip="' '{print $2}' | cut -d '"' -f1`;;
			esac
			return 0
		fi
	fi

	if [ "$ROLE" == "sir" ]
        then
                NumberOfSIRNode=`grep "SIRNode ip" $ISHOME/etc/${HOSTNAME}/sircluster.conf| grep active | wc -l`
		NumberOfSIRStandby=`grep "SIRNode ip" $ISHOME/etc/${HOSTNAME}/sircluster.conf| grep standby | wc -l`
		if [ -z $NumberOfSIRNode ] || [[ $NumberOfSIRNode -eq 0 ]]
		then
			Architecture="Unfinished Architecture"
			NumberOfAllNodes=1
			return 0
		fi
		Node_1_IP=`grep "SIRNode ip" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep $HOSTNAME | awk -F'ip="' '{print $2}' | cut -d '"' -f1`
		if [[ $NumberOfSIRNode -eq 1 ]]
		then
			Architecture=2+1
			NumberOfAllNodes=3
			Node_2_Name=`grep "Primary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'Hostname=\"' '{print $2}' | cut -d '"' -f1`
			Node_2_IP=`grep "Primary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'ip=\"' '{print $2}' | cut -d '"' -f1`
			Node_3_Name=`grep "Secondary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'Hostname=\"' '{print $2}' | cut -d '"' -f1`
			Node_3_IP=`grep "Secondary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'ip=\"' '{print $2}' | cut -d '"' -f1`
			return 0
		elif [[ $NumberOfSIRNode -eq 2 ]]
		then
			Node_2_IP=`grep "SIRNode ip" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep active | grep -v ${Node_1_IP} | sed -n 1p| awk -F'ip="' '{print $2}' | cut -d '"' -f1`
			Node_2_Name=`grep "SIRNode ip" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep active | grep -v ${Node_1_IP} | sed -n 1p | awk -F'name="' '{print $2}' | cut -d '"' -f1`
                        Node_3_Name=`grep "Primary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'Hostname=\"' '{print $2}' | cut -d '"' -f1`
                        Node_3_IP=`grep "Primary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'ip=\"' '{print $2}' | cut -d '"' -f1`
                        Node_4_Name=`grep "Secondary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'Hostname=\"' '{print $2}' | cut -d '"' -f1`
                        Node_4_IP=`grep "Secondary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'ip=\"' '{print $2}' | cut -d '"' -f1`
			if [[ $NumberOfSIRStandby -eq 0 ]]
			then
				Architecture=2+2
				NumberOfAllNodes=4
				return 0
			else
				Architecture=2+2andStandby
				NumberOfAllNodes=5
				Node_5_Name=`grep "SIRNode ip" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep -v ${Node_1_IP} | grep -v ${Node_2_IP} | sed -n 1p | awk -F'name="' '{print $2}' | cut -d '"' -f1`
				Node_5_IP=`grep "SIRNode ip" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep -v ${Node_1_IP} | grep -v ${Node_2_IP} | sed -n 1p | awk -F'ip="' '{print $2}' | cut -d '"' -f1`
				return 0
			fi
		fi
	fi
}



# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_Get_Mega_Log
# Description: Get MegaRAID Event Log
# Parameter:void
# Return:
# **************************************************************************** #

VTL_Get_Mega_Log ()
{
	MegaLog=$DIR/megalog
	ProcessDoneLog=$DIR/ProcessDoneLog
	rm -f $ProcessDoneLog
	InitFile $MegaLog

	$MegaDIR/MegaCli64 -AdpAlILog -aALL >> $MegaLog
	wait
	touch $ProcessDoneLog
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckBaseInfo
# Description: Check Basic Information
# Parameter:
# Return:
# **************************************************************************** #

VTL_CheckBaseInfo ()
{
	InitFile $CHECK_RESULT_FILE

	HOSTNAME=`hostname`
	DEVICE_MODEL="OceanSpace VTL3500"
	VTL_VERSION=`rpm -qa | egrep "vtl|sir"`
	PRODUCT_VERSION="V100R002C01SPC005"
	KERNEL_VERSION=`uname -r`
	OS_VERSION=`cat /etc/issue | egrep "Linux|release"`
	BOMCODE="0235G6PU,0235G6PV"
	NodeName=`hostname`
	CTRLIP=`ifconfig eth0 | grep "inet addr:" | sed -e 's/^.*inet addr://g;s/ Bcast.*$//g'`
	IPMIADDR=`ipmitool -I open lan print 1 | grep "IP Address" | grep -v Source | cut -d':' -f2 | sed -e 's/ //g'`
	DISK_NUMBER=`$MegaDIR/MegaCli64 -PDList -aAll | grep Slot | wc -l`
	RAID_NUMBER=`$MegaDIR/MegaCli64 -cfgdsply -aALL | grep "Number of DISK GROUPS" | cut -d ':' -f2 | sed 's/ //g'`
	VD_NUMBER=`$MegaDIR/MegaCli64 -AdpAllInfo -aAll | grep "Virtual Drives" | cut -d ':' -f2 | sed 's/ //g'`

	VTL_Get_Mega_Log &

	local Memtmp=`grep MemTotal /proc/meminfo | awk '{print $2}'`
	local Mem=`awk 'BEGIN{printf "%d",'$Memtmp'/1048576}'`
	local cpumodel=`grep "model name" /proc/cpuinfo | uniq | awk -F: '{print $2}'`
	
	echo "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" >>$CHECK_RESULT_FILE
	echo "<device>" >>$CHECK_RESULT_FILE
	echo "	<sysinfo>" >>$CHECK_RESULT_FILE
	echo "		<field name=\"systemname\">$HOSTNAME</field>" >>$CHECK_RESULT_FILE
	echo "		<field name=\"model\">${DEVICE_MODEL}</field>" >>$CHECK_RESULT_FILE
	echo "		<field name=\"version\">${PRODUCT_VERSION}</field>" >>$CHECK_RESULT_FILE
	echo "		<field name=\"vtlversion\">${VTL_VERSION}</field>" >>$CHECK_RESULT_FILE
	echo "		<field name=\"osversion\">$OS_VERSION</field>" >>$CHECK_RESULT_FILE
	echo "		<field name=\"kernelversion\">$KERNEL_VERSION</field>" >>$CHECK_RESULT_FILE
	echo "		<field name=\"systemtime\">`date`</field>" >>$CHECK_RESULT_FILE
	        echo "		<field name=\"memtotal\">${Mem}GB</field>" >>$CHECK_RESULT_FILE
        echo "		<field name=\"cpumodel\">$cpumodel</field>" >>$CHECK_RESULT_FILE
        echo "		<field name=\"disknumber\">${DISK_NUMBER}</field>" >>$CHECK_RESULT_FILE
        echo "		<field name=\"raidnumber\">${RAID_NUMBER}</field>" >>$CHECK_RESULT_FILE
        echo "		<field name=\"vdnumber\">${VD_NUMBER}</field>" >>$CHECK_RESULT_FILE
	echo "		<field name=\"ipmiip\">${IPMIADDR}</field>" >>$CHECK_RESULT_FILE
	echo "	</sysinfo>" >>$CHECK_RESULT_FILE
	echo "	<ctrlinfo>" >>$CHECK_RESULT_FILE
        echo "		<controller>" >>$CHECK_RESULT_FILE
        echo "			<field name=\"nodename\">${NodeName}</field>" >>$CHECK_RESULT_FILE
        echo "			<field name=\"ctrlip\">${CTRLIP}</field>" >>$CHECK_RESULT_FILE
        echo "		</controller>" >>$CHECK_RESULT_FILE
	echo "	</ctrlinfo>" >>$CHECK_RESULT_FILE
	echo "</device>" >>$CHECK_RESULT_FILE
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckVTLStatus
# Description: Check VTL Process status
# Parameter:ID:0
# Return:
# **************************************************************************** #

VTL_CheckVTLStatus ()
{
	InitFile $CHECK_RESULT_FILE
	
	WriteResultHeaderToXml "$1"

	local tempnum=`$ROLE status | grep STOP | wc -l`

	if [[ $tempnum -ge 1 ]]
	then
		WriteErrorItemResultToXml "$10" 0 "VTL status||VTL not running"
	else
		WriteNormalItemResultToXml "$10" 0 "VTL Status||VTL running"
	fi

	WriteResultEndToXml

	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckTomcatStatus
# Description: Check Tomcat Service status,It is from the webpage downloading ISM necessary condition
# Parameter:ID:1
# Return:
# **************************************************************************** #

VTL_CheckTomcatStatus ()
{
	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1

	local tmpret=`ps aux | grep tomcat | grep -v grep | wc -l`

	if [[ $tmpret -lt 1 ]]
	then
		WriteErrorItemResultToXml "$10" 0 "ISM Not Downloadable"
	else
		[ -f /home/tools/ISM/ISMforVTL6000.exe ]
		if [ $? -eq 0 ]
		then
			WriteNormalItemResultToXml "$10" 0 "ISM Downloadable"
		else
			WriteErrorItemResultToXml "$10" 0 "ISM Not Downloadable"
		fi
	fi
	WriteResultEndToXml
	return 0
}



# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckKdumpStatus
# Description: Check Kdump Service status
# Parameter:ID:2
# Return:
# **************************************************************************** #

VTL_CheckKdumpStatus ()
{
	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1

	grep -q crashkernel /proc/cmdline 
	tmpret1=$?
	
	service kdump status | grep -q "Kdump is operational"
	tmpret2=$?

	tmpret=$[$tmpret1 + $tmpret2]

	if [[ $tmpret -eq 0 ]]
	then
		WriteNormalItemResultToXml "$10" 0 "Kdump Status running"
	else
		WriteErrorItemResultToXml "$10" 0 "Kdump Status not running"
	fi
	WriteResultEndToXml
	return 0
}



# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckFCStatus
# Description: Check FC Device status
# Parameter:ID:3
# Return:
# **************************************************************************** #

VTL_CheckFCStatus ()
{
	local count=0
	FC_DEVICE_DIR=/proc/scsi/qla2xxx

	InitFile $CHECK_RESULT_FILE

	WriteResultHeaderToXml "$1"

	templist=`ls $FC_DEVICE_DIR` 2>/dev/null
    if [[ $? -ne 0 ]]
    then
        WriteErrorItemResultToXml "$10" 0 "FC Device||No Connection"
        WriteResultEndToXml
        return 0
    else
        for i in $templist
        do
                tempnum=`grep "Link_Data_Rate" $FC_DEVICE_DIR/$i | awk '{print $3}'`
                hbanum=`echo ${i:2:1}`
                modenum=`grep ^connection $ISHOME/etc/fshba.conf | grep "hba$hbanum=" | cut -d"=" -f2`
                mode=`VTL_modeTransfer $modenum`
                local msg="$msg||HBA_$i:Rate=${tempnum}GB:Mode=${mode}"
        done
        WriteNormalItemResultToXml "$10" 0 "$msg"

        local tmpstr=`grep "FibreChannelOption enabled" $IS_CONF | awk -F\" '{print $2}'`
        if [ "$tmpstr" == "true" ]
        then
            WriteNormalItemResultToXml "$11" 0 "FC Device||FC Option enabled"
        else
            WriteNormalItemResultToXml "$11" 0 "FC Device||FC Option unenabled"
        fi
        WriteResultEndToXml
    fi
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckMemStatus
# Description: Check the System memory occupancy rate,and MemTotal Is able to Dedupcation
# Parameter:ID:4
# Return:
# **************************************************************************** #

VTL_CheckMemIdel ()
{
	local MEM_THRESHOLD=2

	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1

	MemTotal=`grep MemTotal /proc/meminfo | awk '{print $2}'`
	MemFree=`grep MemFree /proc/meminfo | awk '{print $2}'`
	local tmpnum=`awk 'BEGIN{printf "%d",'$MemFree'/'$MemTotal'*100}'`
    Swap_used=`free | grep Swap | awk '{print $3}'`
	if [[ $Swap_used -gt 0 ]]
	then
			WriteErrorItemResultToXml "$10" 0 "Swap_used||${Swap_used}"
	else
			WriteNormalItemResultToXml "$10" 0 "Swap_used||${Swap_used}"
	fi

	MemTotal=$[$MemTotal/1024]
	if [ "$ROLE" == "sir" ]
	then
		sirdatatmp=$DIR/sirdatatmp
		InitFile $sirdatatmp

		local SIRDATA=0
		iscon getvdevlist -s 127.0.0.1 | grep SIR | awk '{print $3}' > $sirdatatmp
		local sirdatanumber=`cat $sirdatatmp | wc -l`
		for ((i=1; i<=sirdatanumber; i++))
		do
			local tmpdata=`sed -n "$i"p $sirdatatmp`
			SIRDATA=$[$SIRDATA+$tmpdata]
		done

		local MemMin=$[$SIRDATA/1024]
		local MemThreshould=$[6*1024]
		if [[ $MemTotal -gt $MemMin ]] && [[ $MemTotal -gt $MemThreshould ]]
		then
			WriteNormalItemResultToXml "$11" 0 "MemTotal for Dedup||${MemTotal}MB"
		else
			WriteErrorItemResultToXml "$11" 0 "MemTotal for Dedup||${MemTotal}MB"
		fi
		rm -f $sirdatatmp
	else
		WriteNormalItemResultToXml "$11" 0 "MemTotal||${MemTotal}MB"
	fi

	WriteResultEndToXml
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckPowerSupply
# Description: Check Power Supply Status
# Parameter:ID:5
# Return:
# **************************************************************************** #

VTL_CheckPowerSupply ()
{
        InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1
	
	local PS_Number=`ipmitool -I open sdr type "Power Supply" | grep "Presence detected" | wc -l`
	if [[ ${PS_Number} -le 1 ]]
	then
		WriteErrorItemResultToXml "$10" 0 "Power Supply||No Redundancy"
	else
		WriteNormalItemResultToXml "$10" 0 "Power Supply||Redundancy"
	fi

	local VoltageStauts=`ipmitool -I open sdr type "Voltage" | awk -F'|' '{print $3}' | egrep -v "ok|ns" | wc -l`
	if [[ $VoltageStatus -gt 0 ]]
	then
		WriteErrorItemResultToXml "$11" 0 "Voltage||abnormal"
	else
		WriteNormalItemResultToXml "$11" 0 "Voltage||normal"
	fi

	WriteResultEndToXml
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckHostname
# Description: Check the Hostname
# Parameter:ID:6
# Return:
# **************************************************************************** #

VTL_CheckHostname ()
{
	CTRLIP=`ifconfig eth0 | grep "inet addr:" | sed -e 's/^.*inet addr://g;s/ Bcast.*$//g'`

	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1
	
    local HOSTNAME1=`hostname`
    local HOSTNAME3=`grep "SANBridge name=" $IS_CONF | awk -F '<SANBridge name=' '{print $2}' | cut -d ' ' -f 1 | sed -e 's/\"//g'`
    msg="\$HOSTNAME: $HOSTNAME1||ipstor.conf: $HOSTNAME3"
    if [ "$HOSTNAME1" == "$HOSTNAME3" ]
    then
        WriteNormalItemResultToXml "$10" 0 "$msg"
    else
        WriteErrorItemResultToXml "$10" 0 "$msg"
    fi
    WriteResultEndToXml

	return 0		
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckXinetdStatus
# Description: Check xinetd server status
# Parameter:ID:
# Return:
# **************************************************************************** #

VTL_CheckXinetdStatus ()
{
	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1

	service xinetd status &>/dev/null
	if [ $? -eq 0 ]
	then
		WriteNormalItemResultToXml "$10" 0 "[Xinetd]:OK"
	else
		WriteErrorItemResultToXml "$10" 0 "[Xinetd]:STOPPED"
	fi	
	WriteResultEndToXml
	return 0
}



# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckHBADriver
# Description: Check HBA Driver
# Parameter:ID:7
# Return:
# **************************************************************************** #

VTL_CheckHBADriver ()
{
        InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1

	local Driver_NPIV=`grep NPIV /proc/scsi/qla2xxx/* | cut -d'=' -f2 | sed -e 's/ //g' | uniq`
	if [ "$Driver_NPIV" == "Disabled" ]
	then
		WriteNormalItemResultToXml "$10" 0 "HBA Driver||Qlogic"
	else
		WriteNormalItemResultToXml "$10" 0 "HBA Driver||QlogicNPIV"
	fi
        WriteResultEndToXml
        return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CPULoadCheck
# Description: Check Load of CPU
# Parameter:ID:8
# Return:
# **************************************************************************** #

VTL_CPULoadCheck ()
{
	local CPU_THRESHOLD=10
	local CPU_TEMP_THRESHOLD=99

	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1

	CPUIdle=`sar 3 1 | grep "Average" |awk '{print $8}' |awk -F. '{print $1}'`
	local msg="CPU Idle||${CPUIdle}%"

	if [[ $CPUIdle -gt $CPU_THRESHOLD ]]
	then
		WriteNormalItemResultToXml "$10" 0 "$msg"
	else
		WriteErrorItemResultToXml "$10" 0 "$msg"
	fi

	ipmitool -I open sdr type "Temperature" | grep CPU | awk -F '|' '{print $3}' | grep -q ok
	if [[ $? -eq 0 ]]
	then
		WriteNormalItemResultToXml "$11" 0 "CPU Temprature ||normal"
	else
		WriteErrorItemResultToXml "$11" 0 "CPU Temprature ||abnormal"
	fi
	WriteResultEndToXml
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_EthLinkCheck
# Description: Check EthLink Status
# Parameter:ID:9
# Return:
# **************************************************************************** #

VTL_EthLinkCheck ()
{
    local nic

    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $1
    local eth_ip=`ifconfig | sed '/HWaddr.*$/N; s/\n//g' | grep "inet addr" | grep -v 127.0.0.1 | sed 's/Link.*inet addr://g' | sed 's/://g' | awk '{print $1":"$2}'`
    local msg=`echo $eth_ip | sed 's/ /||/g'`
    local NICLIST=`ifconfig | sed '/HWaddr.*$/N; s/\n//g' | grep "inet addr" | grep -v 127.0.0.1 | awk '{print $1}'`
    for nic in $NICLIST
    do
        local Linked=`ethtool ${nic} |grep "Link detected" |awk '{print $3}'`
        if [ "$Linked" == "no" ]
        then
            WriteErrorItemResultToXml "$10" 0 "${msg}"
            WriteResultEndToXml
            return 0
        fi
    done
    WriteNormalItemResultToXml "$10" 0 "${msg}"


	WriteResultEndToXml
	return 0			
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_HardDiskCheck
# Description: Check Performance and Load of HardDisk
# Parameter:ID:10
# Return:
# **************************************************************************** #

VTL_HardDiskCheck ()
{
	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1
	local HD_USED_THRESHOLD=75

	local MOUNT_POINT=`df / | sed -n 2p | awk '{print $1}'`
	local HD_USED=`df / | sed -n 2p | awk '{print $5}' | sed -e 's/%//g'`

	if [[ ${HD_USED} -gt ${HD_USED_THRESHOLD} ]]
	then
		WriteErrorItemResultToXml "$10" 0 "HardDiskUsed=${HD_USED}%"
	else
		WriteNormalItemResultToXml "$10" 0 "HardDiskUsed=${HD_USED}%"
	fi

	local DEVICE_READ_TIMINGS=`hdparm -t $MOUNT_POINT | grep "buffered disk read" | awk -F"seconds = " '{print $2}'`
	local CACHE_READ_TIMINGS=`hdparm -T $MOUNT_POINT | grep "cached reads" | awk -F"seconds = " '{print $2}'`
	WriteNormalItemResultToXml "$11" 0 "DeviceReadTimings=${DEVICE_READ_TIMINGS}||CacheReadTimings=${CACHE_READ_TIMINGS}"

        Await_File=$DIR/await_file
        InitFile $Await_File
        local count=0
        local r_msg=""
        iostat -x | grep sd | awk '{print $1" "$10}' >> $Await_File
        Await_List=`cat $Await_File | awk '{print $2}' | awk 'ORS=" "{print $0}'`
        for i in $Await_List
        do
		local num=`awk 'BEGIN{ print ('$i' > 100); }'`
                if [[ $num -eq 1 ]]
                then
                        r_msg=`grep $i $Await_File`"ms""||"$r_msg
                        ((count=count+1))
                fi
        done
        if [[ $count -gt 0 ]]
        then
                WriteErrorItemResultToXml "$16" 0 "$r_msg"
        else
                WriteNormalItemResultToXml "$16" 0 "Disk Response||ok"
        fi

	$MegaDIR/MegaCli64 -LDInfo -Lall -aALL | grep "RAID Level" | awk -F':' '{print $2}' | cut -d ',' -f1 | uniq | grep -q "Primary-1"
	if [[ $? -ne 0 ]]
	then
		WriteErrorItemResultToXml "$12" 0 "RAID_LEVEL||Wrong"
	else
		WriteNormalItemResultToXml "$12" 0 "RAID_LEVEL||OK"
	fi

	HD_TEMP_MSG=`ipmitool -I open sdr type "Temperature" | grep SCSI | awk -F'|' '{print $5}'`
	HD_TEMP_STATUS=`ipmitool -I open sdr type "Temperature" | grep SCSI | grep -v ok | wc -l`
	if [[ $HD_TEMP_STATUS -gt 0 ]]
	then
		WriteErrorItemResultToXml "$13" 0 "HardDiskTemperature||$HD_TEMP_MSG"
	else
		WriteNormalItemResultToXml "$13" 0 "HardDiskTemperature||$HD_TEMP_MSG"
	fi
	
	HD_DEGRAED=`$MegaDIR/MegaCli64 -AdpAllinfo -aALL | grep Degraded | cut -d ':' -f2 | sed 's/ //g'`
	HD_OFFLINE=`$MegaDIR/MegaCli64 -AdpAllinfo -aALL | grep ffline | egrep -v "Force|Cache" | cut -d ':' -f2 | sed 's/ //g'`
	if [[ $HD_DEGRAED -ne 0 || $HD_OFFLINE -ne 0 ]]
	then
		WriteErrorItemResultToXml "$14" 0 "RaidGroupStatus||abnormal"
	else
		WriteNormalItemResultToXml "$14" 0 "RaidGroupStatus||normal"
	fi

	$MegaDIR/MegaCli64 -PDList -aAll | grep "Firmware state" | cut -d ':' -f2 | grep -q Hotspare
	HD_HSP=$?

	DISK_NUMBER=`$MegaDIR/MegaCli64 -PDList -aAll | grep Slot | wc -l`
	if [[ $HD_HSP -ne 0 ]] && [[ $DISK_NUMBER -ge 10 ]]
	then
		WriteErrorItemResultToXml "$15" 0 "HotSpare||abnormal"
	else
		WriteNormalItemResultToXml "$15" 0 "HotSpare||normal"
	fi

	WriteResultEndToXml
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckVTL
# Description: Check VTL Configuration
# Parameter:ID:11
# Return:
# **************************************************************************** #

VTL_CheckVTL ()
{
	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1
    local LICENSE_TEST=`iscon getlicense -s 127.0.0.1 | grep Keycode | wc -l`
    local LICENSE_msg=`iscon getlicense -s 127.0.0.1 | egrep "Standard|Trial" | awk 'ORS="||"{print $1":Registration:"$3}'`
    if [[ $LICENSE_TEST -eq 1 ]]
    then
        WriteNormalItemResultToXml "$10" 0 "${LICENSE_msg}"
    else
        WriteErrorItemResultToXml "$10" 0 "${LICENSE_msg}"
    fi
    echo ${LICENSE_msg} | grep -q Done
    if [[ $? -eq 0 ]]
    then
        WriteNormalItemResultToXml "$11" 0 "${LICENSE_msg}"
    else
        WriteErrorItemResultToXml "$11" 0 "${LICENSE_msg}"
    fi

	WriteResultEndToXml
	return 0
}

# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckNBU
# Description: Check NBU Configuration
# Parameter:ID:12
# Return:
# **************************************************************************** #

VTL_CheckNBU ()
{
	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1

	local plist="nbsl bprd vmd bpcompatd nbstserv nbsvcmon nbars NB_dbsrv bpdbm nbjm nbrmms nbrb nbemm nbpem nbevtmgr"
	local count=0
	for i in $plist
	do
		local num=`ps aux | grep $i | grep -v grep | wc -l`
		if [[ $num -lt 1 ]]
		then
			(( count=count+1 ))
			local msg="$msg||Process "$i" not running"
		fi
	done
	if [[ $count -eq 0 ]]
	then
		WriteNormalItemResultToXml "$10" 0 "[NBU Process]||ALL RUN"
	else
		WriteErrorItemResultToXml "$10" 0 "$msg"
	fi

	grep -q "^AllowRemoteRoot=true" /etc/gdm/custom.conf
	local ret=$?
	grep -q "Enable=true" /etc/gdm/custom.conf
	(( ret=ret+$? ))
	local num=`grep initdefault: /etc/inittab | awk -F: '{print $2}'`
	if [[ $ret -eq 0 ]] && [[ $num -eq 5 ]]
	then
		WriteNormalItemResultToXml "$11" 0 "[AllowRemoteX11]||YES"
	else
		WriteErrorItemResultToXml "$11" 0 "[AllowRemoteX11]||NO"
	fi
	
	WriteResultEndToXml
	return 0		
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_AmbientTemp
# Description: Check AmbientTemp
# Parameter:ID:13
# Return:
# **************************************************************************** #

VTL_AmbientTemp ()
{
	local AmbientTemp_THRESHOLD=50

	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1
	
	Env_Temp=`ipmitool -I open sdr type "Temperature" | grep Env | grep -v SCSI | grep -v ok | wc -l`
	if [[ $Env_Temp -ne 0 ]]
	then
		WriteErrorItemResultToXml "$10" 0 "Enviroment Temp||abnormal"
	else
		WriteNormalItemResultToXml "$10" 0 "Enviroment Temp||normal"
	fi

        WriteResultEndToXml
        return 0	
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_FANCheck
# Description: Check FAN Status
# Parameter:ID:14
# Return:
# **************************************************************************** #

VTL_CheckFAN ()
{
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $1

	local FAN_NUMBER_DEFAULT=4
	local IS_ERROR=0
	local msg=""
	local FAN_LIST=`ipmitool sdr | grep FAN | grep RPM | sed 's/ RPM.*$//g' | sed 's/ .*|/:/g'`
	local FAN_NUMBER=`echo $FAN_LIST | grep -o FAN | wc -l`
	local msg=`echo "$FAN_LIST" | sed 's/_RPM.*$//g' | awk 'ORS="||"{print $0}'`
	if [[ $FAN_NUMBER -lt $FAN_NUMBER_DEFAULT ]]
	then
		WriteErrorItemResultToXml "$10" 0 "$msg"
	else
		WriteNormalItemResultToXml "$10" 0 "$msg"
	fi

	local rpm_msg=`echo "$FAN_LIST" | awk 'ORS="||"{print $0}'`
	WriteNormalItemResultToXml "$11" 0 "${rpm_msg}"
	
	WriteResultEndToXml
    return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckDatabase
# Description: Check Database Status
# Parameter:ID:15
# Return:
# **************************************************************************** #

VTL_CheckDatabase ()
{
        InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1

	if [ "$ROLE" == "sir" ]
	then
		WriteNormalItemResultToXml "$10" 0 "SIR No Need Check"
		WriteNormalItemResultToXml "$11" 0 "SIR No Need Check"
		WriteResultEndToXml
		return 0
	fi

	local quorumGuid=`sed -n -e '/<VirtualDev id=.* type="TLE"/,/<\/VirtualDev>/ { p }' $IS_CONF | grep Umap | head -n 1 | awk -F ' guid="' '{ print $2 }' | cut -d'"' -f1`
	TLE_ENABLED=`grep TLEOption $IS_CONF | awk -F'enabled="' '{print $2}' | cut -d'"' -f1`

	if [ "$TLE_ENABLED" != "true" ]
	then
		WriteErrorItemResultToXml "$10" 0 "TLE Enabled||No"
		WriteErrorItemResultToXml "$11" 0 "TLE Enabled||No"
		WriteResultEndToXml
		return 0
	else
		WriteNormalItemResultToXml "$10" 0 "TLE Enabled||yes"
	fi

	local TLE_ONLINE=`grep $quorumGuid $IS_CONF | grep "LSI:MegaRAID" | wc -l`
	if [[ $TLE_ONLINE -lt 1 ]]
	then
		WriteErrorItemResultToXml "$11" 0 "TLE Online||No"
	else
		WriteNormalItemResultToXml "$11" 0 "TLE Online||Yes"
	fi
	WriteResultEndToXml

	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckInVault
# Description: Check If Tape In Vault
# Parameter:ID:16
# Return:
# **************************************************************************** #

VTL_CheckInVault ()
{
        InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1

	TLE_ENABLED=`grep TLEOption $IS_CONF | awk -F'enabled="' '{print $2}' | cut -d'"' -f1`


	if [ "$TLE_ENABLED" != "true" ]
	then
		WriteNormalItemResultToXml "$10" 0 "NoDatabase"
		WriteResultEndToXml
		return 0
	fi
	
	if [ "$ROLE" == "sir" ]
        then
                WriteNormalItemResultToXml "$10" 0 "SIR No Need Check"
                WriteResultEndToXml
                return 0
        fi

	local TAPEINVAULT=`iscon getvdevlist -s 127.0.0.1 | sed -n '/Virtual Tapes In Vault/,/Physical Libraries/'p | grep None | wc -l`
	if [[ $TAPEINVAULT -eq 1 ]]
	then
		WriteNormalItemResultToXml "$10" 0 "TapeInVault||No"
	else
		WriteErrorItemResultToXml "$10" 0 "TapeInVault||Yes"
	fi

	WriteResultEndToXml
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckHA
# Description: Check Failover Status
# Parameter:ID:17
# Return:
# **************************************************************************** #

VTL_CheckHA ()
{
        InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1

	local FailoverState=`iscon getfailoverstatus -s 127.0.0.1 | grep "Failover State" | grep "has occurred" | wc -l`
	if [[ $FailoverState -eq 1 ]]
	then
		WriteErrorItemResultToXml "$10" 0 "Failover||occurred"
	else
		WriteNormalItemResultToXml "$10" 0 "Failover||Normal"
	fi

	WriteResultEndToXml
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckSIRAssociate
# Description: Check SIR associate
# Parameter:ID:18
# Return:
# **************************************************************************** #

VTL_CheckSIRAssociate ()
{
	InitFile $CHECK_RESULT_FILE
	WriteResultHeaderToXml $1
	
	if [ "$ROLE" == "vtl" ]
	then
		local SirClusterName=`grep "SirCluster name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | awk -F'SirCluster name="' '{print $2}' | awk -F'"' '{print $1}'`
		if [ -z $SirClusterName ]
		then
			WriteErrorItemResultToXml "$10" 0 "SIR Associate||No Associate"
		else
			WriteNormalItemResultToXml "$10" 0 "SIR Associate||Associated"
		fi
	fi

	if [ "$ROLE" == "sir" ]
	then
		local VTLAssociate=`sed -n '/<VTL>/,/<\/VTL>/'p $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'ip="' '{print $2}' | awk -F'"' '{print $1}' | sed '/^$/'d`
		if [ -z "$VTLAssociate" ]
		then
			WriteErrorItemResultToXml "$10" 0 "SIR Associate||No Associate"
		else
			WriteNormalItemResultToXml "$10" 0 "SIR Associate||Associated"
		fi
	fi

	WriteResultEndToXml
	return 0		
}

# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckSIRIPMI
# Description: Check SIR cluster + standby IPMI address
# Parameter:ID:19
# Return:
# **************************************************************************** #

VTL_CheckSIRIPMI ()
{
	InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1

	NumberOfSIRStandby=`grep "SIRNode ip" $ISHOME/etc/${HOSTNAME}/sircluster.conf| grep standby | wc -l`
        NumberOfSIRNode=`grep "SIRNode ip" $ISHOME/etc/${HOSTNAME}/sircluster.conf| grep active | wc -l`
				
	if [[ $NumberOfSIRNode -lt 2 ]] || [[ $NumberOfSIRStandby -lt 1 ]]
	then
		WriteNormalItemResultToXml "$10" 0 "VTL/2+1 No Need Check"
	else
		local IPMINetSeg=`grep ipmiIP $ISHOME/etc/${HOSTNAME}/sircluster.conf | awk -F'ipmiIP="' '{print $2}' | awk -F'"' '{print $1}' | awk -F '.' '{print $1$2$3}' | uniq | wc -l`
		local IPMIIPList=`grep ipmiIP $ISHOME/etc/${HOSTNAME}/sircluster.conf | awk -F'ipmiIP="' '{print $2}' | awk -F'"' 'ORS=" "{print $1}'`
		if [[ $IPMINetSeg -ne 1 ]]
		then
			WriteErrorItemResultToXml "$10" 0 "IPMINetSeg||$IPMIIPList"
		else
			WriteNormalItemResultToXml "$10" 0 "IPMINetSeg||$IPMIIPList"
		fi
	fi
	
	WriteResultEndToXml
	return 0
}



# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckStorage
# Description: Check Storage Status
# Parameter:ID:20
# Return:
# **************************************************************************** #

VTL_CheckStorage ()
{
        InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1

	local Lunlist=`iscon getpdevinfo -s 127.0.0.1 | egrep "LSI|Data" | awk '{print $3}' | sort -n | uniq`
        local Lunnumber=`echo $Lunlist | awk '{print NF}'`
	local LunIsNew=`grep LSI:MegaRAID $IS_CONF | grep isNew="true" | wc -l`

	
	if [[ $Lunnumber -ge 1 ]] && [[ $LunIsNew -eq 0 ]]
	then
        	WriteNormalItemResultToXml "$10" 0 "Lunnumber||$Lunnumber"
	else
		WriteErrorItemResultToXml "$10" 0 "Lunnumber||$Lunnumber"
	fi
	
	if [ "$ROLE" == "vtl" ]
	then
		local OwnerIsWrong=0
		local hostlist=$DIR/hostlist
		InitFile $hostlist
        	ownerlist=`sed -n '/PhysicalDevs/,/\/PhysicalDevs/'p $IS_CONF | grep owner | awk -F"owner=" '{print $2}' | cut -d' ' -f1 | sed -e 's/\"//g'`
        	grep "SANBridge name=" $IS_CONF | awk -F '<SANBridge name=' '{print $2}' | cut -d ' ' -f 1 | sed -e 's/\"//g' >> $hostlist
        	grep "secondaryServer=" $IS_CONF | awk -F 'secondaryServer=' '{print $2}' | cut -d ' ' -f 1 | sed -e 's/\"//g' >> $hostlist
        	if [ -n "$ownerlist" ]
        	then
                	for i in $ownerlist
                	do
                        	grep -q $i $hostlist
                        	if [ $? -ne 0 ]
                        	then
                                	WriteErrorItemResultToXml "$11" 0 "LunOwner||Have Wrong Owner"
					OwnerIsWrong=1
					break
                        	fi
                	done
        	fi
        	[[ $OwnerIsWrong -eq 0 ]] && WriteNormalItemResultToXml "$11" 0 "LunOwner||All right"
		rm -f $hostlist
	else
		WriteNormalItemResultToXml "$11" 0 "SIR No Need Check"
	fi

	if [ "$SIR_Option" == "false" ]
	then
		WriteNormalItemResultToXml "$13" 0 "No enabled SIR"
		WriteResultEndToXml 
		return 0
	fi

	local DataLunNumber=`iscon getpdevinfo -s 127.0.0.1 | grep Data | wc -l`

	if [[ $DataLunNumber -le 0 ]]
	then
		WriteErrorItemResultToXml "$13" 0 "[DataLunnumber]||0"
		WriteResultEndToXml
		return 0
	fi
	
	VTL_CheckSIRProportion
	if [[ ${SIR_Proportion} -lt 4 ]]
	then
		WriteErrorItemResultToXml "$13" 0 "IndexLunCapacity||${SIR_Proportion}%"
	else
		WriteNormalItemResultToXml "$13" 0 "IndexLunCapacity||${SIR_Proportion}%"
	fi

	WriteResultEndToXml
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_SIRmonitor
# Description: Check SIR Capacity Usage
# Parameter:ID:21
# Return:
# **************************************************************************** #

VTL_SIRmonitor ()
{

        InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1


	SIR_Monitor=$DIR/sirmonitor
	InitFile $SIR_Monitor

	DEFAULT_REPOSITORY_USAGE=90
	DEFAULT_INDEX_USAGE=90
	DEFAULT_RAM_USAGE=75

	if [ "$SIR_Option" == "false" ]
	then
		WriteNormalItemResultToXml "$10" 0 "SIR No enabled"
		WriteNormalItemResultToXml "$11" 0 "SIR No enabled"
		WriteNormalItemResultToXml "$12" 0 "SIR No enabled"
		WriteResultEndToXml
		return 0
	fi

	sirconfig stat >> $SIR_Monitor
	
	DataStoreInfo=`cat ${SIR_Monitor} | grep ReclaimedSectors`
	HashStoreInfo=`cat ${SIR_Monitor} | grep TotalRecords`
	HashStoreCacheInfo=`cat ${SIR_Monitor} | grep OverflowBucketSize`

	rep_storage_info=`echo "${DataStoreInfo}" | awk 'BEGIN{FS="\""}{printf("%s %s", $6, $4);}'`
	rep_pct=`pct ${rep_storage_info}`
	
	index_storage_info=`echo "${HashStoreInfo}" | awk 'BEGIN{FS="\""}{printf("%s %s", $6, $4);}'`
	idx_pct=`pct ${index_storage_info}`

	RAM_info=`echo "${HashStoreCacheInfo}" | awk 'BEGIN{FS="\""}{printf("%s %s", $14, $4);}'`
	sch_pct=`pct ${RAM_info}`

	if [[ ${rep_pct} -ge ${DEFAULT_REPOSITORY_USAGE} ]]
	then
		WriteErrorItemResultToXml "$10" 0 "Repository Usage||${rep_pct}"
	else
		WriteNormalItemResultToXml "$10" 0 "Repository Usage||${rep_pct}"
	fi

	if [[ ${idx_pct} -ge ${DEFAULT_INDEX_USAGE} ]]
	then
		WriteErrorItemResultToXml "$11" 0 "Index Usage||${idx_pct}"
	else
		WriteNormalItemResultToXml "$11" 0 "Index Usage||${idx_pct}"
	fi

	if [[ ${sch_pct} -ge ${DEFAULT_RAM_USAGE} ]]
	then
		WriteErrorItemResultToXml "$12" 0 "Cache Usage||${sch_pct}"
	else
		WriteNormalItemResultToXml "$12" 0 "Cache Usage||${sch_pct}"
	fi

	WriteResultEndToXml
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckVDAttribute
# Description: Check Virtual Device Attribute
# Parameter:ID:22
# Return:
# **************************************************************************** #

VTL_CheckVDAttribute ()
{
        InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1
	
	Size_msg=`$MegaDIR/MegaCli64 -LDInfo -Lall -aAll | grep Size | grep -v Stripe | cut -d ':' -f2 | awk 'ORS="||"{print "VD"NR" "$0}'`
	WriteNormalItemResultToXml "$10" 0 "${Size_msg}"

	Stripe_Size_msg=`$MegaDIR/MegaCli64 -LDInfo -Lall -aAll | grep Stripe | cut -d ':' -f2 | awk 'ORS="||"{print "VD"NR$0}'`
	WriteNormalItemResultToXml "$11" 0 "${Stripe_Size_msg}"

	Read_Policy_msg=`$MegaDIR/MegaCli64 -LDInfo -Lall -aAll | grep Current | cut -d ',' -f2 | awk 'ORS="||"{print "VD"NR$0}'`
	WriteNormalItemResultToXml "$12" 0 "${Read_Policy_msg}"

	Write_Policy_msg=`$MegaDIR/MegaCli64 -LDInfo -Lall -aAll | grep Current | cut -d ',' -f1 | cut -d ':' -f2 | awk 'ORS="||"{print "VD"NR$0}'`
	WriteNormalItemResultToXml "$13" 0 "${Write_Policy_msg}"

	IO_Policy_msg=`$MegaDIR/MegaCli64 -LDInfo -Lall -aAll | grep Current | cut -d ',' -f3 | awk 'ORS="||"{print "VD"NR$0}'`	
	WriteNormalItemResultToXml "$14" 0 "${IO_Policy_msg}"

	WriteResultEndToXml
	return 0
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckLog
# Description: Check Event Log
# Parameter:ID:23
# Return:
# **************************************************************************** #

VTL_CheckLog ()
{
        InitFile $CHECK_RESULT_FILE
        WriteResultHeaderToXml $1

	while true 
	do
		if [ -f $ProcessDoneLog ]
		then
			rm -f $ProcessDoneLog
			break
		else
			sleep 1
		fi
	done

	grep -q "Predictive failure" $MegaLog 
	if [[ $? -ne 0 ]]
	then
		WriteNormalItemResultToXml "$10" 0 "Predictive failure||No Log"
		WriteResultEndToXml
        	return 0
	fi
	
	local DATE_ONE_WEEK=$((3600*24*7))
	local count=0
	DATE_NOW=`date +%s`
	local Log_Date_List=`grep "Predictive failure" $MegaLog | cut -d':' -f2 | cut -d ',' -f1 | sed 's/ //g' | awk 'ORS=" "{print $0}'`
	for i in $Log_Date_List
	do
		DATE_LOG=`date -d "$i" +%s`
		DATE_DIFF=`awk 'BEGIN{printf "%d",'$DATE_NOW'-'$DATE_LOG'}'`
		if [[ $DATE_DIFF -le $DATE_ONE_WEEK ]]
		then
			local msg_err=$msg_err"||"`grep $i $MegaLog`
			((count=count+1))
		fi
	done
	if [[ $count -eq 0 ]]	
	then
		WriteNormalItemResultToXml "$10" 0 "Predictive failure||No Log"
	else
		WriteErrorItemResultToXml "$10" 0 "Predictive failure||$msg_err"	
	fi

        WriteResultEndToXml
        return 0		
}


# --------------------------  functions  ----------------------- ------------- #
# **************************************************************************** #
# Function Name: VTL_CheckOfflineTape
# Description: Check offline tape
# Parameter:ID:24
# Return:
# **************************************************************************** #

VTL_CheckOfflineTape ()
{
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $1

        offlineTape=`grep "barcode" $ISHOME/etc/$HOSTNAME/rde/Tape*.xml | grep status | sed 's/^.*barcode="\(.*\)" dataReplicated.*status="\(.dataReplicated.*status="\(.*\)" tapeID.*$/\1:\2/g'`
        local msg=`echo $offlineTape | sed 's/ /||/g'`
        echo $msg | grep OFFLINE
        if [[ $? -ne 0 ]]
        then
                WriteNormalItemResultToXml "$10" 0 "No offline tape"
        else
                WriteErrorItemResultToXml "$10" 0 "$msg"
        fi

        WriteResultEndToXml
        return 0
}




################################  main  #########################################

INPUT_PARA_NUM=$#

if [[ $INPUT_PARA_NUM -lt 2 ]]
then
	echo "Usage:"
fi

VTL_CheckAndExecCommand $*
exit 0
