#!/bin/sh
#
#
# File Name:    inspect.sh
# Description:  This scripts is used to ISM Service Tool for VTL6900.
# Input:        sh inspect.sh -c <baseinfo|0...23>
# Output:       xml format to the standard output
# Author:       
# Revise history
# 2012.06.18  initial
# 2014.8.11   add no ipmitool check ,add check nas module
#

ROLE=`cat $ISHOME/etc/role | cut -d'=' -f2`
CHECK_RESULT_FILE=/tmp/check_result.xml
MEGACLI='/opt/MegaRAID/MegaCli/MegaCli64'
MEGA_ADP=0
SAS2IRCU='/home/tools/sas2ircu'
IPMITOOL=`find $(echo $PATH |tr ':' ' ') -name ipmitool 2>/dev/null |awk 'NR==1'`

#*********************************************************************************#
# Function: Discover_Topology
# Description: Get node name, node ip, architecture and node number of vtl topology
# Input Parameters: $1
#   $1=1: This function only get ALL_NODES_NUM and ARCHITECTURE if $1=1.
#   $1=0: 
#*********************************************************************************#
function Discover_Topology ()
{
    local sas_pattern=""
    local role_only="$1"
    local manage_network="eth2"

    lspci | grep "SAS2008" > /dev/null 2>&1 && sas_pattern="SAS"
    lspci | grep "SAS 2208" > /dev/null 2>&1 && sas_pattern="RAID"

    if [ -z "${sas_pattern}" ] ; then
        return 1
    fi
    
    if [ "$role_only" -eq 0 ] ; then
        NODE_1_NAME=`hostname`
        # 1 VTL pattern
        NODE_1_IP=`ifconfig ${manage_network} | grep "inet addr" | awk -F':' '{print $2}' | awk '{print $1}'`
    fi

    if [ "${sas_pattern}" = "RAID" ] ; then
        ALL_NODES_NUM=1
        if [ "${ROLE}" = 'vtls' ] ; then
            ARCHITECTURE="VTL-IS"
        elif [ "${ROLE}" = 'vtl' ] ; then
            ARCHITECTURE="VTL-I"
        else
            echo "ERROR:the role is not identification"
            return 2
        fi
        return 0
    fi

    if [ "${ROLE}" = "vtls" ] ; then
        ARCHITECTURE="VTL-AIO"
        ALL_NODES_NUM=1
        return 0

    elif [ "${ROLE}" = "vtl" ] ; then
        local vtlpartner=`grep "SANBridge name" $IS_CONF | awk -F'groupPartner="' '{print $2}' | awk -F'"' '{print $1}'`
        if [ -z "${vtlpartner}" ] ; then
            ARCHITECTURE="VTL-SL"
            ALL_NODES_NUM=1
            return 0
        fi

        # HA pattern
        if [ "$role_only" -eq 0 ] ; then
            NODE_1_IP=`sed -n '/FailoverPrimary/,/\/FailoverPrimary/'p $IS_CONF \
                | grep clientConnectionIP | awk -F'clientConnectionIP="' '{print $2}' | cut -d'"' -f1`
            NODE_2_NAME=`sed -n '/FailoverPrimary/,/\/FailoverPrimary/'p $IS_CONF \
                | grep secondaryServer | awk -F'secondaryServer="' '{print $2}' | cut -d '"' -f1`
            NODE_2_IP=${vtlpartner}
        fi

        local sir_node_num=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | wc -l`
        case $sir_node_num in
            0)
                ARCHITECTURE="VTL-HA"
                ALL_NODES_NUM=2
                ;;
            1)
                ARCHITECTURE="2+1"
                ALL_NODES_NUM=3
                if [ "$role_only" -eq 0 ] ; then
                    NODE_3_NAME=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | awk -F'name="' '{print $2}' | cut -d '"' -f1`
                    NODE_3_IP=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf | awk -F'ip="' '{print $2}' | cut -d '"' -f1`
                fi
                ;;
            2)
                ARCHITECTURE="2+2"
                ALL_NODES_NUM=4
                if [ "$role_only" -eq 0 ] ; then
                    NODE_3_NAME=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf \
                        | sed -n 1p | awk -F'name="' '{print $2}' | cut -d '"' -f1`
                    NODE_3_IP=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf \
                        | sed -n 1p |awk -F'ip="' '{print $2}' | cut -d '"' -f1`
                    NODE_4_NAME=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf \
                        | sed -n 2p | awk -F'name="' '{print $2}' | cut -d '"' -f1`
                    NODE_4_IP=`grep "SirNode name" $ISHOME/etc/$HOSTNAME/vtlrde.conf \
                        | sed -n 2p | awk -F'ip="' '{print $2}' | cut -d '"' -f1`
                fi
                ;;
            *)
                echo "ERROR: SIR number is $sir_node_num"
                ;;
        esac
        return 0

    elif [ "$ROLE" = "sir" ] ; then
        local sir_node_num=`grep -E "SIRNode ip|SIRNode" $ISHOME/etc/${HOSTNAME}/sircluster.conf | grep active | wc -l`
        local sir_standby_num=`grep -E "SIRNode ip|SIRNode" $ISHOME/etc/${HOSTNAME}/sircluster.conf| grep standby | wc -l`
        # local device ip
        if [ "$role_only" -eq 0 ] ; then
            NODE_1_IP=`grep -E "SIRNode ip|SIRNode" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep $HOSTNAME | awk -F'ip="' '{print $2}' | cut -d '"' -f1`
        fi

        case $sir_node_num in
            1)
                ARCHITECTURE="2+1"
                ALL_NODES_NUM=3
                if [ "$role_only" -eq 0 ] ; then
                    NODE_2_NAME=`grep "Primary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'Hostname=\"' '{print $2}' | cut -d '"' -f1`
                    NODE_2_IP=`grep "Primary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'ip=\"' '{print $2}' | cut -d '"' -f1`
                    NODE_3_NAME=`grep "Secondary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'Hostname=\"' '{print $2}' | cut -d '"' -f1`
                    NODE_3_IP=`grep "Secondary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'ip=\"' '{print $2}' | cut -d '"' -f1`
                fi
                ;;
            2)
                ARCHITECTURE="2+2"
                ALL_NODES_NUM=4
                if [ "$role_only" -eq 0 ] ; then
                    NODE_2_NAME=`grep -E "SIRNode ip|SIRNode" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep active | grep -v ${NODE_1_IP} \
                        | sed -n 1p | awk -F'name="' '{print $2}' | cut -d '"' -f1`
                    NODE_2_IP=`grep -E "SIRNode ip|SIRNode" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep active | grep -v ${NODE_1_IP} \
                        | sed -n 1p | awk -F'ip="' '{print $2}' | cut -d '"' -f1`
                    NODE_3_NAME=`grep "Primary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'Hostname=\"' '{print $2}' | cut -d '"' -f1`
                    NODE_3_IP=`grep "Primary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'ip=\"' '{print $2}' | cut -d '"' -f1`
                    NODE_4_NAME=`grep "Secondary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'Hostname=\"' '{print $2}' | cut -d '"' -f1`
                    NODE_4_IP=`grep "Secondary Hostname" $ISHOME/etc/$HOSTNAME/sircluster.conf | awk -F'ip=\"' '{print $2}' | cut -d '"' -f1`
                fi
                if [ $sir_standby_num -eq 0 ] ; then
                    ARCHITECTURE="2+2"
                    ALL_NODES_NUM=4
                else
                    ARCHITECTURE="2+2andStandby"
                    ALL_NODES_NUM=5
                    if [ "$role_only" -eq 0 ] ; then
                        NODE_5_NAME=`grep -E "SIRNode ip|SIRNode" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep -v ${NODE_1_IP} | grep -v ${NODE_2_IP} \
                            | grep -E '(active|standby)' | sed -n 1p | awk -F'name="' '{print $2}' | cut -d '"' -f1`
                        NODE_5_IP=`grep -E "SIRNode ip|SIRNode" $ISHOME/etc/$HOSTNAME/sircluster.conf | grep -v ${NODE_1_IP} | grep -v ${NODE_2_IP} \
                            | grep -E '(active|standby)' | sed -n 1p | awk -F'ip="' '{print $2}' | cut -d '"' -f1`
                    fi
                fi
                ;;
            *)
                echo "ERROR: SIR number is $sir_node_num"
                ARCHITECTURE="Unfinished Architecture"
                ALL_NODES_NUM=1
                ;;
        esac
        return 0
    fi

}

#*********************************************************************************#
# Function: VTL_CheckBaseInfo
# Description: xml format to output base information
# Input Parameters: 
#*********************************************************************************#
function VTL_CheckBaseInfo ()
{
    InitFile $CHECK_RESULT_FILE

    local device_model="OceanStor VTL6900"
    local VTL_version=`rpm -qa | egrep "vtl|sir"`
    local product_version=`grep "Product Version" /etc/Version | cut -d':' -f2 | sed -e 's/ //g'`
    local kernel_version=`uname -r`
    local OS_version=`cat /etc/redhat-release`
    
    Discover_Topology '0' || return 1

    local mem=`dmidecode | grep -E -A 6 "^Memory Array Mapped Address$" | grep "Range Size" | awk -F':' '{print $2}' | sed 's/^[ \t]*//'`
    local CPU_model=`grep "model name" /proc/cpuinfo | awk -F':' '{if(NR == 1) print $2}' | sed 's/\(^.*\)\(I.*\)/\2/'`
    local node_prefix="" node_name="" node_ip=""
    
    echo "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" >>$CHECK_RESULT_FILE
    echo "<device>" >>$CHECK_RESULT_FILE
    echo "  <sysinfo>" >>$CHECK_RESULT_FILE
    echo "      <field name=\"systemname\">$HOSTNAME</field>" >>$CHECK_RESULT_FILE
    echo "      <field name=\"model\">$device_model</field>" >>$CHECK_RESULT_FILE
    echo "      <field name=\"version\">$product_version</field>" >>$CHECK_RESULT_FILE
    echo "      <field name=\"vtlversion\">$VTL_version</field>" >>$CHECK_RESULT_FILE
    echo "      <field name=\"osversion\">$OS_version</field>" >>$CHECK_RESULT_FILE
    echo "      <field name=\"kernelversion\">$kernel_version</field>" >>$CHECK_RESULT_FILE
    echo "      <field name=\"architecture\">$ARCHITECTURE</field>" >>$CHECK_RESULT_FILE  
    echo "      <field name=\"systemtime\">`date`</field>" >>$CHECK_RESULT_FILE
    echo "      <field name=\"memtotal\">${mem}</field>" >>$CHECK_RESULT_FILE
    echo "      <field name=\"cpumodel\">$CPU_model</field>" >>$CHECK_RESULT_FILE
    echo "  </sysinfo>" >>$CHECK_RESULT_FILE
    echo "  <ctrlinfo>" >>$CHECK_RESULT_FILE

    for (( i=1; i<=$ALL_NODES_NUM; i++))
    do
        node_prefix=NODE_$i
        eval node_name=\$${node_prefix}_NAME
        eval node_ip=\$${node_prefix}_IP
            echo "      <controller>" >>$CHECK_RESULT_FILE
            echo "          <field name=\"nodename\">${node_name}</field>" >>$CHECK_RESULT_FILE
            echo "          <field name=\"ctrlip\">${node_ip}</field>" >>$CHECK_RESULT_FILE
            echo "      </controller>" >>$CHECK_RESULT_FILE
    done
    echo "  </ctrlinfo>" >>$CHECK_RESULT_FILE
    echo "</device>" >>$CHECK_RESULT_FILE
}

#*********************************************************************************#
# Function: InitFile
# Description: create file
# Input Parameters: $1
# $1: $CHECK_RESULT_FILE
#*********************************************************************************#
function InitFile ()
{
    local FileName="$1"
    [ -f $FileName ] && rm -rf $FileName 
    touch $FileName && chmod 640 $FileName
}

#*********************************************************************************#
# Function: WriteResultHeaderToXml
# Description: create header of xml format
# Input Parameters: $1
# $1: check item id 
#*********************************************************************************#
function WriteResultHeaderToXml()
{
    echo "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" >>$CHECK_RESULT_FILE
    echo "<device>" >>$CHECK_RESULT_FILE
    echo "  <cmditem id=\""$1"\">"  >>$CHECK_RESULT_FILE    
}

#*********************************************************************************#
# Function: WriteNormalItemResultToXml
# Description: create normal info with xml format
# Input Parameters: $1 $2 $3
# $1: check item sub-id 
# $2: output type, general conditions, the value is 0.
# $3: common info, each line info separated by +
#*********************************************************************************#
function WriteNormalItemResultToXml()
{
    echo "      <item id=\""$1"\">" >>$CHECK_RESULT_FILE
    local msg="$3"
    if [ -z "$msg" ] ; then
        return 1
    fi
    local param_num=`echo $msg | awk -F'+' '{print NF}'`
    for((i=1; i<="$param_num"; i++))
    do
        local param=\$$i
        local param_val=`echo "$msg" | awk -F'+' '{print '$param'}' | sed 's/^[ \t]*//'`
        if [ -n "$param_val" ] ; then
            echo "          <commonvalue type=\"$2\">"$param_val"</commonvalue>" >>$CHECK_RESULT_FILE
        fi
    done
    echo "          <judge>"0"</judge>" >>$CHECK_RESULT_FILE
    echo "      </item>" >>$CHECK_RESULT_FILE
}

#*********************************************************************************#
# Function: WriteErrorItemResultToXml
# Description: create error info with xml format
# Input Parameters: $1 $2 $3
# $1: check item sub-id 
# $2: output type, general conditions, the value is 0.
# $3: common info, each line info separated by +
#*********************************************************************************#
function WriteErrorItemResultToXml()
{
    echo "      <item id=\""$1"\">" >>$CHECK_RESULT_FILE
    local msg="$3"
    if [ -z "$msg" ] ; then
        return 1
    fi
    local param_num=`echo $msg | awk -F'+' '{print NF}'`
    for((i=1; i<="$param_num"; i++))
    do
        local param=\$$i
        local param_val=`echo "$msg" | awk -F'+' '{print '$param'}' | sed 's/^[ \t]*//'`
        if [ -n "$param_val" ] ; then
            echo "          <errorvalue type=\"$2\">"$param_val"</errorvalue>" >>$CHECK_RESULT_FILE
        fi
    done
    echo "          <judge>"1"</judge>" >>$CHECK_RESULT_FILE
    echo "      </item>" >>$CHECK_RESULT_FILE
}

#*********************************************************************************#
# Function: WriteExceptItemResultToXml
# Description: output information, whtn the condition is not met.
# Input Parameters: $1 $2 $3
# $1: check item sub-id 
# $2: output type, general conditions, the value is 0.
# $3: common information.
#*********************************************************************************#
WriteExceptItemResultToXml()
{
    echo "		<item id=\""$1"\">" >>$CHECK_RESULT_FILE
    echo "			<commonvalue type=\"$2\">$3</commonvalue>" >>$CHECK_RESULT_FILE
    echo "			<judge>"2"</judge>" >>$CHECK_RESULT_FILE
    echo "		</item>" >>$CHECK_RESULT_FILE
}

#*********************************************************************************#
# Function: WriteResultEndToXml
# Description: create ending of xml format
# Input Parameters:
#*********************************************************************************#
function WriteResultEndToXml()
{   
    echo "  </cmditem>" >>$CHECK_RESULT_FILE
    echo "</device>" >>$CHECK_RESULT_FILE
}

#*********************************************************************************#
# Function: T3000_info_sdr
# Description: get 'ipmitool sdr elist' information when the condition is met.
# Input Parameters: $1
# $1: grep condition.
#*********************************************************************************#
function T3000_info_sdr
{
    IPMI_SDR_FILE="/tmp/sdrinfo.tmp"
    for((i=0; i<5; i++))
    do
        $IPMITOOL sdr elist | grep "$1" > "${IPMI_SDR_FILE}" 2>&1 
        if [ $? -eq 0 ] ; then
            return 0
        fi
    done

    if [ $i -eq 5 ] ; then
        echo "ERROR: Unable to open SDR for reading."
        rm -rf "${IPMI_SDR_FILE}" > /dev/null 2>&1
        return 1
    fi
}

#*********************************************************************************#
# Function: T3000_info_sensor
# Description: get 'ipmitool sensor' information when the condition is met.
# Input Parameters: $1
# $1: grep condition.
#*********************************************************************************#
function T3000_info_sensor
{
    IPMI_SENSOR_FILE="/tmp/sensor.tmp"
    for((i=0; i<5; i++))
    do
        $IPMITOOL sensor | grep "$1" > "${IPMI_SENSOR_FILE}" 2>&1 
        if [ $? -eq 0 ] ; then
            return 0
        fi
    done

    if [ $i -eq 5 ] ; then
        echo "ERROR: Unable to open sensor for reading."
        rm -rf "${IPMI_SENSOR_FILE}" > /dev/null 2>&1
        return 1
    fi
}

#*********************************************************************************#
# Function: Output_CPU_Temp
# Description: output common info of CPU temperature with xml format. 
# Input Parameters:
#*********************************************************************************#
function Output_CPU_Temp()
{
    local msg_a="" no_go=0
    # CPU core temperature offset 
    local temp_offset=85
    # CPU core temperature initial threshold 
    local cpu_thrd_human=80
    
    local cpu_thrd=(`cat "${IPMI_SENSOR_FILE}" | grep CPU._Core_temp | awk -F'|' '{print $8}' | awk -F'.' '{print $1}'| sed 's/^[ \t]*//'`)     
    let cpu_thrd_human=cpu_thrd[0]+temp_offset

    for i in 0 1
    do
        local cpu_temp=`cat "${IPMI_SENSOR_FILE}" | grep CPU"$i"_Core_temp \
            | awk -F'|' '{print $2}'| awk -F'.' '{print $1}' | sed 's/^[ \t]*//'`
        if [ -z "$cpu_temp" ] ; then
            cpu_temp="none"
        else
            let cpu_temp=cpu_temp+temp_offset
        fi

        if [ -z "$cpu_temp" -o "$cpu_temp" = "none" ] ; then
            continue 
        else
            msg_a="$msg_a+CPU ID:CPU$i || Temperature:$cpu_temp || Threshold:${cpu_thrd_human} || Unit:degrees C"
            if [ "$cpu_temp" -ge "${cpu_thrd_human}" ] ; then
                no_go=1
            fi
        fi
    done

    if [ $no_go -eq 1 ] ; then
        WriteErrorItemResultToXml 0 0 "$msg_a"
    else
        WriteNormalItemResultToXml 0 0 "$msg_a"
    fi
    
    return 0
}

#*********************************************************************************#
# Function: Output_CPU_Status
# Description: output common info of CPU status with xml format. 
# Input Parameters:
#*********************************************************************************#
function Output_CPU_Status()
{
    local cpu_num=2
    local cpusts_msg="" no_go=0
    local read_num=`cat $IPMI_SDR_FILE | grep "CPU._Status" | wc -l`

    for i in 0 1
    do
        local cpu_sts=`cat "${IPMI_SDR_FILE}" | grep CPU"$i"_Status | awk -F'|' '{print $5}' 2>/dev/null` 
        [ -z "$cpu_sts" ] && cpu_sts='Failure detected'
        cpusts_msg="$cpusts_msg+CPU ID:CPU$i || Status:$cpu_sts"
    done 

    cat "${IPMI_SDR_FILE}" | grep "CPU._Status" | grep -v "Presence" > /dev/null 2>&1
    if [ "$?" -eq 0 -o "$read_num" -ne "$cpu_num" ] ; then
        WriteErrorItemResultToXml 0 0 "$cpusts_msg"
    else
        WriteNormalItemResultToXml 0 0 "$cpusts_msg"   
    fi
    return 0
}

#*********************************************************************************#
# Function: Output_CPU_Load
# Description: output common info of CPU load with xml format. 
# Input Parameters:
#*********************************************************************************#
function Output_CPU_Load()
{
    local cpu_idle_threshold=25
    local cpu_idle=`sar 3 1 | grep "Average" |awk '{print $8}' |awk -F. '{print $1}'`
    local msg="CPU Used:$((100-cpu_idle))%"

    if [ $cpu_idle -gt $cpu_idle_threshold ] ; then
        WriteNormalItemResultToXml 0 0 "$msg"
    else
        WriteErrorItemResultToXml 0 0 "$msg"
    fi

    return 0
}

#*********************************************************************************#
# Function: VTL_CPUCheck
# Description: output whole info of CPU temperature with xml format. 
# Input Parameters:$1
# $1: ckeck item id 8
#*********************************************************************************#
function VTL_CPUCheckStatus ()
{
    local check_item="$1"

    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item
	
    if [ "x" == "x$IPMITOOL" ]; then
        WriteExceptItemResultToXml 0 0 "ipmitool is not install, please install and try again."
        WriteResultEndToXml 
        return 0
    fi

    T3000_info_sdr "^CPU"
    if [ $? -ne 0 ] ; then
        WriteErrorItemResultToXml 0 0 "Get CPU information failed."
        WriteResultEndToXml
        return 1
    fi
    Output_CPU_Status
    WriteResultEndToXml
    rm -rf "${IPMI_SDR_FILE}" > /dev/null 2>&1
}

#*********************************************************************************#
# Function: VTL_CPUCheck
# Description: output whole info of CPU temperature with xml format. 
# Input Parameters:$1
# $1: ckeck item id 8
#*********************************************************************************#
function VTL_CPUCheckTemp ()
{
    local check_item="$1"

    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item
	
	if [ "x" == "x$IPMITOOL" ]; then
        WriteExceptItemResultToXml 0 0 "ipmitool is not install, please install and try again."
        WriteResultEndToXml 
        return 0
    fi

    T3000_info_sensor "^CPU"
    if [ $? -ne 0 ] ; then
        WriteErrorItemResultToXml 0 0 "Get CPU information failed."
        WriteResultEndToXml
        return 1
    fi
    Output_CPU_Temp
    WriteResultEndToXml
    rm -rf "${IPMI_SENSOR_FILE}" > /dev/null 2>&1
}

#*********************************************************************************#
# Function: VTL_CPULoadCheck
# Description: output whole info of CPU load with xml format. 
# Input Parameters:$1
# $1: ckeck item id 23
#*********************************************************************************#
function VTL_CPULoadCheck ()
{
    local check_item="$1"

    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item
    # output CPU load
    Output_CPU_Load
    WriteResultEndToXml
}

#*********************************************************************************#
# Function: FC_modeTransfer
# Description: get FC mode. 
# Input Parameters:$1
# $1: 0, 1
#*********************************************************************************#
function FC_modeTransfer ()
{
    if [ $1 -eq 1 ] ; then
        echo p2p
    else
        echo loop
    fi
}

#*********************************************************************************#
# Function: VTL_CheckFCStatus
# Description: output whole info of FC status with xml format. 
# Input Parameters:$1
# $1: ckeck item id 3
#*********************************************************************************#
function VTL_CheckFCStatus ()
{
    local fc_device_dir=/proc/scsi/qla2xxx
    local check_item="$1"
    local fc_msg=""

    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"
    
    #Get_VTL_Role
    Discover_Topology '1'
    if [ "$ARCHITECTURE" = "VTL-I" -o "$ARCHITECTURE" = "VTL-IS" ] ; then
        WriteExceptItemResultToXml 0 0 "$ARCHITECTURE does not need to check this item."
        WriteResultEndToXml
        return 0
    fi

    local fc_list=`ls $fc_device_dir` 2>/dev/null
    if [ -z "$fc_list" ] ; then 
        WriteErrorItemResultToXml 0 0 "no device is exist."
        WriteResultEndToXml
        return 1
    fi  

    local no_go=0
    for i in $fc_list
    do
        local fc_rate=`grep "Link_Data_Rate" $fc_device_dir/$i | awk '{print $3}'`
        local fc_topology=`cat $fc_device_dir/$i | grep "^Topology" | awk -F'=' '{print $2}'| grep -v Unknown`
        if [ -z "$fc_topology" -o "$fc_rate" -eq 0 ] ; then
            no_go=1
        fi
        local hbanum=`echo ${i:2:1}`
        local modenum=`grep ^connection $ISHOME/etc/fshba.conf | grep "hba$hbanum=" | cut -d"=" -f2`
        local mode=`FC_modeTransfer $modenum`
                
        fc_msg="$fc_msg+Name:HBA_$i||Rate:${fc_rate}GB||Mode:${mode}"
    done

    if [ "$no_go" -eq 1 ] ; then
        WriteErrorItemResultToXml 0 0 "$fc_msg"
    else
        WriteNormalItemResultToXml 0 0 "$fc_msg"
    fi
    WriteResultEndToXml 
    return 0
}

#*********************************************************************************#
# Function: VTL_CheckMemIdel
# Description: output whole info of swap with xml format. 
# Input Parameters:$1
# $1: ckeck item id 4
#*********************************************************************************#
function VTL_CheckMemIdel ()
{
    local check_item="$1"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item

    local mem_total=`grep MemTotal /proc/meminfo | awk '{print $2}'`
    local mem_free=`grep MemFree /proc/meminfo | awk '{print $2}'`
    local swap_used=`free | grep Swap | awk '{print $3}'`
    if [ $swap_used -gt 0 ]
    then
        WriteErrorItemResultToXml 0 0 "Unit(MB):Memory || Total:$((mem_total/1024)) || Free:$((mem_free/1024)) || Swap:$swap_used"
    else
        WriteNormalItemResultToXml 0 0 "Unit(MB):Memory || Total:$((mem_total/1024)) || Free:$((mem_free/1024)) || Swap:$swap_used"
    fi

    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: VTL_CheckPowerSupply
# Description: output whole info of power status with xml format. 
# Input Parameters:$1
# $1: ckeck item id 5
#*********************************************************************************#
function VTL_CheckPowerSupply ()
{
    local check_item="$1"
    local pow_num=2
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item
	
	if [ "x" == "x$IPMITOOL" ]; then
        WriteExceptItemResultToXml 0 0 "ipmitool is not install, please install and try again."
        WriteResultEndToXml 
        return 0
    fi

    T3000_info_sdr "Power_Supply"
    if [ $? -ne 0 ] ; then
        WriteErrorItemResultToXml 0 0 "Get power information failed."
        WriteResultEndToXml
        return 1
    fi

    local record_num=`cat "$IPMI_SDR_FILE" | wc -l`
    local pwr_msg="" no_go=0
    for i in 1 2
    do
        local pwr_sts=`cat "${IPMI_SDR_FILE}" | grep Power_Supply_"$i" | awk -F'|' '{print $5}' 2>/dev/null` 
        pwr_msg="$pwr_msg+Power ID:Power_Supply_$i || Status:$pwr_sts"
    done
    
    cat "${IPMI_SDR_FILE}" | grep "Failure" > /dev/null 2>&1
    if [ $? -eq 0 -o "$record_num" -ne "$pow_num" ] ; then
        WriteErrorItemResultToXml 0 0 "$pwr_msg"
    else
        WriteNormalItemResultToXml 0 0 "$pwr_msg"
    fi
    rm -rf "${IPMI_SDR_FILE}" > /dev/null 2>&1
    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: VTL_CheckSubrackTemperature
# Description: output whole info of subrack temperature with xml format. 
# Input Parameters:$1
# $1: ckeck item id 13
#*********************************************************************************#
function VTL_CheckSubrackTemperature ()
{
    local check_item="$1"
    local subrack_file="/tmp/subrack.tmp"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item
	
	if [ "x" == "x$IPMITOOL" ]; then
        WriteExceptItemResultToXml 0 0 "ipmitool is not install, please install and try again."
        WriteResultEndToXml 
        return 0
    fi

    T3000_info_sdr "TEMP" && T3000_info_sensor "^TEMP"
    if [ $? -ne 0 ] ; then
        WriteErrorItemResultToXml 0 0 "Get subrack temperature information failed."
        WriteResultEndToXml
        return 1
    fi

    cat $IPMI_SENSOR_FILE | awk -F'|' '{print "Sensor:"$1 "||" "Temperature:"$2 "||" "Threshold:"$10 "||" "Unit:"$3}' | \
        sed 's/ \{1,\}/ /g' > $subrack_file 2>/dev/null
    
    local subrack_val=(`cat $IPMI_SENSOR_FILE | awk -F'|' '{print $2}' | awk -F'.' '{print $1}'`)
    local subrack_low_thrd=(`cat $IPMI_SENSOR_FILE | awk -F'|' '{print $5}' | awk -F'.' '{print $1}'`)
    local subrack_high_thrd=(`cat $IPMI_SENSOR_FILE | awk -F'|' '{print $10}' | awk -F'.' '{print $1}'`)
    local read_num=`cat $IPMI_SENSOR_FILE | wc -l`

    local subrack_msg="" no_go=0
    for((i=1; i<=$read_num; i++))
    do
            subrack_msg="${subrack_msg}+`cat "${subrack_file}" | awk '{if(NR=='$i') {print $0}}'`"
    done    
    
    for ((i=0; i<$read_num; i++))
    do
        # high threshold judgement
        if [ "${subrack_val[i]}" -ge "${subrack_high_thrd[i]}" ] ; then
            no_go=1
            break
        fi
        # low threshold judgement
        if [ "${subrack_low_thrd[i]}" = 'na' ] ; then
            continue
        fi
        if [ "${subrack_val[i]}" -le "${subrack_low_thrd[i]}" ] ; then
            no_go=1
            break
        fi
    done
   
    cat "${IPMI_SDR_FILE}" | awk -F '|' '{print $3}' | grep -E "(nc|cr|nr)" > /dev/null 2>&1
    if [ $? -eq 0 -o "$no_go" -eq 1 ] ; then
        WriteErrorItemResultToXml 0 0 "${subrack_msg}"
    else
        WriteNormalItemResultToXml 0 0 "${subrack_msg}"
    fi
    rm -rf "${IPMI_SDR_FILE}" "${IPMI_SENSOR_FILE}" "${subrack_file}" > /dev/null 2>&1
    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: Output_Fan_Status
# Description: output common info of fan status with xml format. 
# Input Parameters:
#*********************************************************************************#
function Output_Fan_Status()
{
    local faninfo_file="/tmp/fan_state.tmp"
    local fan_num=5
    # get fan status info
    cat "${IPMI_SDR_FILE}" | grep -E "FAN._STATUS" | awk -F'|' '{print "Fan:"$1, "||", "Status:"$5}' \
        | sed 's/ \{1,\}/ /g' > "$faninfo_file" 2>/dev/null
    local record_num=`cat "$faninfo_file" | wc -l`
    local fan_msg="" i=1
    for((i=1; i<=$record_num; i++))
    do
            fan_msg="${fan_msg} + `cat "${faninfo_file}" | awk '{if(NR=='$i') {print $0}}'`"
    done
    
    cat "$faninfo_file" | grep -v "Device Present" > /dev/null 2>&1
    if [ $? -eq 0 -o "$record_num" -ne $fan_num ] ; then
        WriteErrorItemResultToXml 0 0 "$fan_msg"
    else
        WriteNormalItemResultToXml 0 0 "$fan_msg"
    fi

    rm -rf $faninfo_file > /dev/null 2>&1
    return 0
}

#*********************************************************************************#
# Function: Output_Fan_RPM
# Description: output common info of fan rpm with xml format. 
# Input Parameters:
#*********************************************************************************#
function Output_Fan_RPM()
{
    local faninfo_file="/tmp/fan_rpm.tmp"
    
    local fan_msg="" i=1 no_go=0

    # get fan rpm info
    cat "${IPMI_SENSOR_FILE}" | grep "RPM" | awk -F'|' '{print "Fan:"$1, "||", "RPM:"$2, "||", "Threshold:"$5}' \
        | sed 's/ \{1,\}/ /g' > "$faninfo_file" 2>/dev/null

    local record_num=`cat "$faninfo_file" | wc -l` 
    for((i=1; i<=$record_num; i++))
    do
            fan_msg="${fan_msg} + `cat "${faninfo_file}" | awk '{if(NR=='$i') {print $0}}'`"
    done
   
    cat "${IPMI_SDR_FILE}" | grep RPM | awk -F '|' '{print $3}' | grep -E "(nc|cr|nr)" > /dev/null 2>&1
    local rval=$?

    local rpm_val=(`cat $IPMI_SENSOR_FILE | grep -E "FAN._RPM" | awk -F'|' '{print $2}' | awk -F'.' '{print $1}'`)
    local rpm_thrd=(`cat $IPMI_SENSOR_FILE | grep -E "FAN._RPM" | awk -F'|' '{print $5}'| awk -F'.' '{print $1}'`)
    
    for ((i=0; i<$record_num; i++))
    do
        if [ "${rpm_val[i]}" -le "${rpm_thrd[i]}" ] ; then
            no_go=1
        fi

    done

    if [ "$rval" -eq 0 -o "$no_go" -eq 1 ] ; then
        WriteErrorItemResultToXml 0 0 "$fan_msg"
    else
        WriteNormalItemResultToXml 0 0 "$fan_msg"
    fi

    rm -rf $faninfo_file > /dev/null 2>&1
    return 0
}

#*********************************************************************************#
# Function: VTL_CheckFANStatus
# Description: output whole info of fan with xml format. 
# Input Parameters:$1
# $1: ckeck item id 14
#*********************************************************************************#
function VTL_CheckFANStatus ()
{
    local check_item="$1"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item
	
	if [ "x" == "x$IPMITOOL" ]; then
        WriteExceptItemResultToXml 0 0 "ipmitool is not install, please install and try again."
        WriteResultEndToXml 
        return 0
    fi

    T3000_info_sdr "FAN" 
    if [ $? -ne 0 ] ; then
        WriteErrorItemResultToXml 0 0 "Get fan information failed."
        WriteResultEndToXml
        return 1
    fi
    Output_Fan_Status
    rm -rf "${IPMI_SDR_FILE}" > /dev/null 2>&1
    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: VTL_CheckFANRPM
# Description: output whole info of fan with xml format. 
# Input Parameters:$1
# $1: ckeck item id 14
#*********************************************************************************#
function VTL_CheckFANRPM ()
{
    local check_item="$1"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item
	
	if [ "x" == "x$IPMITOOL" ]; then
        WriteExceptItemResultToXml 0 0 "ipmitool is not install, please install and try again."
        WriteResultEndToXml 
        return 0
    fi

    T3000_info_sdr "FAN" && T3000_info_sensor "FAN"
    if [ $? -ne 0 ] ; then
        WriteErrorItemResultToXml 0 0 "Get fan information failed."
        WriteResultEndToXml
        return 1
    fi
    Output_Fan_RPM
    rm -rf "${IPMI_SDR_FILE}" "${IPMI_SENSOR_FILE}" > /dev/null 2>&1

    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: VTL_EthLinkCheck
# Description: output whole info of ethernet with xml format. 
# Input Parameters:$1
# $1: ckeck item id 9
#*********************************************************************************#
function VTL_EthLinkCheck()
{
    local check_item="$1"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item
    
    local net_list=(`ifconfig | grep eth | awk '{print $1}' | grep -v ':'`)
    local net_name=""

    if [ -z "${net_list}" ] ; then
        echo "ERROR: no network is configed."
        return 1    
    fi

    local no_go=0 net_ip="" net_msg="" net_linked="" 
    for net_name in ${net_list[*]}
    do
        ifconfig ${net_name} | grep "RUNNING" > /dev/null 2>&1
        if [ $? -eq 0 ] ; then
            net_linked="yes"
        else
            net_linked="no"
            no_go=1
        fi
    
        net_ip=`ifconfig "$net_name" | grep "inet addr" | awk '{print $2}' | awk -F':' '{print $2}'`
        net_msg="${net_msg} + Interface:${net_name} || IP:${net_ip} || Link detected:${net_linked}"
    done

    if [ ${no_go} -eq 0 ] ; then
        WriteNormalItemResultToXml 0 0 "${net_msg}"
        
    else
        WriteErrorItemResultToXml 0 0 "${net_msg}"
    fi

    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: Output_Integration_Disk
# Description: output common info of disk status with xml format when 
#              the disk is controled by SAS2208
# Input Parameters:
#*********************************************************************************#
function Output_Integration_Disk()
{
    local disk_msg="" i=1
    local disk_file="/tmp/diskinfo.tmp"
    local disk_slot_array=(0 1 2 3 4 5 6 7 8 9 10 11 25 26)
    # get disk info of 12 data disks and 2 system disks if the disk is controled by SAS 2208
    $MEGACLI -PDList -a $MEGA_ADP -nolog | \
        awk -F: '/Slot Number/{gsub(/^[ \t]+/,"",$2);printf "Slot Number:"$2}
        /Device Id/{gsub(/^[ \t]+/,"",$2);printf "||Device Id:"$2}
        /PD Type/{gsub(/^[ \t]+/,"",$2);printf "||PD Type:"$2}
        /Raw Size/{gsub(/^[ \t]+/,"",$2);gsub(/\[.*\]/,"",$2);printf "||Raw Size:"$2}
        /Firmware state/{gsub(/^[ \t]+/,"",$2);printf "||Firmware state:"$2}
        /Inquiry Data/{gsub(/^[ \t]+|[ \t]+$/,"",$2);sub(/[ \t]+/," ",$2);printf "||Inquiry Data:"$2"\n"}' | \
            sed 's/Information//;s/Hotspare Information$//' > $disk_file 2>/dev/null

    local disk_no_info="Device Id:no info||PD Type:no info||Raw Size:no info||Firmware state:no info||Inquiry Data:no info"
    for i in ${disk_slot_array[*]}
    do
        local disk_flag=0
        cat $disk_file | awk -F'|' '{print $1}' | grep -w "$i" > /dev/null 2>&1 && disk_flag=1
        if [ $disk_flag -eq 1 ] ; then
            continue
        else
            if [ "$i" -eq 0 ] ; then
                sed -i '1i Slot Number:1||'"$disk_no_info"'' $disk_file > /dev/null 2>&1
            elif [ "$i" -gt 0 -a "$i" -le 11 ] ; then
                sed -i ''"$i"'a Slot Number:'"$i"'||'"$disk_no_info"'' $disk_file > /dev/null 2>&1
            elif [ "$i" -eq 25 ] ; then
                sed -i '12a Slot Number:25||'"$disk_no_info"'' $disk_file > /dev/null 2>&1
            elif [ "$i" -eq 26 ] ; then
                sed -i '$a Slot Number:26||'"$disk_no_info"'' $disk_file > /dev/null 2>&1
            fi
        fi
    done
    
    for((i=1; i<=${#disk_slot_array[*]}; i++))
    do
            disk_msg="${disk_msg} + `cat "${disk_file}" | awk '{if(NR=='$i') {print $0}}'`"
    done

    local pd_num=`$MEGACLI -pdgetnum -a $MEGA_ADP -nolog | awk -F: '/Number/{gsub(/^[ \t]+/,"",$1);print $2}'`
    cat $disk_file | awk -F'|' '{print $9}' | grep -E -v '(Online|Hotspare|no info)' > /dev/null 2>&1
    if [ $? -ne 1 ] ; then
        WriteErrorItemResultToXml 0 0 "$disk_msg"
    else
        WriteNormalItemResultToXml 0 0 "$disk_msg"
    fi

    rm -rf $disk_file
    return 0
}

#*********************************************************************************#
# Function: Output_root_used
# Description: output common info of root used with xml format.
# Input Parameters:
#*********************************************************************************#
function Output_root_used()
{
    local root_threshold=75
    local root_used=`df -h | grep -w '/' | awk '{print $5}' | sed 's/%//'`
    if [ "$root_used" -gt $root_threshold ] ; then
        WriteErrorItemResultToXml 0 0 "Mounted on:/ || Use%:$root_used%"
    else
        WriteNormalItemResultToXml 0 0 "Mounted on:/ || Use%:$root_used%"
    fi
    return 0
}

#*********************************************************************************#
# Function: Output_SAS_Disk
# Description: output common info of 2 system disks status with xml format when 
#              the disk is controled by SAS2008
# Input Parameters:
#*********************************************************************************#
function Output_SAS_Disk()
{
    local sas_disk_file='/tmp/sas_disk.tmp'
    local disk_slot="25 26"
    if [ ! -x $SAS2IRCU ] ; then
        WriteErrorItemResultToXml 0 0 "The command $SAS2IRCU is not exist."
        return 1
    fi

    for i in $disk_slot
    do    
    $SAS2IRCU 0 display | sed -n '/Slot #[ \t]*:[ \t]*'$i'/,/Drive Type/'p | \
        awk -F: '/Slot #/{gsub(/^[ \t]*/,"",$2);printf "Slot Number:"$2}
        /State/{gsub(/^[ \t]*/,"",$2);printf "||State:"$2}
        /Serial No/{gsub(/^[ \t]*/,"",$2);printf "||Serial No:"$2}
        /Protocol/{gsub(/^[ \t]*/,"",$2);printf "||Type:"$2"\n"}'  >> $sas_disk_file 2>/dev/null
    done
    
    local sas_disk_msg=""
    local disk_no_info="State:no info||Serial No:no info||Type:no info"
    
    for i in $disk_slot
    do
        cat $sas_disk_file | grep "Slot Number:$i" > /dev/null 2>&1
        if [ $? -ne 0 ] ; then
            [ "$i" -eq 25 ] && sed -i '1i Slot Number:25||'"$disk_no_info"'' $sas_disk_file > /dev/null 2>&1
            [ "$i" -eq 26 ] && sed -i '$a Slot Number:26||'"$disk_no_info"'' $sas_disk_file > /dev/null 2>&1
        fi
    done

    for i in 1 2
    do
            sas_disk_msg="${sas_disk_msg} + `cat "${sas_disk_file}" | awk '{if(NR=='$i') {print $0}}'`"
    done

    cat $sas_disk_file | grep -E -v "Ready|Optimal|Rebuilding" > /dev/null 2>&1
    if [ "$?" -eq 0 ] ; then
        WriteErrorItemResultToXml 0 0 "$sas_disk_msg"
    else
        WriteNormalItemResultToXml 0 0 "$sas_disk_msg"
    fi

    rm -rf $sas_disk_file
    return 0
}

#*********************************************************************************#
# Function: Output_SAS_IR_Info
# Description: output common info of system VD status with xml format when 
#              the disk is controled by SAS2008
# Input Parameters:
#*********************************************************************************#
function Output_SAS_IR_Info()
{
    local sas_vd_file='/tmp/sas_disk.tmp'
    if [ ! -x $SAS2IRCU ] ; then
        WriteErrorItemResultToXml 0 0 "The command $SAS2IRCU is not exist."
        return 1
    fi
    $SAS2IRCU 0 display | \
        awk -F: '/Volume ID/{gsub(/^[ \t]*/,"",$2);printf "Volume ID:"$2}
        /Status of volume/{gsub(/^[ \t]*/,"",$2);printf "||Volume state:"$2}
        /RAID level/{gsub(/^[ \t]*/,"",$2);printf "||RAID level:"$2}
        /Size \(in MB\)[ \t]/{gsub(/^[ \t]*/,"",$2);printf "||Size(in MB):"$2"\n"}' > $sas_vd_file 2>/dev/null

    local vd_num=`$SAS2IRCU 0 display | grep "Volume ID" | wc -l`
    local vd_msg="" i=1
    for((i=1; i<=$vd_num; i++))
    do
            vd_msg="${vd_msg} + `cat "${sas_vd_file}" | awk '{if(NR=='$i') {print $0}}'`"
    done

    cat $sas_vd_file | grep -E -v "Okay" > /dev/null 2>&1
    if [ "$?" -eq 0 ] ; then
        WriteErrorItemResultToXml 0 0 "$vd_msg"
    else
        WriteNormalItemResultToXml 0 0 "$vd_msg"
    fi

    rm -rf $sas_vd_file
    return 0

}

#*********************************************************************************#
# Function: Output_VD_Info
# Description: output common info of VD status with xml format when 
#              the disk is controled by SAS2208
# Input Parameters:
#*********************************************************************************#
function Output_VD_Info()
{
    local vd_file='/tmp/vdinfo.tmp'
    local i=1
    $MEGACLI -ldpdinfo -a $MEGA_ADP -nolog | \
        awk -F: '/Virtual Drive/{gsub(/^[ \t]*/,"",$2);gsub(/[ \t]*\(.*/,"",$2);printf "Target Id:"$2}
        /RAID Level/{gsub(/^[ \t]*/,"",$2);gsub(/,.*/,"",$2);printf "|| RAID Level:"$2}
        /^Size/{gsub(/^[ \t]*/,"",$2);printf "|| Size:"$2}
        /^State/{gsub(/^[ \t]*/,"",$2);printf "|| State:"$2}
        /Strip Size/{gsub(/^[ \t]*/,"",$2);printf "|| Strip Size:"$2}
        /Number Of Drives/{gsub(/^[ \t]*/,"",$2);printf "|| Number Of Drives:"$2"\n"}' > $vd_file 2>/dev/null
    local vd_num=`cat $vd_file | grep "Target Id" | wc -l`
    local vd_msg=""
    
    for((i=1; i<=$vd_num; i++))
    do
            vd_msg="${vd_msg} + `cat "${vd_file}" | awk '{if(NR=='$i') {print $0}}'`"
    done

    cat $vd_file | grep -v "Optimal" > /dev/null 2>&1
    if [ "$?" -eq 0 ] ; then
        WriteErrorItemResultToXml 0 0 "$vd_msg"
    else
        WriteNormalItemResultToXml 0 0 "$vd_msg"
    fi

    rm -rf $vd_file
    return 0
}

#*********************************************************************************#
# Function: VTL_HardDiskCheck
# Description: output whole info of disk status with xml format
# Input Parameters: $1
# $1: check item id 10
#*********************************************************************************#
function VTL_HardDiskCheck()
{
    local check_item="$1"
    local i=0
   
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"

    #Get_VTL_Role
    Discover_Topology '1'
    if [ "$ARCHITECTURE" = "VTL-I" -o "$ARCHITECTURE" = "VTL-IS" ] ; then
        Output_Integration_Disk
    else
        Output_SAS_Disk
    fi
    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: VTL_RAID_Check
# Description: output whole info of VD status with xml format
# Input Parameters: $1
# $1: check item id 12
#*********************************************************************************#
function VTL_RAID_Check()
{
    local check_item="$1"
    local i=0
   
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"

    #Get_VTL_Role
    Discover_Topology '1'
    if [ "$ARCHITECTURE" = "VTL-I" -o "$ARCHITECTURE" = "VTL-IS" ] ; then
        Output_VD_Info
    else
        Output_SAS_IR_Info
    fi
    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: VTL_Root_Capacity
# Description: output whole info of root capacity with xml format
# Input Parameters: $1
# $1: check item id 22
#*********************************************************************************#
function VTL_Root_Capacity()
{
    local check_item="$1"
    local i=0
   
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"
    Output_root_used
    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: VTL_CheckHostname
# Description: output whole info of hostname and ip with xml format
# Input Parameters: $1
# $1: check item id 6
#*********************************************************************************#
VTL_CheckHostname ()
{
    local check_item="$1"
    local no_go=0

    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"
    
    
    local name=`hostname`
    local hosts_num=`cat /etc/hosts | sed 's/#.*//' | grep "$name" | wc -l`
    if [ "$hosts_num" -ne 1 ] ; then
        no_go=1
    fi

    
    local hosts_ip=`cat /etc/hosts | sed 's/#.*//' | grep "$name" | sed -e 's/[ \t]*'"$name"'[ \t]*//'`
    ifconfig | grep "$hosts_ip" > /dev/null 2>&1
    if [ $? -ne 0 ] ; then
        no_go=1
    fi

    local name_cfg=`cat $IS_CONF | grep "SANBridge name" | awk -F'"' '{print $2}'`
    echo "$name_cfg" | grep "$name" > /dev/null 2>&1
    if [ $? -ne 0 ] ; then
        no_go=1
    fi

    local name_msg="hostname:$name||IP:$hosts_ip"

    if [ "$no_go" -eq 1 ] ; then
        WriteErrorItemResultToXml 0 0 "$name_msg"
    else
        WriteNormalItemResultToXml 0 0 "$name_msg"
    fi

    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: show_process
# Description: output om process status
# Input Parameters: $1
# $1: process name
#*********************************************************************************#
function show_process()
{
    local serivce_name=$1
    ps -efw | grep -E -v '(grep|sh)' |grep "$serivce_name" > /dev/null 2>&1
    if [ $? -eq 0 ]; then
        echo "Process:"$serivce_name"||State:running"
        return 0
    else
        echo "Process:"$serivce_name"||State:unused"
        return 1
    fi
}

#*********************************************************************************#
# Function: apache_service_status
# Description: output apache process status
# Input Parameters: $1
# $1: process name
#*********************************************************************************#
function apache_service_status ()
{
    local tomcatt_num=`ps ax --width=1000 | grep "[o]rg.apache.catalina.startup.Bootstrap start" | wc -l`
    if [ $tomcatt_num -gt 0 ]; then
            echo "Process:apached||State:running"
            return 0
    else
            echo "Process:apached||State:unused"
            return 1
    fi 
}

#*********************************************************************************#
# Function: VTL_OM_Service_Check
# Description: output whole info of om and apache services status with xml format
# Input Parameters: $1
# $1: check item id 1
#*********************************************************************************#
function VTL_OM_Service_Check ()
{
    local check_item="$1"

    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"

    local om_msg="" line=""
    local om_name=(amf2 om_tlv_server om_client_alarm om_attest_module \
        om_tlv_agent om_mail om_client_hba om_client_bmc om_os om_snmp om_client_vtl)
    
    for line in ${om_name[*]}
    do
        om_msg="${om_msg}+`show_process "$line"`"
    done

    om_msg="${om_msg}+`apache_service_status`"

    echo $om_msg | grep "unused" > /dev/null 2>&1
    if [ $? -eq 0 ]; then
        WriteErrorItemResultToXml 0 0 "$om_msg"
    else
        WriteNormalItemResultToXml 0 0 "$om_msg"
    fi

    WriteResultEndToXml
    return 0 
}

#*********************************************************************************#
# Function: VTL_CheckKdumpStatus
# Description: output whole info of kdump service status with xml format
# Input Parameters: $1
# $1: check item id 2
#*********************************************************************************#
function VTL_CheckKdumpStatus ()
{
    local check_item="$1"
    local kdump_msg=""

    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"

    service kdump status | grep -q "Kdump is operational" > /dev/null 2>&1
    if [ $? -eq 0 ]; then
        kdump_msg="Service:kdump||State:running"
    else
        kdump_msg="Service:kdump||State:unused"
    fi

    echo $kdump_msg | grep -q "unused" > /dev/null 2>&1
    if [ $? -eq 0 ]; then
        WriteErrorItemResultToXml 0 0 "$kdump_msg"
    else
        WriteNormalItemResultToXml 0 0 "$kdump_msg"
    fi

    WriteResultEndToXml
    return 0 
}

#*********************************************************************************#
# Function: VTL_CheckVTLSIRStatus
# Description: output whole info of vtl/sir service status with xml format
# Input Parameters: $1
# $1: check item id 0
#*********************************************************************************#
function VTL_CheckVTLSIRStatus ()
{
    local check_item="$1"
    local kdump_msg=""
    local status_cmd=""

    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"

    if [ "$ROLE" = "vtl" -o "$ROLE" = "vtls" ] ; then
        status_cmd="vtl"
    elif [ "$ROLE" = "sir" ] ; then
        status_cmd="sir"
    fi

    $status_cmd status | grep -E "^Status" | grep -q -v 'RUNNING'
    if [ $? -eq 0 ] ; then
        WriteErrorItemResultToXml 0 0 "Service:"$status_cmd"||State:unused"
    else
        WriteNormalItemResultToXml 0 0 "Service:"$status_cmd"||State:running"
    fi

    WriteResultEndToXml
    return 0 
}

#*********************************************************************************#
# Function: VTL_License_Check
# Description: output whole info of license status with xml format
# Input Parameters: $1
# $1: check item id 11
#*********************************************************************************#
function VTL_License_Check()
{
    local check_item="$1"
    local Keycode=""  no_go=0
    local Keycode="" license_type="" Registration="" license_msg=""
    
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"

    local license_info=`iscon getlicense -s 127.0.0.1 | grep -E "Trial|Standard"`
    if [ -z "$license_info" ] ; then
        WriteErrorItemResultToXml 0 0 "No license is found"
        WriteResultEndToXml
        return 1 
    fi

    Keycode=`echo "$license_info" | awk '{print $1}'`
    echo "$license_info" | grep -q "Trial" && license_type='Trial'
    echo "$license_info" | grep -q "Standard" && license_type='Standard'
    echo "$license_info" | grep -q "No" && Registration='No'
    echo "$license_info" | grep -q "Done" && Registration='Done'

    license_msg="Keycode:$Keycode||Type:$license_type||Registration:$Registration"

    echo "$license_info" | grep -q "Done"
    if [ $? -ne 0 ] ; then
        WriteErrorItemResultToXml 0 0 "$license_msg"
    else
        WriteNormalItemResultToXml 0 0 "$license_msg"
    fi

    WriteResultEndToXml
    return 0 
}

#*********************************************************************************#
# Function: VTL_CheckDatabase
# Description: output whole info of database status with xml format
# Input Parameters: $1
# $1: check item id 15
#*********************************************************************************#
function VTL_CheckDatabase()
{
    local check_item="$1"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"

    if [ "$ROLE" = "sir" ] ; then
        WriteExceptItemResultToXml 0 0 "SIR do not need to check this item."
        WriteResultEndToXml
        return 0
    fi
    local run_status="" online_status=""
    local tle_enabled=`grep TLEOption $IS_CONF | sed 's/.*enabled="//' | awk -F'"' '{print $1}'`
    if [ "$tle_enabled" = "true" ] ; then
        run_status="yes"
    else
        run_status="no"
    fi

    local r_val=""
    local quorumGuid=`cat $IS_CONF | sed -n '/<VirtualDev id=.* type="TLE"/,/<\/VirtualDev>/'p | \
        grep Umap | sed 's/.*guid="//' | awk -F'"' '{print $1}'`
    if [ -z "$quorumGuid" ] ; then
        r_val=1
    else
        cat $IS_CONF | grep "$quorumGuid" | grep -q "PhysicalDev name"
        r_val=$?
    fi
    if [ $r_val -eq 0 ] ; then
        online_status="yes"
    else
        online_status="no"
    fi
    
    if [ "$run_status" = "yes" -a "$online_status" = "yes" ] ; then
        WriteNormalItemResultToXml 0 0 "Option:TLE||Enabled:$run_status||Online:$online_status"
    else
        WriteErrorItemResultToXml 0 0 "Option:TLE||Enabled:$run_status||Online:$online_status"
    fi
    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: VTL_CheckHA
# Description: output whole info of HA vtl status with xml format
# Input Parameters: $1
# $1: check item id 17
#*********************************************************************************#
function VTL_CheckHA ()
{
    local check_item="$1"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"
    
    Discover_Topology '1'

    if [ "$ROLE" != "vtl" -o "$ALL_NODES_NUM" -lt 2 ] ; then
        WriteExceptItemResultToXml 0 0 "The device (role $ROLE, architecture $ARCHITECTURE) do not need to check this item."
        WriteResultEndToXml
        return 0
    fi

    iscon getfailoverstatus -s 127.0.0.1 | grep -q "Failover State=Normal"
    if [ $? -ne 0 ] ; then
        WriteErrorItemResultToXml 0 0 "Option:failover||Status:abnormity"
    else
        WriteNormalItemResultToXml 0 0 "Option:failover||Status:normal"
    fi

    WriteResultEndToXml
    return 0
}

#*********************************************************************************#
# Function: VTL_CheckSIRAssociate
# Description: output whole info of sir-vtl connection status with xml format
# Input Parameters: $1
# $1: check item id 18
#*********************************************************************************#
function VTL_CheckSIRAssociate ()
{
    local check_item="$1"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"

    if [ "$ROLE" != "sir" ] ; then
        WriteExceptItemResultToXml 0 0 "Only the role sir need to check this item."
        WriteResultEndToXml 
        return 0
    fi

    local vtl_ip=(`sed -n '/<VTL>/,/<\/VTL>/'p $ISHOME/etc/$HOSTNAME/sircluster.conf | \
        awk -F'ip="' '{print $2}' | awk -F'"' '{print $1}' | sed '/^$/'d`)
    
    local ip_tmp="" sir_msg="" no_go=0
    for ip_tmp in ${vtl_ip[*]}
    do
        ping -c 2 "$ip_tmp" > /dev/null 2>&1
        if [ $? -eq 0 ] ; then
            sir_msg="$sir_msg+Role:vtl||IP:$ip_tmp||Connectivity:normal"
        else
            sir_msg="$sir_msg+Role:vtl||IP:$ip_tmp||Connectivity:abnormality"
            no_go=1
        fi
    done

    if [ $no_go -eq 0 ] ; then
        WriteNormalItemResultToXml 0 0 "$sir_msg"
    else
        WriteErrorItemResultToXml 0 0 "$sir_msg"
    fi

    WriteResultEndToXml
    return 0        
}

#*********************************************************************************#
# Function: VTL_CheckStorage
# Description: output whole info of LUN owner with xml format
# Input Parameters: $1
# $1: check item id 20
#*********************************************************************************#
function VTL_CheckStorage()
{
    local check_item="$1"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml "$check_item"

    if [ "$ROLE" = "sir" ] ; then
        WriteExceptItemResultToXml 0 0 "The role sir do not need to check this item."
        WriteResultEndToXml 
        return 0
    fi
    
    local ownerlist=`cat $IS_CONF | sed -n '/PhysicalDevs/,/\/PhysicalDevs/'p | grep owner | \
            sed 's/.*owner="//' | awk -F'"' '{print $1}'`
    if [ -z "$ownerlist" ] ; then
        WriteErrorItemResultToXml 0 0 "Get owner information failed"
        WriteResultEndToXml
        return 1
    fi

    local hostlist="" lun_msg=""
    hostlist[0]=`cat $IS_CONF | grep "SANBridge name=" | sed 's/.*name="//' | awk -F'"' '{print $1}'`
    hostlist[1]=`cat $IS_CONF | grep "secondaryServer=" | sed 's/.*secondaryServer="//' | awk -F'"' '{print $1}'`
    
    local result=`echo $ownerlist | sed -n 's/'"${hostlist[0]}"'//g;s/'"${hostlist[1]}"'//g'`
    if [ -z "$result" ] ; then
        lun_msg="Option:LUN owner||State:normal"
        WriteNormalItemResultToXml 0 0 "$lun_msg"
    else
        lun_msg="Option:LUN owner||State:abnormality"
        WriteErrorItemResultToXml 0 0 "$lun_msg"
    fi

    WriteResultEndToXml
    return 0     
}

#*********************************************************************************#
# Function: Normal_Item_Output
# Description: xml output is converted to standard output
# Input Parameters: $1
# $1: xml file 
#*********************************************************************************#
function Normal_Item_Output()
{
    local xml_file=$1
    local item_id=0
    local tmpfile="/tmp/item_$item_id.out"
    cat $xml_file | sed -n '/<item id=\"'$item_id'\">/,/<\/item>/'p | grep -E "commonvalue|errorvalue" \
        | sed 's/.*">//;s/<.*//' | sed 's/||/|/g' > $tmpfile
    local column_num=`cat $tmpfile | awk -F'|' '{if(NR==1) print NF}'`
    local row_num=`cat $tmpfile | wc -l`
    local judge=`cat $xml_file | sed -n '/<item id=\"'$item_id'\">/,/<\/item>/'p | grep judge | sed 's/<\/\{0,\}judge>//g' | awk '{print $1}'`
    
    local title="" title_msg=""
    for((j=1;j<=$column_num;j++))
    do
        cat $tmpfile | grep ':' > /dev/null 2>&1
        if [ $? -eq 0 ] ; then
            title[$j]=`cat $tmpfile | awk -F'|' '{if(NR==1){gsub(/:.*/,"",$'$j'); print $'$j'}}' | sed 's/^[ \t]*//;s/[ \t]*$//'`
            title_msg="$title_msg | ${title[$j]}"
        fi
    done
    title_msg=`echo $title_msg | sed 's/^[ \t|]*//'`
        
    for((j=1;j<=$column_num;j++))
    do
        cat $tmpfile | awk -F'|' 'BEGIN{OFS="|"} {gsub(/.*:/,"",$'$j');print $0}' >> $tmpfile 
        sed -i '1,'$row_num'd' $tmpfile
    done
    if [ -n "$title_msg" ] ; then
        
        sed -i '1i '"$title_msg"'' $tmpfile
         
        cat $tmpfile | sed 's/[ \t]*|[ \t]*/|/g' >> $tmpfile
        sed -i '1,'$((row_num+1))'d' $tmpfile
    fi
    
    sed -i '1i 'status:"$judge"@'' $tmpfile
    
    column -s'|' -t $tmpfile
    rm -rf $tmpfile
}

function Normal_SYS_Output()
{
    local xml_file=$1
    local tmpfile="/tmp/sysctl_$item_id.out"
    cat $xml_file | sed -n '/<sysinfo>/,/<\/sysinfo>/'p | grep -E "field name" \
        | sed 's/.*name="//;s/\">/|/;s/<\/.*//' > $tmpfile
    echo " | " >> $tmpfile
    echo -e "NodeName|ControlIP" >> $tmpfile
    cat $xml_file | sed -n '/<controller>/,/<\/controller>/p' | grep "field name" \
        | sed 's/.*\">//;s/<\/.*//' | sed 'N;s/\n/|/' >> $tmpfile
    
    sed -i '1i status:0@' $tmpfile   
    column -s'|' -t $tmpfile  
    rm -rf $tmpfile
}

function Normal_Output()
{
    local xml_file=$1
    cat $xml_file | grep "<item id=\".\">" > /dev/null 2>&1
    if [ $? -eq 0 ] ; then
        Normal_Item_Output $xml_file
    else
        Normal_SYS_Output $xml_file
    fi
}

function get_nas_status ()
{
    local mod=$*
	vtl status nas |grep "$mod" |grep "RUNNING" >/dev/null 2>/dev/null
    if [ $? -eq 0 ];then
        echo "Process:"$mod"||State:running"
        return 0
    else
        echo "Process:"$mod"||State:stopped"
        return 1
    fi		

}

function VTL_CheckNASStatus ()
{
    local check_item="$1"
    InitFile $CHECK_RESULT_FILE
    WriteResultHeaderToXml $check_item

    if [ -f /etc/.is.sh ]
    then
        . /etc/.is.sh
    else
	    WriteErrorItemResultToXml 0 0 "Installation is not complete. Environment profile is missing in /etc"
        WriteResultEndToXml
        return 1
    fi
	
	local nas_msg=""
	local status_01="FUSE"
	local status_02="ROFS Rescan"
	local status_03="NAS MGTD"
	local status_04="NAS NMBD"
	local status_05="NAS SMBD"
	local status_06="NAS UNFSD"
	local status_07="Outgoing Replication Throttling"
	
    local FQDN=`hostname`
    local HOSTNAME=${FQDN%%.*}
	local XMLCONF="$ISHOME/etc/$HOSTNAME/ipstor.conf"
    
	
    if [ "`xmlstat $XMLCONF isnasenabled`" = "1" ]; then
        	
        nas_msg="${nas_msg}+`get_nas_status "$status_01"`"
        nas_msg="${nas_msg}+`get_nas_status "$status_02"`"
        nas_msg="${nas_msg}+`get_nas_status "$status_03"`"
        nas_msg="${nas_msg}+`get_nas_status "$status_04"`"
        nas_msg="${nas_msg}+`get_nas_status "$status_05"`"
        nas_msg="${nas_msg}+`get_nas_status "$status_06"`"
        nas_msg="${nas_msg}+`get_nas_status "$status_07"`"
		
	    echo $nas_msg |grep "stopped" >/dev/null 2>/dev/null	
        if [ $? -eq 0 ]; then
	        WriteErrorItemResultToXml 0 0 "$nas_msg"
        else
            WriteNormalItemResultToXml 0 0 "$nas_msg"
		fi
	else
	    WriteExceptItemResultToXml 0 0 "NAS module is no enable, not need to check this item."
        WriteResultEndToXml 
        return 0
	fi
	
	WriteResultEndToXml
	return 0

}

#*********************************************************************************#
# Function: VTL_CheckAndExecCommand
# Description: Branch selection
# Input Parameters: $1
# $1: check item id, contain one or more parameters
#*********************************************************************************#
function VTL_CheckAndExecCommand ()
{
    while [ -n "$1" ]
    do
        local check_item="$1"
        case $check_item in
            baseinfo)
                VTL_CheckBaseInfo 
                ;;
            0)
                VTL_CheckVTLSIRStatus $check_item
                ;;
            1)
                VTL_OM_Service_Check $check_item
                ;;
            2)
                VTL_CheckKdumpStatus $check_item
                ;;                
            3)
                VTL_CheckFCStatus $check_item
                ;;
            4)
                VTL_CheckMemIdel $check_item
                ;;
            5)
                VTL_CheckPowerSupply $check_item
                ;;
            6)
                VTL_CheckHostname $check_item
                ;;
            7)
                VTL_CPUCheckStatus $check_item
                ;;
            8)
                VTL_CPUCheckTemp $check_item
                ;;
            9)
                VTL_EthLinkCheck $check_item
                ;;
            10)
                VTL_HardDiskCheck $check_item
                ;;
            11)
                VTL_License_Check $check_item
                ;;
            12)
                VTL_RAID_Check $check_item
                ;;
            13)
                VTL_CheckSubrackTemperature $check_item
                ;;
            14)
                VTL_CheckFANStatus $check_item
                ;;
            15)
                VTL_CheckFANRPM $check_item
                ;;
            16)
                VTL_CheckDatabase $check_item
                ;;
            17)
                VTL_CheckHA $check_item
                ;;                
            18)
                VTL_CheckSIRAssociate $check_item
                ;;     
            20)
                VTL_CheckStorage $check_item
                ;;    
            22)
                VTL_Root_Capacity $check_item
                ;;
            23)
                VTL_CPULoadCheck $check_item
                ;;
			24)
			    VTL_CheckNASStatus $check_item
                ;;				
            *)
                echo "ERROR: check item is wrong."
                return 1
                ;;

        esac
        shift 1
        if [ "$NORMAL" -eq 1 ] ; then
            Normal_Output $CHECK_RESULT_FILE
        else
            cat $CHECK_RESULT_FILE
        fi
    done

    rm -rf $CHECK_RESULT_FILE
}

#*********************************************************************************#
# Function: MAIN
# Description: main function
# Input Parameters: $*
# $*: all parameters
#*********************************************************************************#
function MAIN()
{
    local all_item="0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 20 22 23"
    NORMAL=0
    case $1 in
        -c)
            shift 1
            VTL_CheckAndExecCommand $*
            ;;
        -o)
            shift 1
            NORMAL=1
            VTL_CheckAndExecCommand $*
            ;;
        -all)
            if [ -n "$2" ] ; then
                echo "ERROR: Parameter error"
                return 1
            fi
            NORMAL=1
            VTL_CheckAndExecCommand $all_item
            ;;
        *)
            echo "ERROR: Parameter error"
            return 1
            ;;
    esac
}

################ MAIN ################
MAIN $*



