# -*- coding: UTF-8 -*-

import re
import json
import traceback

from sh import common as pkg_common
from cli import cliUtil
from frameone.rest import restData
from frameone.rest import restUtil
from frameone.util import contextUtil

from cbb.common.conf.productConfig import NEW_ARM_DEVS_DIFF_PRODUCT_STRING
from cbb.common.conf.productConfig import VersionManager
from cbb.frame.base import config
from cbb.frame.rest import restData as cbbRestData
from cbb.frame.rest import restUtil as cbbRestUtil
from cbb.frame.rest import commonRestUtil
from cbb.frame.cli import cliUtil as cbbCliUtil
from cbb.frame.util import common as cbbCommon
from cbb.common.query.hardware import disk
from cbb.business.product.product_selector import get_product_adapter
from cbb.business.product.expansion_spec import trans_expansion_dict_for_eval
from utils import Products
from java.lang import Exception as JException

# 区分逻辑引擎的型号
LOGIC_ENG_PRODUCTS = ["Dorado3000 V3",
                      "Dorado5000 V3",
                      "Dorado6000 V3",
                      "Dorado18000 V3",
                      "OceanStor Dorado 3000 V6",
                      "OceanStor Dorado 2000",
                      "OceanStor Dorado 2020",
                      "OceanStor Dorado 2100",
                      "OceanStor Dorado 5000 V6",
                      "OceanStor Dorado 5300 V6",
                      "OceanStor Dorado 5500 V6",
                      "OceanStor Dorado 5600 V6",
                      "OceanStor Dorado 5800 V6",
                      "OceanStor Dorado 6000 V6",
                      "OceanStor Dorado 6800 V6",
                      "OceanStor Dorado 8000 V6",
                      "OceanStor Dorado 18000 V6",
                      "OceanStor Dorado 18500 V6",
                      "OceanStor Dorado 18800 V6",
                      "OceanStor Dorado 18800K V6",
                      "OceanStor Dorado 5600K V6",
                      "OceanStor Dorado 3000",
                      "OceanStor Dorado 5000",
                      "OceanStor Dorado 5300",
                      "OceanStor Dorado 5500",
                      "OceanStor Dorado 5600",
                      "OceanStor Dorado 6000",
                      "OceanStor Dorado 6800",
                      "OceanStor Dorado 8000",
                      "OceanStor Dorado 18000",
                      "OceanStor Dorado 18500",
                      "OceanStor Dorado 18800",
                      "OceanStor Dorado 18800K",
                      "OceanStor Dorado 5600K",
                      "OceanStor 2200",
                      "OceanStor 2220",
                      "OceanStor 5120",
                      "OceanStor 2600",
                      "OceanStor 2620",
                      "OceanStor 5210",
                      "OceanStor 5220",
                      "OceanStor 5310",
                      "OceanStor 5310 Capacity Flash",
                      "OceanStor 5510 Capacity Flash",
                      "OceanStor A300",
                      "OceanStor 5320",
                      "OceanStor 5510",
                      "OceanStor 5510S",
                      "OceanStor 5300K",
                      "OceanStor 5500K",
                      "OceanStor 5610",
                      "OceanStor 5810-HS",
                      "OceanStor 6810",
                      "OceanStor 18510",
                      "OceanStor 18500K",
                      "OceanStor 18810",
                      "OceanProtect X3000",
                      "OceanProtect X8000",
                      "OceanProtect E8000",
                      "OceanProtect X9000",
                      "OceanProtect X6000",
                      "OceanProtect X8000K",
                      "OceanProtect X9000K",
                      "OceanStor Micro 1300",
                      "OceanStor Micro 1500",
                      "OceanDisk 1300",
                      "OceanDisk 1500",
                      "OceanDisk 1600",
                      "OceanDisk 1610",
                      "OceanDisk 1500T",
                      "OceanDisk 1600T",
                      "OceanDisk 1610T",
                      "OceanStor 2910",
                      "6900 V3",
                      "6800 V3",
                      "6800F V3"]

# Dorado 单硬盘域版本
SINGLE_DD_VERSION = ["V300R001C00", "V300R001C01", "V300R001C21", "V300R001C30"]

EXPANSION_SPEC = {
    "S5500T":
        {
            "2": "4"
        },
    "S5600T":
        {
            "2": "4"
        },
    "S5800T":
        {
            "2": "4"
        },
    "S6800T":
        {
            "2": "4"
        },
    "2200 V3":
        {
            "1": "2"
        },
    "2600 V3":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "2600 V3 ENHANCED":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "2600F V3":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "2600F V3 ENHANCED":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "2800 V3":
        {
            "1": "2"
        },
    "2600 V3 FOR VIDEO":
        {
            "1": "2"
        },
    "2810 V5":
        {
            "2": "4",
        },
    "5110 V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5110F V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5210 V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5100K V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5200K V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5210F V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5300 V3":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5300 V5 ENHANCED":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5500 V3":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5500F V3":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5300 V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5310 V5":
        {
            "2": "4",
        },
    "5300F V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5310F V5":
        {
            "2": "4",
        },
    "5500 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "5510 V5":
        {
            "2": "4",
        },
    "5500F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "5510F V5":
        {
            "2": "4",
        },
    "DORADO NAS":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5500 V5 ELITE":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5600 V3":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5600 V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5610 V5":
        {
            "2": "4",
        },
    "5600F V3":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5600F V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5610F V5":
        {
            "2": "4",
        },
    "5800 V3":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5800 V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5810 V5":
        {
            "2": "4",
        },
    "5800F V3":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5800F V5":
        {
            "2": "4,8",
            "4": "8",
            "6": "8",
        },
    "5810F V5":
        {
            "2": "4",
        },
    "6800 V3":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "6800 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "6810 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8"
        },
    "6800F V3":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "6800F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "6810F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8"
        },
    "6900 V3":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "18500 V3":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18500 V5":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18510 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8"
        },
    "18500F V3":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18500F V5":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18510F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8"
        },
    "18800 V3":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18800 V5":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18810 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8"
        },
    "18800F V3":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18800F V5":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18810F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8"
        },
    "DORADO3000 V3":
        {
            "2": "4",
        },
    "DORADO5000 V3":
        {
            "2": "4",
        },
    "DORADO5000_SUP6":
        {
            "2": "4,6",
            "4": "6",
        },
    "DORADO5000_SUP8":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "DORADO6000 V3":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "DORADO18000 V3":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "OCEANSTOR DORADO 3000 V6":
        {
            "2": "4",
        },
    "OCEANSTOR DORADO 5000 V6":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "OCEANSTOR DORADO 5300 V6":
        {
            "2": "4",
        },
    "OCEANSTOR DORADO 5500 V6":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "OCEANSTOR DORADO 5600 V6":
        {
            "2": "4",
        },
    "OCEANSTOR DORADO 5800 V6":
        {
            "2": "4",
        },
    "OCEANSTOR DORADO 6000 V6":
        {
            "2": "4",
        },
    "OCEANSTOR DORADO 6800 V6":
        {
            "2": "4,8",
            "4": "8",
        },
    "OCEANSTOR DORADO 8000 V6":
        {
            "2": "4,8",
            "4": "8",
        },
    "OCEANSTOR DORADO 18000 V6":
        {
            "2": "4,8",
            "4": "8",
        },
    "OCEANSTOR DORADO 18500 V6":
        {
            "2": "4,8",
            "4": "8",
        },
    "OCEANSTOR DORADO 18800 V6":
        {
            "2": "4,8",
            "4": "8",
        },
    "OCEANSTOR DORADO 18800K V6":
        {
            "2": "4,8",
            "4": "8",
        },
    "OCEANDISK 1500":
        {
            "2": "4"
        },
    "OCEANDISK 1600":
        {
            "2": "4"
        },
    "OCEANDISK 1610":
        {
           "2": "4"
         },
    "OCEANDISK 1610T":
        {
           "2": "4"
         },
    "OCEANDISK 1500T":
        {
            "2": "4"
        },
    "OCEANDISK 1600T":
        {
            "2": "4"
        }
}

EXPANSION_SPEC_HIGH_C70SPC200 = {
    "6810 V5":
        {
            "2": "4,6,8,10,12",
            "4": "6,8,10,12",
            "6": "8,10,12",
            "8": "10,12",
            "10": "12",
        },
    "6810F V5":
        {
            "2": "4,6,8,10,12",
            "4": "6,8,10,12",
            "6": "8,10,12",
            "8": "10,12",
            "10": "12",
        },
    "18510 V5":
        {
            "2": "4,6,8,10,12",
            "4": "6,8,10,12",
            "6": "8,10,12",
            "8": "10,12",
            "10": "12",
        },
    "18510F V5":
        {
            "2": "4,6,8,10,12",
            "4": "6,8,10,12",
            "6": "8,10,12",
            "8": "10,12",
            "10": "12",
        },
    "18810 V5":
        {
            "2": "4,6,8,10,12",
            "4": "6,8,10,12",
            "6": "8,10,12",
            "8": "10,12",
            "10": "12",
        },
    "18810F V5":
        {
            "2": "4,6,8,10,12",
            "4": "6,8,10,12",
            "6": "8,10,12",
            "8": "10,12",
            "10": "12",
        },
}

# 注意：NEW_ARM_DEVS_DIFF_PRODUCT_STRING中对型号进行了等价映射
# 下表中配置的型号都是映射后的型号
EXPANSION_SPEC_C71 = {
    "2600 V5":
        {
            "2": "4"
        },
    "5110 V5":
        {
            "2": "4"
        },
    "5120 V5":
        {
            "2": "4"
        },
    "5110F V5":
        {
            "2": "4"
        },
    "5210 V5":
        {
            "2": "4"
        },
    "5220 V5":
        {
            "2": "4"
        },
    "5210F V5":
        {
            "2": "4"
        },
    # "2810 V5" 不支持扩控
    "5310 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "5310F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "5510 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "5510F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "5610 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "5610F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "5810 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "5810F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8",
        },
    "6810 V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8"
        },
    "6810F V5":
        {
            "2": "4,6,8",
            "4": "6,8",
            "6": "8"
        },
    "18510 V5":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18510F V5":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18810 V5":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
    "18810F V5":
        {
            "2": "4,6,8,10,12,14,16",
            "4": "6,8,10,12,14,16",
            "6": "8,10,12,14,16",
            "8": "10,12,14,16",
            "10": "12,14,16",
            "12": "14,16",
            "14": "16",
        },
}

EXP_INTERNAL_PRODUCT_MODEL_VERSION = "V300R001C21"

# 硬盘框列表
FRAME_ID_LIST = "frameIdList"
# 硬盘数量列表
DISK_NUM_LIST = "diskNumList"
# 硬盘容量列表
DISK_CAP_LIST = "diskCapacityList"


def hasInnerMetroLicense(devObj):
    """是否存在内双活license

    :param devobj:
    :return:
    """
    rest = contextUtil.getRest(devObj)
    lang = str(devObj.get("lang"))
    errMsg = cliUtil.getMsg(lang, "cannot.get.license.service.auth.info")
    try:
        hasInnerMetroLicense, _ = cbbRestUtil.CommonRest.hasInnerLicense(rest)
        return True, hasInnerMetroLicense, None
    except Exception as exception:
        return False, False, errMsg


def checkAaWorkMode(devObj):
    """检查A/A工作模式是否开启(V500R007C50开始支持)

    :param devObj: 设备对象
    :return: param1：True-命令执行成功（命令不支持视为成功且未开启），False-命令执行失败
             param2: True-开启， False-未开启
             param3: cli回显
             param4: 错误消息
    """
    lang = str(devObj.get("lang"))
    cli = devObj.get("ssh")
    cmd = "show tgt_switch storage_work_mode"
    # 该命令需在developer下执行
    flag, cliRet, errMsg = cliUtil.excuteCmdInDeveloperMode(cli, cmd, True, lang)
    if flag is not True:
        # 命令不支持视为成功且未开启
        if cliUtil.isNotSupport(cliRet):
            return True, False, cliRet, ""
        return False, False, cliRet, errMsg
    else:
        cliRetLines = cliUtil.getVerticalCliRetFilterElabel(cliRet)
        for line in cliRetLines:
            if str(line.get("Switch Status", "")) == "On":
                return True, True, cliRet, ""
        return True, False, cliRet, ""


def getInternalProductModel(devObj):
    """查询内部产品型号

    :param devObj:
    :return:
    """
    lang = str(devObj.get("lang"))
    rest = contextUtil.getRest(devObj)
    errMsg = cliUtil.getMsg(lang, "failed.to.get.internal.product.model")

    params = {}
    uriParamDict = restUtil.CommonRest.getUriParamDict(restData.RestCfg.SpecialUri.INTERNAL_DEVICE_INFO)
    record = restUtil.CommonRest.execCmd(rest, uriParamDict, params, restData.RestCfg.RestMethod.GET)
    if not record:
        # 如果取得为空的话，直接返回None
        return False, None, errMsg
    else:
        internalProductModel = restUtil.CommonRest.getRecordValue(restUtil.CommonRest.getData(record),
                                                                  restData.InternalDeviceInfo.INTERNAL_PRODUCT_MODEL)
        return True, internalProductModel, None


def hasSASbigCard(devObj):
    """是否存在SAS大卡

    :param devObj:
    :return:param1:命令是否执行成功：True-成功，False-失败
            param2:是否存在SAS大卡：True-存在，False-不存在
            param3:命令执行失败的错误信息
    """
    lang = str(devObj.get("lang"))
    cli = devObj.get("ssh")
    cmd = "show interface_module"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        errMsg = cliUtil.getMsg(lang, "failed.to.get.sas.intf")
        return False, False, errMsg

    if "4*12Gb SAS Back-End Interconnec".lower() in cliRet.lower():
        return True, True, ""
    return True, False, ""


def getEngs(devObj):
    """获取引擎号

    :param devObj: 设备对象
    :return:param1:命令是否执行成功：True-成功，False-失败
            param2:引擎ID
            param3:命令执行失败的错误信息
    """
    lang = str(devObj.get("lang"))
    cli = devObj.get("ssh")
    # TV2V2R2C00SPC200产品cli查询筛选命令BUG，增加重试命令
    cmd1 = "show controller general |filterColumn include columnList=Location"
    cmd2 = "show controller general |filterColumn include colunmList=Location"
    flag, cliRet, errMsg = excuteCmdInCliMode4Privilege(cli, [cmd1, cmd2], True, lang)
    if flag is not True:
        errMsg = cliUtil.getMsg(lang, "failed.to.get.eng")
        return False, "", errMsg

    engineList = []
    for line in cliRet.splitlines():
        if ":" in line and "location" in line.lower() and "." in line:
            eng = line.split(":")[-1].strip().split(".")[0]
            if eng not in engineList:
                engineList.append(eng)

    return True, ",".join(engineList), ""


def getDomainIds(devObj):
    """获取硬盘域ID

    :param devObj: 设备对象
    :return: param1:命令是否执行成功：True-成功，False-失败
            param2:硬盘域ID列表
            param3:命令执行失败的错误信息
    """
    lang = str(devObj.get("lang"))
    cli = devObj.get("ssh")

    ddIdList = []
    # TV2V2R2C00SPC200产品cli查询筛选命令BUG，增加重试命令
    cmd1 = "show disk_domain general |filterColumn include columnList=ID"
    cmd2 = "show disk_domain general |filterColumn include colunmList=ID"
    flag, cliRet, errMsg = excuteCmdInCliMode4Privilege(cli, [cmd1, cmd2], True, lang)

    if flag is not True:
        errMsg = cliUtil.getMsg(lang, "failed.to.get.diskdomain")
        return False, ddIdList, errMsg

    if "Command executed successfully".lower() in cliRet:
        return True, ddIdList, ""

    for line in cliRet.splitlines():
        line = line.strip()
        if line.isdigit():
            ddIdList.append(line)

    return True, ddIdList, ""


def is_unique_performance_layer(performance_layer_id, cli, lang):
    """
    是否是独占的性能层
    :param performance_layer_id: 性能层ID
    :param cli: ssh连接
    :param lang: 语言
    :return: True 独占， False 共享
    """
    cmd = "show performance_layer general performance_layer_id={}".format(performance_layer_id)
    flag, cli_ret, err_msg = excuteCmdInCliMode4Privilege(cli, [cmd], True, lang)
    records = cliUtil.getVerticalCliRet(cli_ret)
    for record in records:
        return record.get("Share Strategy") == "Unique"


def get_pools_and_mapping(devObj):
    """获取存储池ID列表和存储池ID与硬盘域ID的对应关系

    :param devObj: 上下文
    :return: param1:命令是否执行成功：True-成功，False-失败
             param2:存储池ID列表
             param3:存储池ID与硬盘域ID的对应关系
             param4:命令执行失败的错误信息
             param5:性能层与是否独享关系
    """
    lang = str(devObj.get("lang"))
    cli = devObj.get("ssh")
    logger = devObj.get("logger")
    is_new_oceanstor = bool(devObj.get("is_new_oceanstor"))
    logger.info("when getting pools, the device is new OceanStor: {}".format(str(is_new_oceanstor)))

    pool_id_list = []
    pool_and_disk_domain_mapping = {}
    smart_pool_is_unique_performance = {}
    cmd = "show storage_pool general"
    flag, cli_ret, err_msg = excuteCmdInCliMode4Privilege(cli, [cmd], True,
                                                          lang)
    if flag is not True:
        err_msg = cliUtil.getMsg(lang, "failed.to.get.storagepool")
        return False, pool_id_list, pool_and_disk_domain_mapping, err_msg, smart_pool_is_unique_performance

    records = cliUtil.getHorizontalCliRet(cli_ret)
    for record in records:
        pool_id = record.get("ID")
        disk_domain_id = record.get("Disk Domain ID")
        performance_layer_id = record.get("Performance Layer ID")
        smart_pool_is_unique_performance[performance_layer_id] = "False"
        # 非共享性能层关联的存储池不展示
        # 如果不是新融合，则跳过
        if not is_new_oceanstor and is_unique_performance_layer(performance_layer_id, cli, lang):
            continue
        if is_unique_performance_layer(performance_layer_id, cli, lang):
            smart_pool_is_unique_performance[performance_layer_id] = "True"
        pool_id_list.append(pool_id)
        pool_and_disk_domain_mapping[pool_id] = disk_domain_id
    logger.info("poolIdList is: {}, poolAndDiskDomainMapping:{}".format(
        pool_id_list, pool_and_disk_domain_mapping))

    return True, pool_id_list, pool_and_disk_domain_mapping, err_msg, smart_pool_is_unique_performance


def getCtrlIds(devObj):
    """获取控制器ID，如0A,0B...

    :param devObj: 设备对象
    :return: param1:命令是否执行成功：True-成功，False-失败
             param2:控制器ID列表
             param3:命令执行失败的错误信息
    """
    lang = str(devObj.get("lang"))
    cli = devObj.get("ssh")
    ctrlList = []
    # TV2V2R2C00SPC200产品cli查询筛选命令BUG，增加重试命令
    cmd1 = "show controller general |filterColumn include columnList=Controller"
    cmd2 = "show controller general |filterColumn include colunmList=Controller"
    flag, cliRet, errMsg = excuteCmdInCliMode4Privilege(cli, [cmd1, cmd2], True, lang)
    if flag is not True:
        errMsg = cliUtil.getMsg(lang, "failed.to.get.ctrl")
        return False, ctrlList, errMsg

    for line in cliRet.splitlines():
        if ":" in line and "controller" in line.lower():
            ctrl = line.split(":")[-1].strip()
            ctrlList.append(ctrl)

    return True, ctrlList, errMsg


def getDevCtrlsNum(devObj):
    from cbb.business.product.product_selector import init_product_adapter
    from cbb.frame.checkitem.context_adapter import InspectContext
    init_product_adapter(InspectContext(devObj).get_context())
    flag, ctrlList, errMsg = getCtrlIds(devObj)
    return flag, str(len(ctrlList)), errMsg


def getDevDomains(devObj):
    flag, ddIdList, errMsg = getDomainIds(devObj)
    return flag, ",".join(ddIdList), errMsg


def is_micro_pass_though_mode(dev_obj):
    cli = dev_obj.get("ssh")
    lang = str(dev_obj.get("lang"))
    return cbbCommon.is_micro_pass_though_mode(cli, lang)


def checkNodeCfg(devObj, devCtrlsNum):
    """检查节点配置是否和在位节点一致

    :param devObj: 设备对象
    :param devCtrlsNum: 集群控制器数量
    :return:
    """
    cli = devObj.get("ssh")
    lang = str(devObj.get("lang"))
    logger = devObj.get("logger")

    cmd = "show system config_model"
    __, cliRet, __ = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if "Single-Controller" in cliRet:
        return True, ""

    flag, nodecfg, errMsg = cliUtil.getNodeCfg(devObj)
    if not flag:
        return False, errMsg

    if str(nodecfg) != str(devCtrlsNum):
        logger.info("[getExpInfo]node cfg is inconsistent:(current num:%s, node cfg:%s)" % (devCtrlsNum, nodecfg))
        errMsg = cliUtil.getMsg(lang, "controllers.number.inconsistent")
        return False, errMsg

    return True, ""


def getExpSpec(devObj):
    """获取扩容配置

    :param devObj: 设备对象
    :return: param1:命令是否执行成功：True-成功，False-失败
             param2:控制器ID列表
             param3:命令执行失败的错误信息
    """
    devType = devObj.get("devType")
    lang = str(devObj.get("lang"))
    logger = devObj.get("logger")
    productVersion = devObj.get("productVersion")

    flag, devCtrlsNum, errMsg = getDevCtrlsNum(devObj)
    if not flag:
        return False, "", errMsg

    flag, errMsg = checkNodeCfg(devObj, devCtrlsNum)
    if not flag:
        return False, "", errMsg

    devUpper = devType.upper()

    product_adapter = get_product_adapter()
    if product_adapter:
        expansion_spec = product_adapter.get_expansion_spec(devCtrlsNum)
        flag, exp_spec, _ = trans_expansion_dict_for_eval(expansion_spec)
        if not exp_spec:
            err_msg = cliUtil.getMsg(lang, "donot.support.expansion")
            return False, "", err_msg
        return flag, exp_spec, ""

    if 'Dorado5000' in devType and getVRCVersion(productVersion) >= EXP_INTERNAL_PRODUCT_MODEL_VERSION:
        __, isNVMe, __ = isNVMeEnclosure(devObj)
        qrySuccess, internalProductModel, errMsg = getInternalProductModel(devObj)
        if qrySuccess:
            logger.info('query internal product model:%s' % internalProductModel)
        else:
            logger.info('query internal product info failed.')
            return False, '', errMsg
        # Dorado5000 V300R002C20 NVMe/purley 支持到8控
        if productVersion >= 'V300R002C20':
            if isNVMe or internalProductModel == 'Dorado5000E_V3':
                devUpper = 'DORADO5000_SUP8'

    # V5R7C60/SPC100 Kunpeng不支持扩控
    if productVersion in VersionManager.V5_NOT_SUP_CTRL_EXP:
        return True, "", cliUtil.getMsg(lang, "version.not.support.expansion")

    # V5 Kunpeng海外版、国内版、安可版本枚举值一致，显示型号不同，区分海外版与老硬件
    if "Kunpeng" in productVersion:
        for common_pdt_model, pdt_model_list in \
                NEW_ARM_DEVS_DIFF_PRODUCT_STRING.items():
            if devType in pdt_model_list:
                devUpper = common_pdt_model.upper()
                break

    logger.info('devUpper is: %s.' % devUpper)
    expCfg = get_expansion_cfg(devUpper, productVersion)

    if str(devCtrlsNum) not in expCfg:
        errMsg = cliUtil.getMsg(lang, "donot.support.expansion")
        return True, "", errMsg
    return True, expCfg.get(devCtrlsNum, ""), ""


def get_expansion_cfg(devUpper, productVersion):
    exp_cfg = {}
    if "Kunpeng" in productVersion \
            and Products.compareVersion(
        productVersion, 'V500R007C71') >= 0:
        exp_cfg = EXPANSION_SPEC_C71.get(devUpper, {})
    if exp_cfg:
        return exp_cfg
    if "Kunpeng" in productVersion \
            and Products.compareVersion(
        productVersion, 'V500R007C70SPC200') >= 0:
        exp_cfg = EXPANSION_SPEC_HIGH_C70SPC200.get(devUpper, {})
    if exp_cfg:
        return exp_cfg
    exp_cfg = EXPANSION_SPEC.get(devUpper, {})
    return exp_cfg


def getEngHeight(devObj):
    """获取控制框高度

    :param devObj: 设备对象
    :return: param1:命令是否执行成功：True-成功，False-失败
             param2:高度
             param3:命令执行失败的错误信息
    """
    lang = str(devObj.get("lang"))
    cli = devObj.get("ssh")
    height = ""
    cmd = "show enclosure enclosure_id=CTE0"
    __, cliRet, __ = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    for line in cliRet.splitlines():
        line = line.strip()
        if "Height(U)" in line:
            height = line.split(":")[-1].strip()
            return True, height, ""

    cmd = "show enclosure enclosure_id=ENG0"
    __, cliRet, __ = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    for line in cliRet.splitlines():
        line = line.strip()
        if "Height(U)" in line:
            height = line.split(":")[-1].strip()
            return True, height, ""

    errMsg = cliUtil.getMsg(lang, "failed.to.get.eng")
    return False, height, errMsg


def isAllCtrlIdsHasDomain(devObj, domainList, allCtrls):
    logger = devObj.get("logger")
    cli = devObj.get("ssh")
    lang = str(devObj.get("lang"))

    ctrlsSet = set()
    allCtrlsSet = set(allCtrls)

    for domainId in domainList:
        cmd = "show disk_domain general disk_domain_id=%s" % domainId
        __, cliRet, __ = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        cliRet = cli.execCmd(cmd)
        lines = cliRet.splitlines()
        for line in lines[::-1]:
            if "Controller    " not in line:
                continue

            ctrlInfo = line.split(":")[-1].strip()
            ctrls = ctrlInfo.split(",")
            ctrlsSet = ctrlsSet | set(ctrls)
            logger.info("get disk domain info ctrlsSet=%s, allCtrlsSet=%s" % (ctrlsSet, allCtrlsSet))
            if ctrlsSet == allCtrlsSet:
                return False

    return True


def isDomainFullForDorado(devObj, height, ctrlList):
    devType = devObj.get("devType").upper()
    productVersion = devObj.get("productVersion")
    # 获取VRC版本
    vrcVersion = getVRCVersion(productVersion)

    flag, domainInfo, __ = ExpMode(devObj, height, ctrlList).getDiskDomainInfo()
    if not flag:
        return False

    flag, logicEngs, __ = ExpMode(devObj, height, ctrlList).getLogicEngs()
    if not flag:
        return False
    # 单个逻辑引擎最大硬盘域数量
    if vrcVersion in SINGLE_DD_VERSION:
        maxDDNumSingleEng = 1
    elif devType == "DORADO3000 V3":
        maxDDNumSingleEng = 2
    else:
        maxDDNumSingleEng = 4

    for eng in logicEngs:
        ddCount = 0
        for dd_eng in domainInfo.keys():
            if eng in dd_eng:
                ddCount += 1
        if ddCount < maxDDNumSingleEng:
            return True

    return False


def isExistedCreateDomain(devObj):
    """扩容控制器都可以新建硬盘域，扩容硬盘/框/柜是一个逻辑引擎只能一个硬盘域

    :param devObj: 设备对象
    :return: True-可新建硬盘域；False-不可新建硬盘域
    """
    # 非Dorado设备可新建硬盘域
    devType = devObj.get("devType")
    if "DORADO" not in devType.upper():
        return True,

    # 获取硬盘域ID
    __, ddIdList, __ = getDomainIds(devObj)
    if not ddIdList:
        return True,

    flag, height, __ = getEngHeight(devObj)
    if not flag:
        return False,

    # 获取全部控制器ID列表
    __, ctrlList, __ = getCtrlIds(devObj)

    return isDomainFullForDorado(devObj, height, ctrlList),


def getDDInfo(devObj):
    """获取硬盘域信息

    :param devObj: 设备对象
    :return:
    """
    devType = devObj.get("devType")
    if devType not in LOGIC_ENG_PRODUCTS:
        return True, "", ""

    flag, height, errMsg = getEngHeight(devObj)
    if not flag:
        return False, "", errMsg

    flag, ctrlList, errMsg = getCtrlIds(devObj)
    if not flag:
        return False, "", errMsg

    flag, domainInfo, errMsg = ExpMode(devObj, height, ctrlList).getDomainCtrInfo()
    if not flag:
        return False, "", errMsg

    return True, domainInfo, errMsg


def getExpModeInfo(devObj):
    """获取扩容模式

    :param devObj:
    :return: param1:命令是否执行成功：True-成功，False-失败
             param2:用于选择配置显示的模式信息，如：逻辑引擎0:硬盘域0,逻辑引擎1:硬盘域1
             param3:用于镜像验证的模式信息，如：逻辑引擎0:硬盘域0,逻辑引擎1:硬盘域1
             param4:命令执行失败的错误信息
    """
    devType = devObj.get("devType")
    if devType not in LOGIC_ENG_PRODUCTS:
        return True, "", "", ""

    flag, height, errMsg = getEngHeight(devObj)
    if not flag:
        return False, "", "", errMsg

    flag, ctrlList, errMsg = getCtrlIds(devObj)
    if not flag:
        return False, "", "", errMsg

    flag, expMode, domainInfo, errMsg = ExpMode(devObj, height, ctrlList).getExpMode()
    if not flag:
        return False, "", "", errMsg

    expModeInfo = []
    for exp in expMode:
        expModeInfo.append("%s:%s" % (exp[0], exp[1]))

    ddInfo = []
    for e in domainInfo:
        ddInfo.append("%s:%s" % (e[0][1], e[1]))

    return True, ",".join(expModeInfo), ",".join(ddInfo), ""


def isNVMeEnclosure(devObj):
    """查询是否为NVME控制框 针对内部型号Dorado5000_V3

    :param devObj:
    :return:
    """
    cli = devObj.get("ssh")
    lang = str(devObj.get("lang"))
    cmd = "show enclosure enclosure_id=CTE0"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if not flag:
        return False, "", errMsg
    if "NVMe" in cliRet:
        return True, True, ""
    return True, False, ""


def getDisksOfNVMe(devObj):
    """获取NVMe引擎框上的硬盘数

    :param devObj: 设备对象
    :return:
    """
    cli = devObj.get("ssh")
    lang = str(devObj.get("lang"))
    cmd = "show enclosure enclosure_id=CTE0"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if not flag:
        return False, "", errMsg

    if "NVMe" not in cliRet:
        return True, "", ""
    # TV2V2R2C00SPC200产品cli查询筛选命令BUG，增加重试命令
    cmd1 = "show disk general |filterColumn include columnList=ID"
    cmd2 = "show disk general |filterColumn include colunmList=ID"
    flag, cliRet, errMsg = excuteCmdInCliMode4Privilege(cli, [cmd1, cmd2], True, lang)
    if not flag:
        return False, "", errMsg
    info = {}
    lines = cliUtil.getHorizontalCliRet(cliRet)
    for line in lines:
        dId = line.get("ID", "")
        if len(dId) < 4:
            continue

        engId = dId[3]
        info[engId] = info.get(engId, 0) + 1

    result = []
    for k in info:
        result.append("%s:%s" % (k, info.get(k)))

    return True, ",".join(result), ""


class ExpMode:
    NODE_MAP = {
        "A": 0,
        "B": 0,
        "C": 1,
        "D": 1,
    }

    def __init__(self, devObj, height, ctrlList):
        self.devObj = devObj
        self.logger = devObj.get("logger")
        self.cli = devObj.get("ssh")
        self.lang = str(devObj.get("lang"))
        self.devType = devObj.get("devType")
        self.height = height
        self.ctrlList = ctrlList

    def getLogicEngId(self, ctrlId):
        engId = int(ctrlId[0])
        node = ctrlId[-1]
        if str(self.height) != "6":
            return str(engId)
        else:
            return str(engId * 2 + self.NODE_MAP.get(node, 0))

    def getLogicEngs(self):

        logicEngs = set()
        flag, ctrlList, errMsg = getCtrlIds(self.devObj)
        if not flag:
            return False, logicEngs, errMsg

        for ctrl in ctrlList:
            logicEngs.add(self.getLogicEngId(ctrl))
        return True, logicEngs, ""

    def getDomainCtrInfo(self):

        domainInfo = {}
        # TV2V2R2C00SPC200产品cli查询筛选命令BUG，增加重试命令
        cmd1 = "show disk_domain general|filterColumn include columnList=ID,Controller"
        cmd2 = "show disk_domain general|filterColumn include colunmList=ID,Controller"
        flag, cliRet, errMsg = excuteCmdInCliMode4Privilege(self.cli, [cmd1, cmd2], True, self.lang)
        if flag == cliUtil.RESULT_NOCHECK:
            cmd1 = "show disk_domain general|filterColumn include columnList=ID"
            cmd2 = "show disk_domain general|filterColumn include colunmList=ID"
            flag, cliRet, errMsg = excuteCmdInCliMode4Privilege(self.cli, [cmd1, cmd2], True, self.lang)

        if not flag:
            return False, domainInfo, errMsg

        cliList = cliUtil.getHorizontalCliRet(cliRet)
        for line in cliList:
            domainId = line.get("ID")
            ctrls = self.ctrlList
            if "Controller" in line.keys():
                ctrls = line.get("Controller", ).split(",")
            domainInfo[domainId] = ctrls

        return True, domainInfo, ""

    def getDiskDomainInfo(self):

        domainInfo = {}
        # TV2V2R2C00SPC200产品cli查询筛选命令BUG，增加重试命令
        cmd1 = "show disk_domain general|filterColumn include columnList=ID,Controller"
        cmd2 = "show disk_domain general|filterColumn include colunmList=ID,Controller"
        flag, cliRet, errMsg = excuteCmdInCliMode4Privilege(self.cli, [cmd1, cmd2], True, self.lang)
        if flag == cliUtil.RESULT_NOCHECK:
            cmd1 = "show disk_domain general|filterColumn include columnList=ID"
            cmd2 = "show disk_domain general|filterColumn include colunmList=ID"
            flag, cliRet, errMsg = excuteCmdInCliMode4Privilege(self.cli, [cmd1, cmd2], True, self.lang)

        if not flag:
            return False, domainInfo, errMsg

        cliList = cliUtil.getHorizontalCliRet(cliRet)
        for line in cliList:
            domainId = line.get("ID")
            ctrls = self.ctrlList
            if "Controller" in line.keys():
                ctrls = line.get("Controller", ).split(",")

            for ctrl in ctrls:
                engId = self.getLogicEngId(ctrl)
                domainInfo[(domainId, str(engId))] = domainId

        return True, domainInfo, ""

    def getExpMode(self):
        """获取硬盘域扩容配置信息

        :return: result1:是否获取成功
                 result2:用于选择配置显示的配置信息，结构：[(逻辑引擎ID,硬盘域ID)]
                 result3:用于镜像验证的配置信息，当前DoradoV3才有，结构：[(逻辑引擎ID,硬盘域ID)]
        """
        expModes = []
        flag, domainInfo, errMsg = self.getDiskDomainInfo()
        if not flag:
            return False, [], [], errMsg

        domainInfos = sorted(domainInfo.items(), key=lambda x: x[0], reverse=False)
        # dorado一个引擎一个硬盘域，选择配置和镜像判断的硬盘域信息一样
        if "dorado" in self.devType.lower():
            for domIf in domainInfos:
                expModes.append((domIf[0][1], domIf[1]))
            return True, expModes, domainInfos, ""

        flag, logicEngs, errMsg = self.getLogicEngs()
        if not flag:
            return False, [], [], errMsg

        domains = domainInfo.values()
        logicEngList = list(logicEngs)
        logicEngList.sort()
        domains.sort()
        for eng in logicEngList:
            for domain in domains:
                expModes.append((eng, domain))

        self.logger.info("[getExpInfo] expMode:%s" % expModes)
        return True, expModes, domainInfos, ""


def getVRCVersion(version):
    """获取产品的VRC版本信息

    :param version: 产品版本
    :return: 产品VRC版本
    """
    # 数字版本直接返回
    if "." in version:
        return version
    if len(version) <= 11:
        return version
    else:
        return version[0:11]


def excuteCmdInCliMode4Privilege(cli, cmdList, isHasLog, lang, endWithSign=None):
    """TV2 V2R2C00SPC200产品cli查询筛选命令有bug，通过此方法兼容

    :param cli: cli对象
    :param cmdList: 命令列表
    :param isHasLog: 是否需要日志
    :param lang: 语言
    :param endWithSign: 指定结束符
    :return:
    """
    flag = False
    cliRet = ""
    errMsg = ""
    for cmd in cmdList:
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, isHasLog, lang, endWithSign=endWithSign)
        if not cliUtil.hasCliExecPrivilege(cliRet):
            continue
        return flag, cliRet, errMsg
    return flag, cliRet, errMsg


def get_all_engine_disk_info(devObj):
    """
    获取硬盘域成员盘和空闲盘信息
    :param devObj: 环境变量
    :return flag: True(获取成功)/False(获取失败)
    :return free_disk_info: 空闲盘信息
    :return member_disk_info: 成员盘信息
    """
    lang = str(devObj.get("lang"))
    logger = devObj.get("logger")
    rest = devObj.get("SCRIPT_DEF_CONTEXT").get("REST_CONNECTION")
    pool_id = str(devObj.get("poolId", None))
    disk_domain_id = str(devObj.get("diskDomainId", None))
    cli = devObj.get("ssh")

    flag, free_disk_list, member_disk_list = \
        commonRestUtil.get_formatted_engine_disk_info(rest, pool_id,
                                                      disk_domain_id, logger,
                                                      lang)
    if not flag:
        logger.error("cannot get free disk and member disk.")
        return flag, free_disk_list, member_disk_list

    flag, elabel_capacity_map, err_msg = \
        cbbCliUtil.get_elabel_capacity(cli, lang, logger)
    if not flag:
        logger.error("cannot get e-label capacity.")
        return flag, free_disk_list, member_disk_list

    free_disk_list = \
        match_elabel_capacity(elabel_capacity_map, free_disk_list, logger)
    member_disk_list = \
        match_elabel_capacity(elabel_capacity_map, member_disk_list, logger)

    return True, json.dumps(free_disk_list), json.dumps(member_disk_list)


def check_exp_performance_layer_engine_disk_num(dev_obj, exp_disk_list):
    logger = dev_obj.get("logger")
    err_msg_list = []
    exp_engines = dev_obj.get("expEngines")
    logger.info("exp_engines:" + str(exp_engines))
    flag, member_disk_dict, smart_pool_strategy_dict = get_smart_pool_member_disk_info(dev_obj)
    logger.info("member_disk_dict:" + str(member_disk_dict))
    exp_disk_info_dict = {}
    for exp_disk in exp_disk_list:
        # 不是性能层，则继续循环
        if str(exp_disk.get("performanceLayerModel")) != "1":
            continue
        engine_id = str(exp_disk.get("eng"))
        disk_num = int(exp_disk.get("diskNum")) if str(exp_disk.get("diskNum")) != "0" else 0
        layer_id = exp_disk.get("diskDomain")
        temp_exp_dict = exp_disk_info_dict.get(layer_id, {})
        temp_exp_dict[engine_id] = temp_exp_dict.get(engine_id, 0) + disk_num
        exp_disk_info_dict[layer_id] = temp_exp_dict
    logger.info("exp_disk_info_dict:" + str(exp_disk_info_dict))
    for layer_id, exp_disk_info in exp_disk_info_dict.items():
        # 共享性能层
        # 共享性能层需要判断是否是多引擎场景，如果是则需要判断成员盘数量 + exp数量是否相同。
        exp_eng = dev_obj.get("expEngines")
        if not smart_pool_strategy_dict.get(layer_id):
            # 独享性能层原始成员盘是否是多引擎, 如果是多引擎则需要判断成员盘数量 + exp数量是否相同。
            exp_eng = [eng for eng, member_info in member_disk_dict.get(layer_id, {}).items()]
            exp_eng.extend([eng for eng, member_info in exp_disk_info.items() if eng not in exp_eng])
        if not check_all_engine_disk_num_consistence(exp_disk_info, member_disk_dict.get(layer_id, {}), exp_eng):
            err_msg_list.append(
                cliUtil.getMsg(str(dev_obj.get("lang")), "performance.layer.exp.engine.disk.num.diff", layer_id))
    return err_msg_list


def check_all_engine_disk_num_consistence(exp_disk_info, member_disk_info, exp_engines):
    all_eng_disk_list = []
    for eng in exp_engines:
        all_eng_disk_list.append(exp_disk_info.get(eng, 0) + member_disk_info.get(eng, 0))
    return len(list(set(all_eng_disk_list))) == 1


def get_exp_performance_layer_evaluate_result(dev_obj):
    """
    获取性能层评估结果
    @param dev_obj: 环境变量
    @return:
    """
    flag = True
    err_msg_list = []
    logger = dev_obj.get("logger")
    exp_disk_list = dev_obj.get("expDiskList")
    exp_disk_list = get_real_capacity(exp_disk_list)
    if not exp_disk_list:
        return True, ""
    logger.info("exp_disk_list:" + str(exp_disk_list))
    try:
        err_msg_list = check_exp_performance_layer_engine_disk_num(dev_obj, exp_disk_list)
        if err_msg_list:
            return False, "\n".join(err_msg_list)
    except (Exception, JException) as ex:
        logger.error("check_exp_performance_layer_engine_disk_num Exception:{}".format(traceback.format_exc()))
        return False, "\n".join(err_msg_list)

    exp_performance_layer_evaluate_param = get_performance_layer_eval_param(exp_disk_list, dev_obj)
    if not exp_performance_layer_evaluate_param:
        return True, ""

    for performance_layer_id in exp_performance_layer_evaluate_param:
        evaluate_param = exp_performance_layer_evaluate_param.get(performance_layer_id)
        rest_flag, rest_msg = exp_performance_layer_evaluate_by_rest(dev_obj, evaluate_param, logger)
        if not rest_flag:
            flag = False
            err_msg_list += [cliUtil.getMsg(str(dev_obj.get("lang")), "performance.layer.id", performance_layer_id)]
            err_msg_list += rest_msg
    if flag:
        return True, ""
    return False, "\n".join(err_msg_list)


def exp_performance_layer_evaluate_by_rest(dev_obj, evaluate_param, logger):
    """
    性能层评估
    :param dev_obj: 变量上下文
    :param evaluate_param: 扩容评估入参
    :param logger: 日志句柄
    :return:
    """
    try:
        flag = True
        err_msg = []
        lang = str(dev_obj.get("lang"))
        logger.info("lang:" + lang)
        uri_param_dict = cbbRestUtil.CommonRest.getUriParamDict(
            cbbRestData.RestCfg.OBJ.PERFORMANCE_LAYER_EXPANSION_EVAL)
        rest = contextUtil.getRest(dev_obj)
        logger.info("params:" + str(evaluate_param))
        # 接口超时时间
        time_out = 120
        ret_obj = cbbRestUtil.CommonRest.execCmd(
            rest, uri_param_dict, evaluate_param,
            restData.RestCfg.RestMethod.POST, time_out, True)
        ret_data = ret_obj.get("data")
        if ret_data:
            ret_code_str = ret_data[0].get("expandEvaluationRet")
            ret_data_str = ret_data[0].get("expandExpansionDataList")
            if ret_code_str != "0":
                flag = False
                err_msg = pkg_common.get_error_info(
                    lang, ret_code_str, ret_data_str, evaluate_param.get("id"))

        return flag, err_msg
    except Exception as ex:
        logger.error("exp_evaluate_by_rest Exception:{}".format(traceback.format_exc()))
        if ex.args[0] == "1077949002":
            # 命令不支持 兼容TR5版本（6.0.RC1） 返回成功
            return True, []
        else:
            raise ex


def get_performance_layer_eval_param(exp_disk_list, dev_obj):
    logger = dev_obj.get("logger")
    exp_performance_layer_evaluate_param = {}
    capacity_number_dict = {}
    is_exp_disk_domain_enclosure_redundant = False
    for line in exp_disk_list:
        # 扩容框级冗余硬盘域
        if str(line.get("performanceLayerModel")) != "1" and str(line.get("needCheckEnclosureRedundant")) == "1":
            is_exp_disk_domain_enclosure_redundant = True
        # 不是性能层直接继续循环
        if str(line.get("performanceLayerModel")) != "1":
            continue
        engine_id = str(line.get("eng"))
        disk_num = int(line.get("diskNum")) if str(line.get("diskNum")) != "0" else -2
        pool_id = line.get("diskDomain")
        disk_model = str(line.get("diskModel"))
        capacity_unit = str(line.get("unit"))
        disk_capacity = str(line.get("diskCapacity"))
        disk_bom = str(line.get("diskBom"))
        disk_enclosure_id = str(line.get("diskEnclosureId"))
        is_enc_redundant = str(line.get("needCheckEnclosureRedundant")) == "1"
        disk_capacity_2_gb = pkg_common.changUnit2GBLabelCap(disk_capacity + capacity_unit)[1]
        disk_capacity_2_gb = str(int(disk_capacity_2_gb))

        tmp_performance_param = dict()
        tmp_performance_param["disk_capacity_2_gb"] = disk_capacity_2_gb
        tmp_performance_param["disk_num"] = disk_num
        tmp_performance_param["engine_id"] = engine_id
        tmp_performance_param["pool_id"] = pool_id
        tmp_performance_param["disk_model"] = disk_model
        tmp_performance_param["disk_bom"] = disk_bom
        tmp_performance_param["diskEnclosureId"] = disk_enclosure_id
        a_performance_evaluate_param = exp_performance_layer_evaluate_param.get(pool_id, {})
        update_a_performance_layer_capacity_param(
            is_enc_redundant, capacity_number_dict, a_performance_evaluate_param, tmp_performance_param, dev_obj
        )
        if is_enc_redundant:
            a_performance_evaluate_param["is_enc_redundant"] = is_enc_redundant
        logger.info("a_performance_evaluate_param: {}".format(a_performance_evaluate_param))
        exp_performance_layer_evaluate_param[pool_id] = a_performance_evaluate_param

    logger.info("exp_performance_layer_evaluate_param:{},"
                "capacity_number_dict:{}".format(exp_performance_layer_evaluate_param, capacity_number_dict))
    set_exp_performance_layer_eval_param(
        capacity_number_dict, exp_performance_layer_evaluate_param, is_exp_disk_domain_enclosure_redundant)
    logger.info("exp_performance_layer_evaluate_param:{}".format(exp_performance_layer_evaluate_param))
    return exp_performance_layer_evaluate_param


def set_exp_performance_layer_eval_param(disk_cap_num_dict, eval_param, is_exp_disk_domain_enclosure_redundant):
    """
    设置性能层容量点和数量
    :param disk_cap_num_dict: 硬盘容量点字典
    :param eval_param: 扩容参数
    :param is_exp_disk_domain_enclosure_redundant: 是否为框级冗余硬盘域
    :return:
    """
    for domain_id, tmp_dict in disk_cap_num_dict.items():
        eval_param.get(domain_id)["id"] = domain_id
        # 框级冗余
        if eval_param.get(domain_id).get("is_enc_redundant"):
            for end_id_cap, cap_num in tmp_dict.items():
                frame_ids = eval_param.get(domain_id).get(FRAME_ID_LIST, [])
                frame_ids.append(end_id_cap.split("_")[0])
                eval_param.get(domain_id)[FRAME_ID_LIST] = frame_ids
                disk_num = eval_param.get(domain_id).get(DISK_NUM_LIST, [])
                disk_num.append(str(cap_num[1]))
                eval_param.get(domain_id)[DISK_NUM_LIST] = disk_num
                disk_cap = eval_param.get(domain_id).get(DISK_CAP_LIST, [])
                disk_cap.append(cap_num[0])
                eval_param.get(domain_id)[DISK_CAP_LIST] = disk_cap
            continue
        for cap, disk_num in tmp_dict.items():
            eval_param.get(domain_id)[DISK_NUM_LIST] = \
                eval_param.get(domain_id).get(DISK_NUM_LIST, []) + [str(disk_num)]
            eval_param.get(domain_id)[DISK_CAP_LIST] = \
                eval_param.get(domain_id).get(DISK_CAP_LIST, []) + [str(cap)]
    for performance_layer_id in eval_param:
        param = eval_param.get(performance_layer_id)
        # 不是框级冗余继续
        if not param.get(FRAME_ID_LIST):
            continue
        # 框级冗余，同时扩性能层和存储池，检查规则参数为1
        if is_exp_disk_domain_enclosure_redundant:
            eval_param.get(performance_layer_id)["checkFrameLevelRule"] = "1"
            continue
        # 单独扩容框级冗余性能层，检查规则参数为0
        eval_param.get(performance_layer_id)["checkFrameLevelRule"] = "0"


def update_a_performance_layer_capacity_param(
        is_enc_redundant, capacity_number_dict, a_performance_evaluate_param, tmp_performance_param, dev_obj
):
    disk_capacity_2_gb = tmp_performance_param.get("disk_capacity_2_gb")
    disk_num = tmp_performance_param.get("disk_num")
    engine_id = tmp_performance_param.get("engine_id")
    pool_id = tmp_performance_param.get("pool_id")
    disk_model = tmp_performance_param.get("disk_model")
    disk_bom = tmp_performance_param.get("disk_bom")
    disk_enclosure_id = tmp_performance_param.get("diskEnclosureId")
    tmp_dict = capacity_number_dict.get(pool_id, {})
    if is_enc_redundant:
        if not tmp_dict.get(disk_enclosure_id + disk_capacity_2_gb):
            tmp_dict[disk_enclosure_id + "_" + disk_capacity_2_gb] = [disk_capacity_2_gb, disk_num]
        else:
            tmp_dict.get(disk_enclosure_id + "_" + disk_capacity_2_gb)[1] += int(disk_num)
    else:
        tmp_dict[disk_capacity_2_gb] = tmp_dict.get(disk_capacity_2_gb, 0) + int(disk_num)
    capacity_number_dict[pool_id] = tmp_dict
    a_performance_evaluate_param["engineIdList"] = list(set(
        a_performance_evaluate_param.get("engineIdList", []) + [
            engine_id]
    ))
    # 615 及之后的需要传入disk type
    if Products.compareVersion(str(dev_obj.get("productVersion")), "6.1.5RC1") >= 0:
        a_performance_evaluate_param["TIER0DISKTYPE"] = commonRestUtil.get_disk_type_enum(disk_model, disk_bom)


def get_real_capacity(cfg_data):
    """将扩容配置中的电子标签容量折算为实际容量

    :param cfg_data: 扩容配置数据
    :return cfg_data: 折算后的配置数据
    """
    # 1GB电子标签容量折算为实际容量的系数, 算法为:
    # 1GB电子标签容量 = 1000*1000*1000B
    # 实际容量 = 1000*1000*1000/1024/1024/1024 = 0.9313225746154785 GB
    to_real_capacity = 0.9313225746154785
    for a_data in cfg_data:
        a_data["diskCapacity"] = float(a_data.get("diskCapacity", "0")) * to_real_capacity
    return cfg_data


def match_elabel_capacity(elabel_capacity_map, engine_disk_info, logger):
    """为实际容量匹配电子标签容量

    :param elabel_capacity_map: 实际容量与对应的电子标签容量映射
    :param engine_disk_info: 硬盘信息列表
    :param logger: 日志句柄
    :return:
    """

    # 参数校验
    if not engine_disk_info or not elabel_capacity_map:
        return engine_disk_info

    # 按硬盘域/存储池循环
    for disk_domain_info in engine_disk_info:
        disks = disk_domain_info.get("disks")
        # 按硬盘类型+硬盘容量+硬盘数量循环
        for disk in disks:
            capacity = disk.get("capacity")
            elabel_capacity, elabel_unit = \
                get_elabel_capacity(capacity, elabel_capacity_map, logger)
            disk["inputCapacity"] = elabel_capacity
            disk["inputUnit"] = elabel_unit

    logger.info("after match e-label capacity, engine_disk_info is: %s" %
                str(engine_disk_info))
    return engine_disk_info


def get_elabel_capacity(capacity, elabel_capacity_map, logger):
    """查询实际容量的电子标签容量

    :param capacity: 实际容量
    :param elabel_capacity_map: 设备中的实际容量与电子标签容量列表
    :param logger: 日志句柄
    :return:
    """
    reg_pattern = re.compile(r"(\d+\.?\d*\s*)([G|T]B?)", re.I)

    for disk_capacity_str in elabel_capacity_map:
        match_object = reg_pattern.match(disk_capacity_str)
        if not match_object:
            continue
        disk_capacity = float(match_object.group(1).strip())
        disk_capacity_unit = match_object.group(2).strip().upper()
        if "TB" in disk_capacity_unit:
            disk_capacity *= 1024
        if abs(float(capacity) - disk_capacity) <= 10.0:
            logger.info("disk_capacity is: %s" % str(disk_capacity))
            logger.info("capacity is: %s" % str(capacity))
            elabel_capacity_str = elabel_capacity_map.get(disk_capacity_str)
            match_object = reg_pattern.match(elabel_capacity_str)
            if not match_object:
                continue
            elabel_capacity = match_object.group(1).strip()
            elabel_capacity_unit = match_object.group(2).strip().upper()
            return elabel_capacity, elabel_capacity_unit

    return "", ""


def get_disk_bom_list(context):
    """获取设备硬盘的BOM信息列表

    :param context: 上下文
    :return:
    """
    cli = context.get("ssh")
    lang = str(context.get("lang"))
    logger = context.get("logger")
    reg_pattern = re.compile(r"(\d+\.?\d*\s*)([G|T]B?)", re.I)
    disk_bom_list = []

    flag, disk_general_list, cli_ret, err_msg = \
        disk.get_disk_general(cli, lang, logger)
    if not flag:
        return flag, json.dumps(disk_general_list)

    flag, elabel_capacity_map, err_msg = \
        cbbCliUtil.get_elabel_capacity(cli, lang, logger)
    if not flag:
        logger.error("cannot get e-label capacity.")
        return False, json.dumps(disk_bom_list)

    for disk_general in disk_general_list:
        disk_bom = {}
        disk_bom["bom"] = disk_general.get("Item")
        disk_bom["type"] = disk_general.get("Type")
        # 根据实际容量取出电子标签容量
        capacity = disk_general.get("Capacity")
        elabel_capacity = elabel_capacity_map.get(capacity, "")
        # 电子标签容量拆分出数字和单位 例如 "960.000GB" 拆出"960.000"和"GB"
        match_object = reg_pattern.match(elabel_capacity)
        if not match_object:
            continue
        disk_bom["inputCapacity"] = match_object.group(1).strip()
        disk_bom["inputUnit"] = match_object.group(2).strip().upper()
        if disk_bom in disk_bom_list:
            continue
        else:
            disk_bom_list.append(disk_bom)

    logger.info("disk_bom_list is: %s" % str(disk_bom_list))
    return True, json.dumps(disk_bom_list)


def has_encryption_license(dev_obj):
    rest = contextUtil.getRest(dev_obj)
    has_enc_license_flag, encryption_license_info, _ = cbbRestUtil.CommonRest.has_encryption_license(rest)
    return has_enc_license_flag


def getRedundancyStrategy(dev_obj):
    """获取硬盘域的冗余策略

    :param dev_obj: 上下文
    :return:
    """
    domain_id_list = dev_obj.get("devDomains")
    cli = dev_obj.get("ssh")
    lang = dev_obj.get("lang")
    logger = dev_obj.get("logger")
    domain_strategy_dict = {}
    encryption_domain_list = []
    has_enc_license = has_encryption_license(dev_obj)
    for domain_id in domain_id_list:
        logger.info("domain_id:" + str(domain_id))
        cmd = ("show disk_domain general disk_domain_id=%s" % str(domain_id),)
        flag, cli_ret, err_msg = excuteCmdInCliMode4Privilege(cli, cmd, True,
                                                              lang)
        if not flag:
            return "", err_msg, ",".join(encryption_domain_list)

        dict_list = cliUtil.getVerticalCliRet(cli_ret)
        for record in dict_list:
            domain_strategy_dict[domain_id] = record.get("Redundancy Strategy", "--")
            if record.get("Disk Encryption Switch") == "On" and has_enc_license:
                encryption_domain_list.append(domain_id)

    return json.dumps(domain_strategy_dict), "", ",".join(encryption_domain_list)


def getFreeDiskEnclosureID(dev_obj):
    """获取所有空闲硬盘框信息

    :param dev_obj:
    :return:
    """
    lang = str(dev_obj.get("lang"))
    logger = dev_obj.get("logger")
    rest = contextUtil.getRest(dev_obj)
    pool_id = None
    disk_domain_id = None
    cli = dev_obj.get("ssh")

    flag, free_disk_list, member_disk_list = \
        commonRestUtil.get_formatted_engine_disk_info(rest, pool_id,
                                                      disk_domain_id, logger,
                                                      lang)
    # 性能层空闲盘需要单独处理
    if dev_obj.get("devType") in config.NEW_DORADO + config.OCEAN_STOR_MICRO_DEVS:
        performance_layer_free_disk_info = commonRestUtil. \
            get_performance_layer_free_disk_info(rest, logger)
        free_disk_list.extend(performance_layer_free_disk_info)
    logger.info("free disk list:" + str(free_disk_list))
    if not flag:
        logger.error("cannot get free disk and member disk.")
        return flag, "", cliUtil.getMsg(lang,
                                        "cannot.get.free.disk.and.member.disk")

    flag, elabel_capacity_map, err_msg = \
        cbbCliUtil.get_elabel_capacity(cli, lang, logger)
    if not flag:
        logger.error("cannot get e-label capacity.")
        return flag, "", err_msg

    free_disk_list = \
        match_elabel_capacity(elabel_capacity_map, free_disk_list, logger)
    logger.info("free disk list after elabel capacity:" + str(free_disk_list))

    return True, json.dumps(free_disk_list), ""


def has_smart_cache_pool(dev_obj):
    """查询设备上是否有Smart Cache Pool

    """
    lang = str(dev_obj.get("lang"))
    logger = dev_obj.get("logger")
    cli = dev_obj.get("ssh")

    flag, has_scm_pool, cli_ret, err_msg = cbbCliUtil \
        .has_smart_cache_pool(cli, lang, logger)
    return flag, has_scm_pool, err_msg


def get_smart_cache_pool_for_nas(dev_obj):
    """获取设备上Smart Cache Pool的ID(只适用于DORADO Nas)
        -1 表示无SmartCachePool
    """
    lang = str(dev_obj.get("lang"))
    logger = dev_obj.get("logger")
    cli = dev_obj.get("ssh")

    flag, scm_pool_id, cli_ret, err_msg = cbbCliUtil \
        .get_smart_cache_pool_for_nas(cli, lang, logger)
    return flag, scm_pool_id, err_msg


def get_smart_cache_pool_disk_for_nas(dev_obj):
    """获取设备上Smart Cache Pool里的Disk ID(只适用于DORADO Nas)

    """
    lang = str(dev_obj.get("lang"))
    logger = dev_obj.get("logger")
    cli = dev_obj.get("ssh")
    pool_id = dev_obj.get("SmartCachePoolID")

    flag, disk_info_list, cli_ret, err_msg = cbbCliUtil \
        .get_smart_cache_pool_disk_for_nas(cli, pool_id, lang, logger)

    if flag:
        return flag, json.dumps(disk_info_list), ""
    else:
        return flag, "", err_msg


def get_free_scm_disk(dev_obj):
    """获取设备上空闲的SCM盘
    """
    lang = str(dev_obj.get("lang"))
    logger = dev_obj.get("logger")
    cli = dev_obj.get("ssh")

    flag, disk_info_list, cli_ret, err_msg = cbbCliUtil \
        .get_free_scm_disk(cli, lang, logger)

    if flag:
        return flag, json.dumps(disk_info_list), ""
    else:
        return flag, "", err_msg


def get_smart_pool(dev_obj):
    """获取设备上Smart Pool的ID
        -1 表示无SmartCachePool
    """
    lang = str(dev_obj.get("lang"))
    logger = dev_obj.get("logger")
    cli = dev_obj.get("ssh")
    is_new_oceanstor = bool(dev_obj.get("is_new_oceanstor"))

    smart_pool_ids = []
    smart_pool_to_is_encryption = {}

    flag, smart_pool_ids, cli_ret, smart_pool_to_domain_ids = get_smart_pool_info_by_cli(cli, lang, logger,
                                                                                         is_new_oceanstor)
    if not flag:
        return True, smart_pool_ids, smart_pool_to_is_encryption

    for smart_pool_id in smart_pool_ids:
        is_encryption = is_smart_pool_encryption(cli, lang, smart_pool_id) and has_encryption_license(dev_obj)
        smart_pool_to_is_encryption[smart_pool_id] = str(is_encryption)
    return flag, smart_pool_ids, smart_pool_to_is_encryption


def get_performance_layer_redundant_strategy(dev_obj):
    lang = str(dev_obj.get("lang"))
    logger = dev_obj.get("logger")
    cli = dev_obj.get("ssh")
    is_new_oceanstor = bool(dev_obj.get("is_new_oceanstor"))
    redundant_strategy_dict = dict()
    flag, smart_pool_ids, cli_ret, smart_pool_to_domain_ids = get_smart_pool_info_by_cli(cli, lang, logger,
                                                                                         is_new_oceanstor)
    for smart_pool_id in smart_pool_ids:
        redundant_strategy_dict[smart_pool_id] = get_one_performance_layer_redundant_strategy(cli, lang, smart_pool_id)
    return (json.dumps(redundant_strategy_dict),)


def get_one_performance_layer_redundant_strategy(cli, lang, performance_layer_id):
    cmd = "show performance_layer general performance_layer_id={}".format(performance_layer_id)
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    dict_list = cliUtil.getVerticalCliRet(cli_ret)
    for record in dict_list:
        return record.get("Redundancy Strategy", "--")


def is_smart_pool_encryption(cli, lang, smart_pool_id):
    cmd = "show performance_layer general performance_layer_id={}".format(smart_pool_id)
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return False
    for record in cliUtil.getVerticalCliRet(cli_ret):
        return record.get("Disk Encryption Switch") == "On"


def is_smart_pool_share(cli, lang, smart_pool_id):
    cmd = "show performance_layer general performance_layer_id={}".format(smart_pool_id)
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return False
    for record in cliUtil.getVerticalCliRet(cli_ret):
        return record.get("Share Strategy") == "Share"


def get_smart_pool_strategy(cli, lang, logger, is_new_oceanstor):
    smart_pool_strategy_dict = {}
    flag, smart_pool_ids, cli_ret, smart_pool_to_domain_ids = get_smart_pool_info_by_cli(
        cli, lang, logger, is_new_oceanstor)
    for smart_pool_id in smart_pool_ids:
        smart_pool_strategy_dict[smart_pool_id] = is_smart_pool_share(cli, lang, smart_pool_id)
    return flag, smart_pool_strategy_dict


def get_smart_pool_info_by_cli(cli, lang, logger, is_new_oceanstor):
    """
    获取设备上的性能层ID
    """
    smart_pool_ids = []
    smart_pool_to_domain_ids = {}
    logger.info("when getting performance infos, the device is new OceanStor: {}".format(str(is_new_oceanstor)))
    if is_new_oceanstor:
        cmd = ("show performance_layer general |filterColumn include columnList=ID,Associated\sDisk\sDomain\sId\sList")
    else:
        cmd = (
            "show performance_layer general |filterRow column=Share\sStrategy "
            "predict=equal_to value=Share|filterColumn include "
            "columnList=ID,Associated\sDisk\sDomain\sId\sList"
        )
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)

    if not cliUtil.hasCliExecPrivilege(cli_ret):
        return True, smart_pool_ids, "", smart_pool_to_domain_ids

    if flag is not True:
        # 6.1.3 会报Error: Failed to process the message.
        return False, smart_pool_ids, cli_ret, smart_pool_to_domain_ids

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return True, smart_pool_ids, cli_ret, smart_pool_to_domain_ids

    ret_dict = cliUtil.getHorizontalCliRet(cli_ret)
    for record in ret_dict:
        smart_pool_id = record.get("ID")
        domain_id = record.get("Associated Disk Domain Id List")
        smart_pool_ids.append(smart_pool_id)
        smart_pool_to_domain_ids[smart_pool_id] = domain_id
    logger.info(
        "smart_pool_ids is: {}, smart_pool_to_domain_ids is: {}".format(smart_pool_ids, smart_pool_to_domain_ids))
    return flag, smart_pool_ids, cli_ret, smart_pool_to_domain_ids


def get_smart_pool_disk_info(dev_obj):
    """
    获取设备上的性能层ID硬盘信息
    :param cli:
    :param lang:
    :param logger:
    :return: 是否执行成功，最大容量，有几种容量的盘, 盘类型
    """
    lang = str(dev_obj.get("lang"))
    logger = dev_obj.get("logger")
    cli = dev_obj.get("ssh")
    is_new_oceanstor = bool(dev_obj.get("is_new_oceanstor"))

    cap_num = []
    disk_type_list = []
    flag, smart_pool_ids, ret, domain_id = get_smart_pool_info_by_cli(cli, lang, logger, is_new_oceanstor)
    if not flag or not domain_id:
        return True, 0, len(cap_num), ",".join(list(set(disk_type_list))), ""

    cmd = "show disk general"
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if not cliUtil.hasCliExecPrivilege(cli_ret):
        return flag, 0, len(cap_num), ",".join(list(set(disk_type_list))), ""

    if flag is not True:
        return flag, 0, len(cap_num), ",".join(list(set(disk_type_list))), ""

    if cliUtil.queryResultWithNoRecord(cli_ret):
        return flag, 0, len(cap_num), ",".join(list(set(disk_type_list))), ""

    smart_pool_ids = [str(x) for x in smart_pool_ids]
    logger.info("the smart_pool_ids is {}".format(smart_pool_ids))
    flag, max_cap, max_cap_unit, member_disk_dict, disk_show_list = get_member_disk_info(
        cap_num, cli, cli_ret, disk_type_list, lang, logger, smart_pool_ids
    )
    cap_num = list(set(cap_num))
    res_str_list = []
    for engine_id, tmp_cap_dict in member_disk_dict.items():
        result_str = [
            "{}:{}".format(key, value) for key, value in tmp_cap_dict.items()
        ]
        res_str_list.append("{}@{}".format(engine_id, ",".join(result_str)))
    return (
        flag,
        str(max_cap) + max_cap_unit,
        len(cap_num),
        ",".join(list(set(disk_type_list))),
        "#".join(res_str_list),
        disk_show_list
    )


def get_member_disk_info(
        cap_num, cli, cli_ret, disk_type_list, lang, logger, smart_pool_ids
):
    ret_dict = cliUtil.getHorizontalCliRet(cli_ret)
    max_cap = 0
    max_cap_unit = ""
    max_cap_value = 0
    flag, elabel_capacity_map, err_msg = cbbCliUtil.get_elabel_capacity(
        cli, lang, logger
    )
    member_disk_dict = {}
    smart_pool_show_disk_dict = {}
    logger.info("the ret_dict is : {}".format(ret_dict))
    for record in ret_dict:
        role = record.get("Role")
        disk_id = record.get("ID", "")
        disk_type = record.get("Type", "")
        eng_id = disk_id.split(".")[0]
        eng_num = re.compile("DAE(\d)").findall(eng_id)
        if eng_num:
            eng_id = "CTE{}".format(eng_num[0])
        # 产品显示未修改，Disk Domain ID实际是flash_layer_id
        performance_layer_id = record.get("Performance Layer ID")
        if performance_layer_id not in smart_pool_ids or role != "Member Disk":
            continue
        elabel_capacity, elabel_unit = get_elabel_cap_value(
            disk_type_list, elabel_capacity_map, logger, record
        )
        logger.info("elabel_capacity:{}{}".format(elabel_capacity, elabel_unit))
        if not elabel_capacity:
            continue
        get_smart_pool_show_disks(
            smart_pool_show_disk_dict, "{}{}".format(elabel_capacity, elabel_unit), disk_type
        )
        tmp_member_disk_dict = member_disk_dict.get(eng_id, {})
        tmp_cap = "{}{}".format(elabel_capacity, elabel_unit)
        tmp_member_disk_dict[tmp_cap] = tmp_member_disk_dict.get(tmp_cap,
                                                                 0) + 1
        member_disk_dict[eng_id] = tmp_member_disk_dict
        max_cap, max_cap_unit, max_cap_value = get_max_disk_cap(
            cap_num, elabel_capacity, elabel_unit,
            max_cap_value, max_cap_unit, max_cap
        )
        logger.info("max_cap:{}, max_cap_value:{}, elabel_capcity:{}".format(
            max_cap, max_cap_value, elabel_capacity)
        )
    disk_show_list = set_smart_pool_show_disks(smart_pool_show_disk_dict)
    return flag, max_cap, max_cap_unit, member_disk_dict, disk_show_list


def get_smart_pool_member_disk_info(dev_obj):
    lang = str(dev_obj.get("lang"))
    logger = dev_obj.get("logger")
    cli = dev_obj.get("ssh")
    is_new_oceanstor = bool(dev_obj.get("is_new_oceanstor"))
    member_disk_dict = {}
    flag, smart_pool_strategy_dict = get_smart_pool_strategy(cli, lang, logger, is_new_oceanstor)
    if not flag or not smart_pool_strategy_dict:
        return True, member_disk_dict, smart_pool_strategy_dict

    cmd = "show disk general"
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return flag, member_disk_dict, smart_pool_strategy_dict
    if cliUtil.queryResultWithNoRecord(cli_ret):
        return flag, member_disk_dict, smart_pool_strategy_dict

    smart_pool_ids = [str(x) for x, value in smart_pool_strategy_dict.items()]
    ret_dict = cliUtil.getHorizontalCliRet(cli_ret)
    cli.close()
    for record in ret_dict:
        role = record.get("Role")
        disk_id = record.get("ID", "")
        eng_id = disk_id.split(".")[0]
        eng_num = re.compile("DAE(\d)").findall(eng_id)
        if eng_num:
            eng_id = "CTE{}".format(eng_num[0])
        performance_layer_id = record.get("Performance Layer ID")
        if performance_layer_id not in smart_pool_ids or role != "Member Disk":
            continue
        temp_disk_dict = member_disk_dict.get(performance_layer_id, {})
        temp_disk_dict[eng_id] = temp_disk_dict.get(eng_id, 0) + 1
        member_disk_dict[performance_layer_id] = temp_disk_dict
    return flag, member_disk_dict, smart_pool_strategy_dict


def get_smart_pool_show_disks(smart_pool_show_disk_dict, elabel_show_cap, disk_type):
    tmp_cap_dict = smart_pool_show_disk_dict.get(disk_type, {})
    tmp_cap_dict[elabel_show_cap] = tmp_cap_dict.get(elabel_show_cap, 0) + 1
    smart_pool_show_disk_dict[disk_type] = tmp_cap_dict


def set_smart_pool_show_disks(smart_pool_show_disk_dict):
    disk_show_list = []
    for disk_type, cap_info in smart_pool_show_disk_dict.items():
        for cap, disk_num in cap_info.items():
            show_disk = dict()
            show_disk["diskNum"] = str(disk_num)
            show_disk["diskType"] = disk_type
            show_disk["capUnit"] = cap
            disk_show_list.append(show_disk)
    return disk_show_list


def get_elabel_cap_value(disk_type_list, elabel_capacity_map, logger, record):
    """
    获取电子标签容量
    :param disk_type_list:
    :param elabel_capacity_map:
    :param logger:
    :param record:
    :return:
    """
    disk_type_list.append(record.get("Type"))
    capacity = record.get("Capacity", "")
    if str(capacity).endswith("TB"):
        # 转为GB比较
        capacity_value = float(capacity.replace("TB", "")) * 1024
    else:
        capacity_value = float(capacity.replace("GB", ""))
    elabel_capacity, elabel_unit = get_elabel_capacity(
        capacity_value, elabel_capacity_map, logger
    )
    return elabel_capacity, elabel_unit


def get_max_disk_cap(
        cap_num, elabel_capacity, elabel_unit, max_cap_value, max_cap_unit, max_cap
):
    elabel_capacity_gb = float(elabel_capacity) if elabel_unit == 'GB' else float(elabel_capacity) * 1000
    if elabel_capacity_gb > max_cap_value:
        max_cap_value = elabel_capacity_gb
        max_cap = elabel_capacity
        max_cap_unit = elabel_unit
    cap_num.append(elabel_capacity_gb)
    return max_cap, max_cap_unit, max_cap_value
