# -*- coding: UTF-8 -*-
import cliResource
from frame.common.regex import Regex
from cbb.frame.rest import restUtil
from com.huawei.ism.exception import IsmException

CLI_EXECUTE_CMD_SUCCESS = "Command executed successfully"
DEBUG_MODEL_FLAG = ":/diagnose>"
DEVELOPER_MODEL_FLAG = "developer:/>"
CLI_RET_END_FLAG = ":/>"
DIAGNOSE_RET_END_FLAG = "diagnose>"
MINISYSTEM_MODEL_FLAG = "minisystem>"

RESULT_NOCHECK = "NOCHECK"
RESULT_NOSUPPORT = "NOSUPPORT"
RESULT_WARNING = "WARNING"
RESULT_DICT = {
   True:"0",
   False:"1",
   RESULT_NOCHECK:"2",
   RESULT_NOSUPPORT:"3",
   RESULT_WARNING:"4",
}

class CheckStatus():
    ERROR = 'EEROR'

#优先级从高到底
RESULT_LIST = [False, RESULT_WARNING, RESULT_NOCHECK, RESULT_NOSUPPORT, True]

def getMsg(lang, msg, args="", resource=cliResource.MESSAGES_DICT):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    '''
    errMsg = "\n--"
    
    try:
        if not resource.has_key(msg):
            return errMsg
        
        localeDict = resource.get(msg)
        if not localeDict.has_key(lang):
            return errMsg
        
        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except:
        return errMsg
    
def isChinese(lang):
    '''
    @summary: 判断lang是否为中文
    @param lang: 语言lang
    @return: 
        True: 中文
        False: 非中文
    '''
    if lang == "zh":
        return True
    return False

def isNoneLicense(cliRet):
    '''
    @summary: 判断回显结果里面是否包含需要license
    @param cliRet: cli回显
    @return: 
        True: cli回显包含需要license
        False: cli回显不包含需要license
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "license" in lowerLine and ("error" in lowerLine or "suggestion" in lowerLine):
            return True
    
    return False

def hasCliExecPrivilege(cliRet):
    '''
    @summary: 判断是否具有执行cli命令的权限
    @param cliRet: cli回显
    @return: 
        True: 具有执行cli命令的权限
        False: 不具有执行cli命令的权限
    '''
    for line in cliRet.splitlines():
        if line.strip() == "^":
            return False
        if "does not have the permission" in line:
            return False
    return True

def queryResultWithNoRecord(cliRet):
    '''
    @summary: 判断回显是否为Command executed successfully
    @param cliRet: cli回显
    @return: 
        True: cli回显中包含Command executed successfully
        False: cli回显中不包含Command executed successfully
    '''
    if Regex.find(CLI_EXECUTE_CMD_SUCCESS, cliRet, Regex.IGNORECASE):
        return True
    return False

def isInternalError(cliRet):
    '''
    @summary: 判断回显结果是否包含内部错误信息
    @param cliRet: cli回显
    @return: 
        True: cli回显包含内部错误信息
        False: cli回显不包含内部错误信息
    '''
    lowerCliRet = cliRet.lower()
    if "internal error" in lowerCliRet:
        return True
    else:
        return False

def isSupportFeature(cliRet):
    '''
    @summary: 判断是否支持某特性
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "error:" in lowerLine and "does not support" in lowerLine:
            return True
    return False
    

def getUserName(cli, lang):
    '''
    @summary: 获取设备上登录的用户名
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回设备上登录的用户名
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""
    userName = ""

    try:
        cmd = "show system general"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True: 
            return checkRet
        
        cliRet = checkRet[1]
        lineList = cliRet.splitlines()
        
        if len(lineList) == 0:
            errMsg = getMsg(lang, "cannot.get.sys.info")
            return (False, cliRet, errMsg)
        
        lastLine = lineList[-1]
        if CLI_RET_END_FLAG in lastLine:
            userName = lastLine.replace(" ", "").replace(CLI_RET_END_FLAG, "")
            return (True, userName, errMsg)
    
    except:
        errMsg = getMsg(lang, "cannot.get.sys.info")
        return (False, cliRet, errMsg)
    
    return (False, cliRet, errMsg)

def getUserPrivilege(cli, lang):
    '''
    @summary: 获取用户权限级别
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回用户权限级别
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""

    try:
        checkRet = getUserName(cli, lang) 
        if checkRet[0] != True:
            return checkRet
        
        userName = checkRet[1]
        cmd = "show user user_name=%s" % userName
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True: 
            return checkRet
        
        cliRet = checkRet[1]
        cliRetLinesList = getHorizontalCliRet(cliRet)
        
        if len(cliRetLinesList) == 0:
            errMsg = getMsg(lang, "cannot.get.user.info")
            return (False, cliRet, errMsg)
        
        for line in cliRetLinesList:
            name = line["Name"]
            if name == userName:
                userPrivilege = line["Level"]
                return (True, userPrivilege, errMsg)

    except:
        errMsg = getMsg(lang, "cannot.get.user.info")
        return (False, cliRet, errMsg)
    
    return (False, cliRet, errMsg)

            
def enterDeveloperMode(cli, lang, debugPasswd=None):
    '''
    @summary: 进入developer模式
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 进入developer模式成功
            False: 进入developer模式失败
        cliRet: cli回显
        errMsg: 进入developer模式失败时的错误消息
    '''
    errMsg = ""
    cliRet = ""
    cliRetAll = ""
    cmd = "change user_mode current_mode user_mode=developer"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]: 
        return checkRet
    cliRet = checkRet[1]
    cliRetAll += cliRet
    
    # 需要确认直接输入y
    cnt = 1
    while "y/n" in cliRet and cnt <= 3:
        cliRet = cli.execCmd("y")
        cnt += 1
    cliRetAll += cliRet
    
    if isInDeveloperMode(cliRet):
        return (True, cliRetAll, errMsg)
    
    if "password" in cliRet.lower():
        errMsg = getMsg(lang, "cannot.access.developer.mode")
        for i in range(0, 3):
            checkRet = excuteCmdInCliMode(cli, debugPasswd, False, lang)
            cliRet = checkRet[1]
            cliRetAll += cliRet
            if isInDeveloperMode(cliRet):
                return (True, cliRetAll, errMsg)
            if "password is wrong" in cliRet.lower():
                errMsg = getMsg(lang, "developer.password.is.wrong")
            
    return (False, cliRetAll, errMsg)

def developerMode2CliMode(cli):
    '''
    @summary: developer模式下退回到cli模式
    @param cli: cli对象
    '''
    cliRet = cli.execCmd("exit")
    if "y/n" in cliRet:
        cli.execCmd("n")
        
    return None
        
def isInDeveloperMode(cliRet):
    '''
    @summary: 判断当前是否在developer模式下
    @param cliRet: cli回显
    @return: 
        True: 当前在developer模式下
        False: 当前不在developer模式下
    '''
    if Regex.find(DEVELOPER_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False

def checkLineInBlackList(cliLine, lang):
    '''
    @summary: 判断cli语句行是否在黑名单中
    @param cliLine: cli语句行
    @param lang: 语言lang
    @return: 
        True: cli语句行在黑名单中
        False: cli语句行不在黑名单中
    '''
    errMsg = ""
    
    for dictItems in cliResource.BLACKLIST_DICT:
        if dictItems.get("key_word") in cliLine:
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)

def excuteCmdInCliMode(cli, cmd, isHasLog, lang):
    '''
    @summary: 获取cli模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        cliRet: cli回显
        errMsg: 错误消息
    '''
    errMsg = ""
    cliRet = ""
    
    if cli is None:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (RESULT_NOCHECK, "", errMsg)
    
    try:
        if isHasLog:
            cliRet = cli.execCmd(cmd)
        else:
            cliRet = cli.execCmdNoLog(cmd)
    except IsmException:
        raise
    
    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)
    
    if not hasCliExecPrivilege(cliRet):
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (RESULT_NOCHECK, cliRet, errMsg)
    
    if isNoneLicense(cliRet):
        errMsg = getMsg(lang, "has.not.license")
        return (RESULT_NOSUPPORT, cliRet, errMsg)
    
    if queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    
    if isInternalError(cliRet):
        return (True, cliRet, errMsg)
    
    if isSupportFeature(cliRet):
        return (RESULT_NOSUPPORT, cliRet, errMsg)
    
    lineList = cliRet.splitlines()
    for line in lineList:
        checkRet = checkLineInBlackList(line, lang)
        if checkRet[0]:
            errMsg = getMsg(lang, "system.status.abnormal")
            return (False, cliRet, errMsg)  
    
    return (True, cliRet, errMsg)

def excuteCmdInDeveloperMode(cli, cmd, isHasLog, lang, debugPasswd=None, preExec=True):
    '''
    @summary: 获取developer模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    isOpened = False
    try:
        if preExec:
            #兼容部分命令视图切换，先尝试在普通用户模式下执行，再尝试在研发模式下执行
            checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
            if checkRet[0] == True:
                return checkRet
            
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet
            
        enterDeveloperCheckRet = enterDeveloperMode(cli, lang, debugPasswd)
        if enterDeveloperCheckRet[0] != True:
            return enterDeveloperCheckRet
    
        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
        cliRet = checkRet[1]
        
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
        return checkRet
    except IsmException:
        raise
    finally:
        if isOpened == True:
            closeDeveloperSwitch(cli, lang)

def getProductModel(cli, lang):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    product_model = restUtil.CommonRest.getOriginProductModel(cli.getdNode())
    if product_model:
        return True, product_model, ""
    cmd = "show system general"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    product_model = ""
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Model":
            product_model = fieldValue
            
        if len(product_model) != 0 and product_model != "--":
            return (True, product_model, "")
        
    return (False, cliRet, getMsg(lang, "cannot.get.product.model.info"))

def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            match = Regex.search("^\s*-+(\s+-+)*\s*$", line)
            if match != None:
                headline = match.group(0)
                break
            i += 1
            
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        tuple_idxs = Regex.getStartEndList("\s*-+\s*", headline)
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break
            
            if len(line.strip()) == 0:
                continue
            
            if len(line) <= requiredLineLen:
                continue
            
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
    
def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break
            
        if Regex.find("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()
            
        fields = line.split(" : ")
        if len(fields) < 2:
            continue
        
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    
    if len(lineDict) > 0:
        dictList.append(lineDict.copy())
    
    return dictList

def getHorizontalDebugRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            match = Regex.search("^\s*-+(\s+-+)*\s*$", line)
            if match != None:
                headline = match.group(0)
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        tuple_idxs = Regex.getStartEndList("\s*-+\s*", headline)
           
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        dictList = []
        for line in field_words:
            if DIAGNOSE_RET_END_FLAG in line:
                break
            
            if len(line.strip()) == 0:
                continue
            
            vals = line.split()
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
    
def getSplitedCliRet(cliRet, splitMark):
    '''
    @summary: 按分割标记从回显中获取仅包含该分割标记对应的的回显
    @param cliRet: cli回显
    @param splitMark: 分割标记 
    '''
    cliRetlist = cliRet.split(splitMark)
    if len(cliRetlist) < 2:
        return ""
    splitedCliRet = cliRetlist[1]

    match = Regex.search("^.*:(/>)*\r*\n*$", splitedCliRet, Regex.MULTILINE)
    if match != None:
        splitedCliRet = splitedCliRet[0:match.start()]
    
    if len(splitedCliRet.strip()) == 0:
        return ""
    else:
        return splitedCliRet

def hasSuperAdminPrivilege(cli, lang):
    '''
    @summary: 判断用户是否具有超级管理员权限
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时
                True: 具有超级管理员权限
                False： 不具有超级管理员权限
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    checkRet = getUserPrivilege(cli, lang)
    if not checkRet[0]:
        return checkRet
    
    userLevel = checkRet[1]
    if userLevel is None or len(userLevel) == 0 or userLevel.lower() not in ["super_admin"]:
        return (True, False, "")
    else:
        return (True, True, "")
    
#------------------------------------------------------------------------------
#debug mode factory.           time:2015/8/25 10:26
def isInDebugMode(cliRet):
    """ 
    @summary: check the ":/diagnose>" is in the return strings.
    """
    if Regex.find(DEBUG_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False

def enterDebugMode(cli, lang, debugPasswd=None):
    """
    @summary: first enter the developer mode,then the debug mode.
    """
    enterDeveloperCheckRet = enterDeveloperMode(cli, lang, debugPasswd)
    if not enterDeveloperCheckRet[0]:
        developerMode2CliMode(cli)
        return enterDeveloperCheckRet
    
    cmd = "debug"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    if isInDebugMode(cliRet):
        return (True, cliRet, "")
    else:
        errMsg = getMsg(lang, "has.not.cli.privilege") 
        return (RESULT_NOCHECK, cliRet, errMsg)
    
def executeCmdInDebugMode(cli, cmd, isHasLog, lang, debugPasswd=None):
    """@summary: execute cmd in debug mode
    """
    isOpened = False
    try:
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet
            
        enterDebugCheckRet = enterDebugMode(cli, lang, debugPasswd)
        if enterDebugCheckRet[0] != True:
            return enterDebugCheckRet
        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
        
        cliRet = checkRet[1]
        if isInDebugMode(cliRet):
            exitDebugModeToCliMode(cli)
        
        return checkRet
    except IsmException:
        raise
    finally:
        if isOpened == True:
            closeDeveloperSwitch(cli, lang)
    
def exitDebugModeToCliMode(cli):
    """
    @summary: first exit the debug mode,then exit the developer mode.
    """    
    cliRet = cli.execCmd("exit")
    if "y/n" in cliRet:
        cliRet = cli.execCmd("n")
    if isInDeveloperMode(cliRet):
        developerMode2CliMode(cli)
        

def GetCliRecsInDebugMode(cli, cmdList, logger, lang, debugPasswd=None):
    """
    @summary: Execute the cli command in the debug mode.
    @param cmdList:one or more cmd.
    @return: echo list. 
    """
    cliRecsList = list()
    for cmd in cmdList:
        try:
            logger.logInfo("Start to execute the cli cmd:%s" % cmd)
            checkRet = executeCmdInDebugMode(cli, cmd, False, lang, debugPasswd)
            cmdRecs = checkRet[1]
        except Exception, msg:
            logger.logInfo("The Exception is:%s" % msg)
            cmdRecs = ""
        logger.logInfo("The cliRecs is %s" % unicode(cmdRecs))
        cliRecsList.append(cmdRecs)
    return cliRecsList

  
def closeCliConnection(cli):
    """
    @summary: 关闭cli连接
    @param cli:cli连接
    """ 
    try:
        cli.close()
    except:
        return
    return   

#------------------------------------------------------------------------------
#open tlv channel.           time:2016/01/19 14:50
def execCmdInDeveloperModePrompt(cli, cmd, isHasLog, lang, prompt="y"):
    '''
    @summary: 获取developer模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @param prompt: 确认命令(y/n)
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    isOpened = False
    try:
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet 
        cliRetAll = ""
        enterDeveloperCheckRet = enterDeveloperMode(cli, lang)
        if not enterDeveloperCheckRet[0]:
            developerMode2CliMode(cli)
            return enterDeveloperCheckRet
        
        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
        cliRet = checkRet[1]
        cliRetAll += cliRet
        
        cnt = 0
        while ("y/n" in cliRet and cnt < 5):
            ret = excuteCmdInCliMode(cli, prompt, isHasLog, lang)
            cliRet = ret[1]
            cliRetAll += cliRet
            cnt += 1
        
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
        
        return (checkRet[0], cliRetAll, checkRet[2])
    finally:
        #关闭开关
        if isOpened == True :
            closeDeveloperSwitch(cli, lang)
            
def changeTlvChannel(cli, lang, switch):
    '''
    @summary: 打开或关闭tlv端口
    @param cli: cli对象
    @param switch: 开关状态
    @return: (falg, ret)
        flag:
            True:  执行成功
            False: 执行失败
        ret: cli回显
    '''
    
    cliRet = ""
    try:
        
        cmd = "change system external_TLV_channel enabled=%s" % switch
        checkRet = execCmdInDeveloperModePrompt(cli, cmd, True, lang)
        cliRet = checkRet[1]
        if not hasCliExecPrivilege(cliRet):
            return (True, cliRet)
        
        if checkRet[0] != True:
            return (False, cliRet)
        
        if CLI_EXECUTE_CMD_SUCCESS in cliRet:
            return (True, cliRet)
        else:
            return (False, cliRet)
    except:
        return (False, cliRet)
    
def openTlvChannel(cli, lang):
    '''
    @summary: 打开或关闭tlv端口
    @param cli: cli对象
    @return: (falg, ret, errMsg, suggestion)
        flag:
            True:  执行成功
            False: 执行失败
        ret: cli回显
        errMsg: 错误消息
        suggestion: 修复建议
    '''
    errMsg = ""
    suggestion = ""
    flag, cliRet = changeTlvChannel(cli, lang, "yes")
    if flag:
        return (True, cliRet, errMsg, suggestion)
    else:
        if lang == "zh":
            errMsg = u"开启TLV通道失败。"
            suggestion = u"请联系技术支持工程师协助处理。"
        else:
            errMsg = "Change the status of the external TLV channel failed."
            suggestion = "Contact technical support engineers for help."
        return (False, cliRet, errMsg, suggestion)

def closeTlvChannel(cli, lang):
    '''
    @summary: 打开或关闭tlv端口，关闭失败不受影响。
    @param cli: cli对象
    @return: (falg, cliRet)
        flag:
            True:  执行成功
            False: 执行失败
        cliRet: cli回显
    '''
    flag, cliRet = changeTlvChannel(cli, lang, "no")
    return (True, cliRet)

def openDeveloperSwitch(cli, lang):
    '''
            打开切换developer模式开关
    '''
    opencmd = "change user_mode enabled user_mode=developer enabled=yes"
    return excuteCmdInCliMode(cli, opencmd, True, lang)

def needOpenDeveloperSwitch(cli, lang):
    '''
    @summary: #兼容V3R6版本，查看切developer模式开关是否打开
    '''
    developercmd = "show user_mode enabled"
    checkRet = excuteCmdInCliMode(cli, developercmd, True, lang)
    if checkRet[0] != True:
        return (CheckStatus.ERROR, checkRet)
    
    cliRetLinesList = getVerticalCliRet(checkRet[1])
    for line in cliRetLinesList:
        if "Disabled" == line.get("Developer View", ""):
            return (True, checkRet)
        
    return (False, checkRet)
        
def closeDeveloperSwitch(cli, lang):
    '''
    @summary: 关闭developer视图开关
    '''
    try:
        closecmd = "change user_mode enabled user_mode=developer enabled=no"
        excuteCmdInCliMode(cli, closecmd, True, lang)
    except:
        return
    return

def getHorizontalNostandardCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合,此方法用来解析CLI回显未对其情况
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines() 
        for line in cliRetList:
            match = Regex.search("^\s*-+(\s+-+)*\s*$", line)
            if match != None:
                headline = match.group(0)
                break
            i += 1
            
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        tuple_idxs = Regex.getStartEndList("\s*-+\s*", headline)
 
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        lenkeys = len(keys)
        dictList = []
        for line in field_words:
            if line.find(":/>") >= 0:
                break
            
            if Regex.find("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            vals = []
            valueList = line.strip().split("  ")
            
            valueSpaceList =[]
            for value in valueList:
                if value != "":
                    vals.append(value.strip().decode("utf8"))
                    valueSpaceList.append(value)
            lenvalue = len(valueSpaceList)
            if lenvalue == lenkeys:
                dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []

def queryDiskInfo(cli, lang, logger):
    '''
    @summary: 查询硬盘信息
    '''
    diskInfoList = {}
    sortedList = {}
    cliRet = ""
    errMsg = ""

    try:
        cmd = "show disk general |filterColumn include columnList=ID,Manufacturer,Model,Type,Role,Firmware\sVersion,Serial\sNumber,Health\sStatus,Running\sStatus,Disk\sDomain\sID"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if not checkRet[0]: 
            logger.info("Execute cmd[%s] error" %cmd)
            return (False, sortedList, checkRet[2])
        
        cliRet = checkRet[1]
        diskInfoList = getHorizontalCliRet(cliRet)
        if len(diskInfoList) == 0:
            logger.info("Get vertical cliRet failed")
            errMsg = getMsg(lang, "cannot.get.disk.info")
            return (False, diskInfoList, errMsg)
        
        return (True, diskInfoList, "")
    except Exception, e:
        logger.error("Query disk info except:%s"%str(e))
        errMsg =  getMsg(lang,"cannot.get.disk.info")
        return (False, sortedList, errMsg)
    
def getSystemDate(cli,lang):
    '''
    @summary: 获取阵列系统时间，时间格式：2017-7-19/12:00:00
    '''
    sysDate = ""
    cmd = "show system general"
    ret = excuteCmdInCliMode(cli, cmd, True, lang)
    if ret[0] != True:
        return (sysDate, ret[1])

    cliRet = ret[1]
    sysInfoList = getVerticalCliRet(cliRet)    
    for sysInfo in sysInfoList:
        sysDateInfo = sysInfo.get("Time","")
        if sysDateInfo == "":
            continue
        sysDateInfoList = sysDateInfo.split(" ")
        if len(sysDateInfoList) < 1:
            return (sysDate, cliRet)
        
        sysDate = sysDateInfoList[0]
        return (sysDate,cliRet)
    
    return (sysDate,cliRet)

def getEventDetail(cli,lang,sequence):
    '''
    @summary: 通过流水号获取event详细信息
    '''
    eventDetail = {}
    cmd = "show event sequence=%s" %str(sequence)
    ret = excuteCmdInCliMode(cli, cmd, True, lang)
    if ret[0] != True:
        return eventDetail
    
    eventDetailList = getVerticalCliRet(ret[1]) 
    if len(eventDetailList) != 0:
        return eventDetailList[0]
    return eventDetail

def getDiskById(cli,lang,diskId):
    '''
    @summary: 获取单个硬盘信息
    '''
    diskDetail = {}
    cmd = "show disk general disk_id=%s" %str(diskId)
    ret = excuteCmdInCliMode(cli, cmd, True, lang)
    if ret[0] != True:
        return diskDetail
    
    diskList = getVerticalCliRet(ret[1]) 
    if len(diskList) != 0:
        return diskList[0]
    return diskDetail

def isInMinisystemMode(cliRet):
    '''
    @summary: 判断当前是否在minisystem模式下
    '''
    if Regex.find(MINISYSTEM_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False

def minisystemMode2CliMode(cli):
    '''
    @summary: developer模式下退回到cli模式
    @param cli: cli对象
    '''
    cliRet = cli.execCmd("exit")
    if "y/n" in cliRet:
        cli.execCmd("y")
    if isInDeveloperMode(cliRet):
        developerMode2CliMode(cli)

    return None

def excuteCmdInMinisystemModel(cli, cmd, lang):
    """
    @summary: 在minisystem模式下执行命令
    """
    isOpened = False
    try:
        cliRet = cli.execCmd(cmd)
        if "password:" in cliRet:
            return True, cliRet, ''

        if isInMinisystemMode(cliRet):
            return (True, cliRet, "")

        if isInDebugMode(cliRet):
            cliRet = cli.execCmd("exit")
            if "(y/n)" in cliRet:
                cliRet = cli.execCmd("y")

        if isInDeveloperMode(cliRet):
            cliRet = cli.execCmd("minisystem")
            if isInMinisystemMode(cliRet):
                cliRet = cli.execCmd(cmd)
                return (True, cliRet, "")

            return (False, cliRet, "")

        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet
        enterDeveloperCheckRet = enterDeveloperMode(cli, lang)
        if not enterDeveloperCheckRet[0]:
            developerMode2CliMode(cli)
            return enterDeveloperCheckRet

        cliRet = cli.execCmd("minisystem")
        if not isInMinisystemMode(cliRet):
            return (False, cliRet, "")

        cliRet = cli.execCmd(cmd)
        return (True, cliRet, "")
    finally:
        if isInMinisystemMode(cliRet):
            minisystemMode2CliMode(cli)
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)

        if isOpened == True :
            closeDeveloperSwitch(cli, lang)

def getFreeMemInCliRet(cliRet):
    '''
    @summary:从CLI回显获取空闲内存
    :param cliRet: 
    :return: 
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    title_list = ""
    for line in cliRetList:
        if "free" in line:
            title_list = line.split()
            continue
        if "Mem:" in line:
            mem_str = line.split(":")[1]
            continue
    free_index = title_list.index("free")
    mem_list = mem_str.split()
    if len(mem_list) != len(title_list):
        return (False, 0)
    free_mem = mem_list[free_index]

    return (True, free_mem)
