# -*- coding: UTF-8 -*-
from com.huawei.ism.tlv import TLVUtils
from com.huawei.ism.tlv.bean import Param
from com.huawei.ism.tlv.docoder import ParamType
from com.huawei.ism.tlv.lang import UnsignedInt32
from com.huawei.ism.tlv.lang import UnsignedInt64
from com.huawei.ism.exception import IsmException
from frame.tlv import tlvDataType
from frame.tlv import tlvData

RETRY_TIMES = 5 #建立连接时重试次数
TLV_CMD_TIMEOUT = 120
CMD_DEFAULT_TIMEOUT = 60*5 #执行TLV命令默认超时时间
TLV_RECORD_END_INDEX = 4095
ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE = "67108849"
    
def getParamValue(paramType, paramValue):
    '''
    @summary: 将参数转换为tlv参数
    '''
    if paramType == tlvDataType.DATA_TYPE_UNSIGN_INT or \
       paramType == tlvDataType.DATA_TYPE_SIGN_INT or \
       paramType == tlvDataType.DATA_TYPE_BOOL or \
       paramType == tlvDataType.DATA_TYPE_ENUM or \
       paramType == tlvDataType.DATA_TYPE_TIME:
        return UnsignedInt32(paramValue)
    
    elif paramType == tlvDataType.DATA_TYPE_U64LONG_TYPE:
        return UnsignedInt64(paramValue)
    
    elif paramType == tlvDataType.DATA_TYPE_STRING or \
         paramType == tlvDataType.DATA_TYPE_ARRAY or \
         paramType == tlvDataType.DATA_TYPE_JSON:
        return str(paramValue)
    
    return paramValue

def getParamList(params):
    '''
    @summary: 获取tlv参数列表
    '''
    paramList = []
    if params is None or len(params) == 0:
        return paramList
    for paramDefine, paramValue in params:
        paramIndex = paramDefine["index"]
        paramType = paramDefine["type"]
        paramValue = getParamValue(paramType, paramValue)
        paramList.append(Param(paramIndex, paramType, paramValue))
    return TLVUtils.paramList(paramList)

def getRecordValue(record, field):
    '''
    @summary: 获取tlv记录中的值
    @param record:
    @param field:
    '''
    fieldIndex = field["index"]
    fieldType = field["type"]
    
    if fieldType == tlvDataType.DATA_TYPE_UNSIGN_INT or \
       fieldType == tlvDataType.DATA_TYPE_SIGN_INT or \
       fieldType == tlvDataType.DATA_TYPE_ENUM or \
       fieldType == tlvDataType.DATA_TYPE_TIME:
        val = record.getParamIntValue(fieldIndex)
        if val:
            return val.intValue()
        else:
            return 0
    
    elif fieldType == tlvDataType.DATA_TYPE_U64LONG_TYPE:
        val = record.getParamIntValue(fieldIndex)
        if val:
            return val.longValue()
        else:
            return 0
        
    elif fieldType == tlvDataType.DATA_TYPE_STRING or \
         fieldType == tlvDataType.DATA_TYPE_ARRAY or \
         fieldType == tlvDataType.DATA_TYPE_BOOL or \
         fieldType == tlvDataType.DATA_TYPE_JSON:
        val = record.getParamStrValue(fieldIndex)
        if val:
            return str(val)
        else:
            return ""
        
    return record.getParamValue(fieldIndex)

def execCmd(tlv, cmd, params, isBatch=False, getBatch=False, timeOut=TLV_CMD_TIMEOUT):
    '''
    @summary: 执行tlv命令
    @param tlv: tlv对象
    @param cmd: 命令字
    @param params: 参数列表
    @param isBatch: 是否为批量处理（参考GET_BATCH_NEXT的实现的批量处理，批处理最后一条记录有结束符标记）
    @param getBatch: 是否为获取批量内容（参考非GET_BATCH_NEXT的实现的批量处理，批处理最后一条记录没有结束符标记）
    @param timeOut: 超时时间
    '''
    try:
        records = None
        paramList = getParamList(params)
        
        if isBatch:
            records = tlv.bacthNext(cmd, paramList, timeOut, RETRY_TIMES)
            if records is not None and len(records) > 1:
                lastRecord = records[-1]
                lastRecordParamList = lastRecord.getParams()
                if lastRecordParamList[0].getIndex() == TLV_RECORD_END_INDEX:
                    records.pop()
            return records
        
        if getBatch:
            return tlv.getBatch(cmd, paramList, timeOut, RETRY_TIMES)
        
        return tlv.invoke(cmd, paramList, timeOut, RETRY_TIMES)

    except IsmException, ismException:
        errCode = ismException.getErrorId()
        #往从节点下发消息失败，可以忽略。
        if ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE in str(errCode):
            return None
        
        errMsg = ismException.getErrorMessage()
        raise Exception(errCode, errMsg)
        return

def getSystemRecord(tlv):
    '''
    @summary: 获取系统信息
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SYSTEM'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params)

def getProductVersion(tlv):
    '''
    @summary: 获取产品版本
    @param tlv: tlv对象
    '''
    record = getSystemRecord(tlv)
    return getRecordValue(record, tlvData.SYSTEM['productVersion'])

def getPdtModel(productModelEnum):
    '''
    @summary: 获取产品类型
    @param tlv: tlv对象
    '''
    return dict([(tlvData.PRODUCT_MODE_E.get(key), key) for key in tlvData.PRODUCT_MODE_E]).get(productModelEnum)

def getProductModel(tlv):
    '''
    @summary: 获取产品类型
    @param tlv: tlv对象
    '''
    record = getSystemRecord(tlv)
    productModelEnum = getRecordValue(record, tlvData.SYSTEM['productMode'])
    return getPdtModel(productModelEnum)


def getUpgradeRecords(tlv, checkItem):
    '''
    @summary: 执行升级检查的tlv命令
    @param tlv: tlv对象
    @param checkItem:升级检查项索引
    '''
    params = []
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXE_UPD_MSGTYPE'], tlvData.UPD_MSG_E['EXE_UPD_MSGTYPE'])
    params.append(param)
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXE_UPD_ACTIVETYPE'], tlvData.UPD_MSG_E['EXE_UPD_ACTIVETYPE'])
    params.append(param)
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_DIAG_CMD'], tlvData.UPD_CMD_DEFINE[checkItem]['cmd'])
    params.append(param)
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_NODE_CFG'], tlvData.UPD_CMD_DEFINE[checkItem]['id'])
    params.append(param)
    
    return execCmd(tlv, tlvData.CMD["OM_MSG_OP_UPD_EXECUTE"], params, getBatch=True, timeOut=CMD_DEFAULT_TIMEOUT)

def getTLVParams(paramDict):
    """@summary: 解析字典获得参数列表
       @param paramDict：包含参数信息的字典 
       @return: 参数排序后返回
    """
    tempDict = dict()
    for (type, value) in paramDict.items():
        paramInfo = tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE[type]
        tempDict[paramInfo["index"]] = (paramInfo, value)
    indexList = tempDict.keys()
    indexList.sort()
    paramList = [tempDict[item] for item in indexList]
    return paramList

def getAllKindsTLVRecords(tlv, checkItem, logger):
    """
    @summary: 获取tlv消息的通用方法
    """
    paramDict = tlvData.UPD_CMD_DEFINE[checkItem]
    params = getTLVParams(paramDict)
    logger.logInfo("params=%s" % str(params))
    return execCmd(tlv, tlvData.UPD_SPECIAL_CMD[checkItem], params, getBatch=True, timeOut=CMD_DEFAULT_TIMEOUT)
    
def getEnclosureRecords(tlv):
    '''
    @summary: 获取当前集群所有框信息的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getSASPortRecords(tlv):
    '''
    @summary: 获取当前集群所有SAS端口的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SAS_PORT'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def isIBCPasswordChanged(tlv):
    '''
    @summary: 检查IBC密码是否为默认密码
    @param tlv: tlv对象
    '''
    params = []
    cmd = tlvData.CMD["CHECK_IBC_PASSWORD"]
    record = execCmd(tlv, cmd , params)
    return getRecordValue(record, tlvData.CHECK_IBC_PASSWORD['passwordChanged'])

def notifyDistributePackage(tlv):
    CMO_NOTIFY_PACKAGE_PATH = {"index": 0, "type": tlvDataType.DATA_TYPE_STRING}
    pkgPathParam = (CMO_NOTIFY_PACKAGE_PATH, "127.127.127.1:/OSM/update/software.tgz")
    params = [pkgPathParam]
    record = execCmd(tlv, tlvData.CMD['OM_MSG_OP_NOTIFY_PACKAGE_PATH'], params)
    return record

def lstSysProgress(tlv):
    '''
    @summary:查询升级进度
    '''
    record = execCmd(tlv, tlvData.CMD['OM_MSG_OP_UPD_LST_SYS_PROGRESS'], params=[])
    return record
