#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import time

from py.common.util import msg_dict
from py.common.java_adapter import java_adapter

DINGNOSE_MODE_ENDSTR = "diagnose>"
TRY_AGAIN_TIMES = 30
DIAGNOSE_CMD_INTERVAL = 1


def get_msg(msg, args=""):
    """
    @summary: 消息国际化
    @param msg: 消息
    @param args: 消息参数
    @return: 经过国际化处理后的消息
    """
    err_msg = "\n--"

    local_dict = msg_dict.MSG_DICT.get(msg)
    if not local_dict:
        return err_msg

    local_msg = local_dict.get(java_adapter.get_language())
    if not local_msg:
        return err_msg
    if "%s" in local_msg or "%i" in local_msg:
        return local_msg % args
    else:
        return local_msg


def get_format_header_ret(ip, ret):
    """
    统一格式化每个节点开始的回文
    :param ip: ip
    :param ret: 回文
    :return:
    """
    return u"\n------{}------:\n{}\n".format(ip, ret)


def command_result_redirect_to_file(ssh_service, task_env, command, result_file_name):
    """
    执行deploy.sh脚本命令,结果重定向到指定文件
    :param ssh_service: ssh连接
    :param task_env: 上下文对象
    :param command: 脚本执行命令
    :param result_file_name: 输出文件绝对路径
    :return:执行结果最后一行
    """
    end_flag = ["This operation is high risk,please input y to continue:"]
    sh_path = task_env.get_task_cache().get("unzip_folder_path")
    ssh_service.execute_cmd("cd {}".format(sh_path))
    cmd = "bash deploy.sh {} > {}".format(command, result_file_name)
    ssh_service.execute_cmd_no_check_result(cmd, end_flag)
    ssh_service.execute_cmd_with_timeout("y", get_timeout(task_env))
    return ssh_service.execute_cmd("tail -1 {}|cat".format(result_file_name))


def command_result_redirect_to_file_strip_cc(ssh_service, task_env, command, result_file_name):
    return strip_control_characters(command_result_redirect_to_file(ssh_service, task_env, command, result_file_name))


def strip_control_characters(string):
    """  移除字符串中的命令字符  """
    res = []
    lines = string.splitlines()
    for line in lines:
        res.append("".join(i for i in line if 31 < ord(i) < 127))
    return "\n".join(res)


def get_list_file_ips(ssh_service, task_env, file_name):
    """
    查询扩容前进程节点ip文件
    :param ssh_service: ssh连接
    :param file_name: 文件路径
    :return: 节点ip列表
    """
    sh_path = task_env.get_task_cache().get("unzip_folder_path")
    ssh_service.execute_cmd("cd {}".format(sh_path))
    list_file_ips = []
    cli_ret = ssh_service.execute_cmd("cat {}".format(file_name))
    if "No such file or directory" in cli_ret:
        return []
    for line in cli_ret.splitlines()[1:-1]:
        if line:
            list_file_ips.append(line.strip())
    return list_file_ips


def get_timeout(task_env):
    """
    @summary: 根据节点数量设置超时时间
    @param task_env: 上下文对象
    @return: 超时时间
    """
    node_number = len(task_env.get_cluster_node_dict())
    # 产品脚本设有超时时间，这里只补充一层防护
    return node_number * 10 * 60


def attach_app(ssh_service, app_name):
    """
    挂载指定进程
    :param ssh_service: ssh连接
    :param app_name: 指定挂载进程的AppName字段
    :return:
    """
    if not switch_to_diagnose(ssh_service):
        return False
    app_lines = ssh_service.execute_cmd_no_check_result("ls", [DINGNOSE_MODE_ENDSTR])
    for line in app_lines.splitlines():
        if line.strip().endswith(app_name):
            app_id = line.strip().split()[0]
            if "success" in ssh_service.execute_cmd("attach {}".format(app_id)):
                return True
    return False


def switch_to_diagnose(ssh_service):
    for _ in range(TRY_AGAIN_TIMES):
        cli_ret = ssh_service.execute_cmd("diagnose_usr --set-cli")
        # diagnose模式下释放连接可能会导致服务异常，下次切换需要等待数秒
        if "client login fail, server is busy, please wait" in cli_ret:
            time.sleep(DIAGNOSE_CMD_INTERVAL)
            continue
        # 存在其他异常，这里不感知
        return DINGNOSE_MODE_ENDSTR in cli_ret
    return False
