#  coding=utf-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

from py.common.util import common
from py.common.entity.check_result import CheckResult
from py.common.entity.check_item import CheckItem
from py.common.util.connection_util import SshService


def execute(env, method):
    return CheckCcdbServer(env).execute(method)


class CheckCcdbServer(CheckItem):
    def __init__(self, env):
        super(CheckCcdbServer, self).__init__(env)
        self.fsm_dev_node = self.task_env.get_master_fsm_dev_node()
        self.expansion_conf_node_list = self.task_env.get_expansion_conf_nodes()
        self.dev_node_dict = self.task_env.get_cluster_node_dict()

    def do_check(self):
        """
        检查CCDB_SERVER进程
        登录每个CCDB_SERVER节点，检查Role是否为master或slave，WorkStatus字段是否为normal
        """
        # 获得CcdbServerNode.List文件节点ip列表
        ccdb_node_ip_list = common.get_list_file_ips(
            SshService(self.fsm_dev_node), self.task_env, "CcdbServerNode.List")
        # 待检查节点=文件记录ip节点+新扩节点
        node_list = []
        for ccdb_server_node_ip in ccdb_node_ip_list:
            node_list.append(self.dev_node_dict.get(ccdb_server_node_ip))
        for expansion_node in self.expansion_conf_node_list:
            node_list.append(self.dev_node_dict.get(expansion_node.getIp()))
        ori_info, err_msg = self.check_ccdb_local_status(node_list)
        if err_msg:
            return CheckResult(CheckResult.NOT_PASS, ori_info, "\n".join(err_msg))
        return CheckResult(CheckResult.PASS, ori_info, "")

    def check_ccdb_local_status(self, node_list):
        # 依次登录节点检查ccdb_server进程状态
        ori_info, err_msg, master_node_list = [], [], []
        for node in node_list:
            ssh_service = SshService(node)
            # 801老版本设备存在ccdb进程名为ccdb_server,需要做兼容
            if (not common.attach_app(ssh_service, "ccdb")) and (not common.attach_app(ssh_service, "ccdb_server")):
                err_msg.append(self.get_msg("attach.app.fail", (node.getIp(), "ccdb")))
                continue
            cli_ret = ssh_service.execute_cmd("ccdb_server srvcls")
            ori_info.append(cli_ret)
            role, work_status = self.get_ccdb_role_and_status(cli_ret)
            if role not in ["master", "slave"] or work_status != "normal":
                err_msg.append(self.get_msg("check.ccdb.status.not.pass", (node.getIp(), role, work_status)))
            if role == "master":
                master_node_list.append(node.getIp())
            ssh_service.release_ssh()
        # master节点必须唯一
        if not master_node_list:
            err_msg.append(self.get_msg("check.ccdb.status.no.master.fail"))
        elif len(master_node_list) != 1:
            err_msg.append(self.get_msg("check.ccdb.status.not.one.master.fail", ",".join(master_node_list)))
        return ori_info, err_msg

    @staticmethod
    def get_ccdb_role_and_status(cli_ret):
        role, work_status = "", ""
        for line in cli_ret.splitlines():
            if not line:
                continue
            line = line.strip().split()
            if line[0] == "Role":
                role = line[-1]
            if line[0] == "WorkStatus":
                work_status = line[-1]
        return role, work_status
