#  coding=utf-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.
import json
import threading

from py.common.util import common
from py.common.util.check_ip_list_util import check_cms_worker_list, check_ls_list, check_osd_list, check_vbs_list, \
    check_ebs_list, check_node_sentry_list, check_cm_list, check_ccdb_monitor_list, check_ccdb_server_list, \
    check_djob_list, check_dlm_vm_list, check_kmm_server_list, check_fdsa_core_list, check_dlm_mm_list
from py.common.entity.check_result import CheckResult
from py.common.entity.check_item import CheckItem
from py.common.util.connection_util import RestService, SshService

PROCESS_DICT = {"VbsNode.List": check_vbs_list, "EBSCtrlNode.List": check_ebs_list,
                "NodesentryNode.List": check_node_sentry_list, "CmNode.List": check_cm_list,
                "CcdbMonitorNode.List": check_ccdb_monitor_list, "CcdbServerNode.List": check_ccdb_server_list,
                "DjobNode.List": check_djob_list, "DlmVmNode.List": check_dlm_vm_list,
                "DlmMmNode.List": check_dlm_mm_list, "KmmServerNode.List": check_kmm_server_list,
                "CmsWorkerNode.List": check_cms_worker_list, "FdsaCoreNode.List": check_fdsa_core_list,
                "LsNode.List": check_ls_list, "OsdNode.List": check_osd_list}
EXPAND_PROCESS_LIST = ["CmNode.List", "CcdbMonitorNode.List", "CcdbServerNode.List", "DjobNode.List",
                       "DlmVmNode.List", "DlmMmNode.List", "KmmServerNode.List", "FdsaCoreNode.List"]

NETWORK_SERVICE_URL = "/api/v2/network_service/servers"


def execute(env, method):
    return CheckCmipAndNsip(env).execute(method)


class CheckCmipAndNsip(CheckItem):
    def __init__(self, env):
        super(CheckCmipAndNsip, self).__init__(env)
        self.fsm_dev_node = self.task_env.get_master_fsm_dev_node()
        self.path = self.task_env.get_task_cache().get("unzip_folder_path")
        self.old_zk_node_list = self.task_env.get_task_cache().get("oldZkNodes")
        self.expansion_conf_node_list = self.task_env.get_expansion_conf_nodes()
        self.dev_node_dict = self.task_env.get_cluster_node_dict()
        self.cluster = self.task_env.get_dev_node()
        self.err_msg = []
        self.ori_info = []
        self.sem = threading.Semaphore(10)

    @staticmethod
    def need_check_in_cfg(ip_address_list, management_ip_list):
        for ip_address in ip_address_list:
            if "management_internal" in ip_address.get("ip_usage"):
                management_ip = ip_address.get("ip_address")
                return management_ip in management_ip_list
        return False

    def do_check(self):
        """
        检查cmiplist和nsiplist相关配置是否正确
        1、根据list文件确定各进程正确配置的节点信息
        2、确定各节点需要检查的进程
        3、依次登录节点进行检查
        """
        ip_process_dict = {}
        # 查询list文件，获取原进程节点
        # 建立节点ip-待检查进程字典
        ssh_server = SshService(self.fsm_dev_node)
        for process in PROCESS_DICT.keys():
            if process == "FdsaCoreNode.List" and self.check_not_exist_fdsacore_cfg():
                continue
            process_ip_list = common.get_list_file_ips(ssh_server, self.task_env, process)
            if not process_ip_list:
                continue
            for ip in self.dev_node_dict.keys():
                if ip in process_ip_list:
                    ip_process_dict.setdefault(ip, []).append(process)

            if process in EXPAND_PROCESS_LIST:
                for node in self.expansion_conf_node_list:
                    ip_process_dict.setdefault(node.getIp(), []).append(process)
        # 获取进程配置ip列表
        management_ip_list = common.get_list_file_ips(ssh_server, self.task_env, "CmNode.List")
        for node in self.expansion_conf_node_list:
            management_ip_list.append(node.getIp())
        ip_front_list, ip_backend_list = self.get_storage_ip_list(management_ip_list)
        process_ip_list = [ip_front_list, ip_backend_list]
        self.start_threads_and_waiting(ip_process_dict, process_ip_list)
        if self.err_msg:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(self.ori_info), "\n".join(self.err_msg))
        return CheckResult(CheckResult.PASS, "\n".join(self.ori_info), "")

    def check_not_exist_fdsacore_cfg(self):
        ssh_service = SshService(self.old_zk_node_list[0])
        cli_ret = ssh_service.execute_cmd("ls /opt/fusionstorage/fdsa/conf/fdsa_cluster_srv.conf")
        ssh_service.release_ssh()
        return "No such file or directory" in cli_ret

    def get_storage_ip_list(self, management_ip_list):
        ip_front_list, ip_backend_list = [], []
        node_ips_info = json.loads(RestService(self.cluster).execute_post(NETWORK_SERVICE_URL))
        for info in node_ips_info.get("data"):
            ip_address_list = info.get("ip_address_list")
            if not self.need_check_in_cfg(ip_address_list, management_ip_list):
                continue
            for ip_address in ip_address_list:
                if "storage_frontend" in ip_address.get("ip_usage"):
                    ip_front_list.append(ip_address.get("ip_address"))
                if "storage_backend" in ip_address.get("ip_usage"):
                    ip_backend_list.append(ip_address.get("ip_address"))
        return ip_front_list, ip_backend_list

    def start_threads_and_waiting(self, ip_process_dict, process_ip_list):
        threads = list()
        for ip, process_list in ip_process_dict.items():
            threads.append(self.init_new_thread(ip, process_list, process_ip_list))
        self.logger.info("start threads:{}".format(threads))
        for thread in threads:
            thread.start()
        for thread in threads:
            thread.join()

    def init_new_thread(self, ip, process_list, process_ip_dict):
        return threading.Thread(
            target=self.check_process_list,
            args=(ip, process_list, process_ip_dict),
            name=ip
        )

    def check_process_list(self, ip, process_list, process_ip_list):
        self.sem.acquire()
        self.logger.info("Thread Start")
        ssh_server = None
        try:
            err_processes = []
            dev_node = self.task_env.get_cluster_node_dict().get(ip)
            ssh_server = SshService(dev_node)
            for process in process_list:
                if not PROCESS_DICT.get(process)(ssh_server, process_ip_list, self.ori_info):
                    err_processes.append(process.split("Node.List")[0])
            if err_processes:
                self.err_msg.append(self.get_msg("check.ip.list.fail", (ip, ",".join(err_processes))))
        finally:
            if ssh_server:
                ssh_server.release_ssh()
            self.logger.info("Thread End")
            self.sem.release()
