# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import threading
import time

from py.common.util import common

from py.common.util.connection_util import SshService
from py.common.entity.check_item import CheckItem
from py.common.entity.check_result import CheckResult

CACHE_WATER_TITLE = ["TOTAL_PAGE", "TOTAL_CHUNK", "TOTAL_SBULOG", "TOTAL_VIRTUAL_PAGE"]


def execute(env, method):
    return CheckMetaDiskPartitions(env).execute(method)


class CheckMetaDiskPartitions(CheckItem):
    def __init__(self, env):
        super(CheckMetaDiskPartitions, self).__init__(env)
        self.fsm_dev_node = self.task_env.get_master_fsm_dev_node()
        self.dev_node_dict = self.task_env.get_cluster_node_dict()
        self.ori_info = list()
        self.err_info = list()

    @staticmethod
    def get_highest_cache_water(cli_ret):
        water_m = 0
        for line in cli_ret.splitlines():
            field = line.strip().split()
            if len(field) >= 5 and (field[1] in CACHE_WATER_TITLE) and field[4].isdigit():
                water_m = max(int(field[4]), water_m)
        return water_m

    def do_check(self):
        """
        检查所有eds节点的水位值
        根据回显中TOTAL_PAGE, TOTAL_CHUNK, TOTAL_SUBLOG, TOTAL_VIRTUAL_PAGE 的WaterM值取最高的水位，作为Cache当前的水位值
        在90秒内每隔5秒查询所有EDS节点的水位值，如果存在节点水位值一直高于40%，则不通过
        :return:
        """
        # 通过EBSCtrlNode.List文件获得需要检查的节点
        eds_node_ip_list = common.get_list_file_ips(
            SshService(self.fsm_dev_node), self.task_env, "EBSCtrlNode.List")
        if not eds_node_ip_list:
            return CheckResult(CheckResult.NOT_INVOLVED, "", "")
        self.start_threads_and_waiting(eds_node_ip_list)
        if self.err_info:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(self.ori_info), "\n".join(self.err_info))
        return CheckResult(CheckResult.PASS, "\n".join(self.ori_info), "")

    def start_threads_and_waiting(self, eds_node_ip_list):
        threads = list()
        for ip in eds_node_ip_list:
            self.dev_node_dict.get(ip)
            threads.append(self.init_new_thread(ip, self.dev_node_dict.get(ip)))
        for thread in threads:
            thread.start()
        for thread in threads:
            thread.join()

    def init_new_thread(self, ip, node):
        return threading.Thread(
            target=self.check_cache_water,
            args=(node,),
            name=ip
        )

    def check_cache_water(self, node):
        ssh_service = None
        try:
            ssh_service = SshService(node)
            if not common.attach_app(ssh_service, "EBS"):
                self.err_info.append(self.get_msg("attach.app.fail", (node.getIp(), "EBS")))
                return
            if self.cache_water_higher_than_40_in_90s(ssh_service):
                self.err_info.append(self.get_msg("check.eds.cache.water.not.pass", node.getIp()))
        finally:
            ssh_service.release_ssh()

    def cache_water_higher_than_40_in_90s(self, ssh_service):
        wait_sec = 0
        while wait_sec < 90:
            try:
                cli_ret = ssh_service.execute_cmd("cache show quota global")
                self.ori_info.append(cli_ret)
                water_m = self.get_highest_cache_water(cli_ret)
                if water_m <= 40:
                    return False
            finally:
                wait_sec += 5
                time.sleep(5)
        return True
