# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import json
from py.common.util import common

from py.common.util.connection_util import SshService, RestService
from py.common.entity.check_item import CheckItem
from py.common.entity.check_result import CheckResult

NO_SUCH_FILE_ERROR = "No such file or directory"
NO_PARTITION = "0"


def execute(env, method):
    return CheckMetaDiskPartitions(env).execute(method)


class CheckMetaDiskPartitions(CheckItem):
    def __init__(self, env):
        super(CheckMetaDiskPartitions, self).__init__(env)
        self.ori_info = list()
        self.err_info = list()

    def do_check(self):
        """
        zk扩容
        所有待扩节点的元数据盘不能有分区,如果有分区不通过
        :return:
        """
        expansion_conf_nodes = self.task_env.get_expansion_conf_nodes()
        if len(expansion_conf_nodes) == 0:
            return CheckResult(CheckResult.NOT_PASS, "", self.get_msg("no.expand.node"))
        self._find_disk_partitions(self.task_env.get_cluster_node_dict(), expansion_conf_nodes)
        if len(self.err_info) > 0:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(self.ori_info), "\n".join(self.err_info))
        return CheckResult(CheckResult.PASS, "\n".join(self.ori_info), "")

    def _find_disk_partitions(self, cluster_nodes, expand_nodes):
        for node in expand_nodes:
            self.logger.info("Start find node [{}] disk partitions".format(node.getIp()))
            disk_sn = node.getMetaDisk()
            if "sys_disk" == disk_sn:
                self.logger.info("Node [{}] disk is sys_disk".format(node.getIp()))
                continue
            disk_device_name = self._query_disk_device_name(disk_sn, node.getIp())
            self.logger.info("Node [{}] disk name is {}".format(node.getIp(), disk_device_name))
            cmd = 'fdisk -l /dev/{0} | grep "^/dev/{0}" | wc -l'.format(disk_device_name)
            check_results = self._execute_cmd(cluster_nodes, cmd, node)
            self._parse_result(check_results, disk_sn, node)

    def _query_disk_device_name(self, disk_sn, mgr_ip):
        rest_service = RestService(self.task_env.get_dev_node())
        result_disks = rest_service.execute_get("/dsware/service/resource/queryAllDisk")
        node_disks = json.loads(result_disks).get("disks", {}).get(mgr_ip, [])
        for disk in node_disks:
            if disk.get("devEsn") == disk_sn:
                return disk.get("devName")
        return ""

    def _execute_cmd(self, cluster_nodes, cmd, node):
        ssh_service = SshService(cluster_nodes.get(node.getIp()))
        try:
            check_result = ssh_service.execute_cmd(cmd).splitlines()
            self.ori_info.append(common.get_format_header_ret(node.getIp(), check_result))
            return check_result
        finally:
            ssh_service.release_ssh()

    def _parse_result(self, check_results, disk_sn, node):
        if len(check_results) < 2 or NO_SUCH_FILE_ERROR in " ".join(check_results):
            self.err_info.append(self.get_msg("no.disk.part", (node.getIp(), disk_sn)))
            return
        disk_partition = check_results[-2]
        self.logger.info("Node [{}] disk partitions num [{}]".format(node.getIp(), disk_partition))
        if disk_partition != NO_PARTITION:
            self.err_info.append(
                self.get_msg("metadata.disk.has.partitions", (node.getIp(), disk_sn)))
