#  coding=utf-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

from py.common.util import common
from py.common.entity.check_result import CheckResult
from py.common.entity.check_item import CheckItem
from py.common.util.connection_util import SshService


def execute(env, method):
    return CheckMasterCmStatus(env).execute(method)


class CheckMasterCmStatus(CheckItem):
    def __init__(self, env):
        super(CheckMasterCmStatus, self).__init__(env)
        self.fsm_dev_node = self.task_env.get_master_fsm_dev_node()
        self.expansion_conf_node_list = self.task_env.get_expansion_conf_nodes()
        self.dev_node_dict = self.task_env.get_cluster_node_dict()
        self.front_ip_dict = self.task_env.get_task_cache().get("front_ip_dict")

    def do_check(self):
        """
        检查CM主状态
        """
        err_msg, ori_info = [], []
        # 获得CmNode.List文件节点ip列表
        cm_node_list = common.get_list_file_ips(SshService(self.fsm_dev_node), self.task_env, "CmNode.List")
        # 登录任一节点,获得CM主节点ip
        dev_node = self.dev_node_dict.get(cm_node_list[0])
        ssh_service = SshService(dev_node)
        cm_master_front_ip = self.get_cm_master_front_ip(ssh_service)
        cm_master_ip = self.get_management_ip(cm_master_front_ip, self.front_ip_dict)
        ssh_service.release_ssh()
        # 进程节点数=文件记录的节点数+新增节点数
        process_node_number = len(cm_node_list) + len(self.expansion_conf_node_list)
        # 登陆主节点，判断各个集群状态
        ssh_service = SshService(self.dev_node_dict.get(cm_master_ip))
        self.check_cm_cluster_status(ssh_service, process_node_number, err_msg, ori_info)
        ssh_service.release_ssh()
        if err_msg:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(ori_info), "\n".join(err_msg))
        return CheckResult(CheckResult.PASS, "", "")

    def check_cm_cluster_status(self, ssh_service, node_number, err_msg, ori_info):
        common.attach_app(ssh_service, "clustermanager")
        cli_ret = ssh_service.execute_cmd("cm show all")
        cli_ret_list = cli_ret.split("\n" + "-"*20)
        # 检查CM进程ROLE字段是否有master和slave
        if not self.check_cm_role(cli_ret_list[0], node_number):
            err_msg.append(self.get_msg("check.cm.master.status.not.pass", "CM"))
            ori_info.append(cli_ret_list[0])
        # 检查ccdb_monitor、djob、kmm_server进程ROLE字段是否有1和0
        for process_info in cli_ret_list:
            is_need_check, process_name = self.is_need_check_process(process_info)
            if is_need_check and (not self.check_process_role(process_info, node_number)):
                err_msg.append(self.get_msg("check.cm.master.status.not.pass", process_name))
                ori_info.append(process_info)

    @staticmethod
    def is_need_check_process(process_info):
        for process in ["ccdb_server", "djob", "kmm_server"]:
            if process in process_info:
                return True, process
        return False, ""

    @staticmethod
    def check_cm_role(process_info, node_number):
        number_master, number_slave = 0, 0
        for line in process_info.splitlines():
            if "slave" in line:
                number_slave += 1
            if "master" in line:
                number_master += 1
        return number_master == 1 and (number_slave + 1) == node_number

    @staticmethod
    def check_process_role(process_info, node_number):
        number_master, number_slave = 0, 0
        for line in process_info.splitlines():
            line = line.strip().split()
            if len(line) >= 2 and line[1] == "0":
                number_slave += 1
            if len(line) >= 2 and line[1] == "1":
                number_master += 1
        return number_master == 1 and (number_slave + 1) == node_number

    @staticmethod
    def get_cm_master_front_ip(ssh_service):
        common.attach_app(ssh_service, "clustermanager")
        cli_ret = ssh_service.execute_cmd("cm show clusterInfo")
        for line in cli_ret.splitlines():
            if "master" in line:
                ip_first = line.strip().split()[1]
                # ipv6场景查出的前端ip可能前后有[],映射字典里没有，需要处理
                return ip_first[1:-1] if ip_first[0] == "[" else ip_first
        return ""

    @staticmethod
    def get_management_ip(cm_master_front_ip, front_ip_dict):
        for front_ip in front_ip_dict.keys():
            if front_ip == cm_master_front_ip.strip():
                return front_ip_dict[front_ip]
        return ""
