#  coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import json
from py.common.util import common

from py.common.entity.check_item import CheckItem
from py.common.util.connection_util import RestService
from py.common.entity.check_result import CheckResult
from py.common.java_adapter import java_adapter

RATE_1024 = 1024
RATE_1000 = 1000


def execute(env, method):
    """
    :return: 执行结果
    """
    return CheckMetaDiskType(env).execute(method)


class CheckMetaDiskType(CheckItem):
    """
    zk扩容
    所有待扩节点的元数据盘和控制集群的元数据盘类型一致且状态健康，大小满足
    """

    def __init__(self, env):
        super(CheckMetaDiskType, self).__init__(env)
        self.ori_info = list()
        self.error_info = list()
        self.cluster = self.task_env.get_dev_node()
        self.version = java_adapter.get_complete_dev_version(self.cluster)
        self.fsm_dev_node = self.task_env.get_master_fsm_dev_node()

    def do_check(self):
        """
        执行任务
        """
        rest_service = RestService(self.cluster)
        metadata_disk_type = self._query_manager_cluster_info(rest_service)
        if len(self.error_info) > 0:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(self.ori_info), "\n".join(self.error_info))

        self._node_meta_disk_type(rest_service, metadata_disk_type)

        if len(self.error_info) > 0:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(self.ori_info), "\n".join(self.error_info))

        return CheckResult(CheckResult.PASS, "\n".join(self.ori_info), "")

    def _node_meta_disk_type(self, rest_service, metadata_disk_type):
        if len(self.task_env.get_expansion_conf_nodes()) == 0:
            self.error_info.append(self.get_msg("no.expand.node"))
            return
        for conf_node in self.task_env.get_expansion_conf_nodes():
            if "sys_disk" == metadata_disk_type == conf_node.getMetaDisk():
                continue
            response_info = rest_service.execute_get(
                "/dsware/service/resource/queryDiskInfo?ip={}".format(conf_node.getIp()), None)
            json_disk_infos = json.loads(response_info).get("disks", [])
            if len(json_disk_infos) == 0:
                self.error_info.append(self.get_msg("no.disk.info"))
            disk_errors = self._is_match_disk(json_disk_infos, conf_node, metadata_disk_type)
            if len(disk_errors) > 0:
                self.error_info.append(self.get_msg("metadata.disk.error",
                                                    (conf_node.getIp(), conf_node.getMetaDisk(),
                                                     ",".join(disk_errors))))

    def _query_manager_cluster_info(self, rest_service):
        manager_cluster_info = rest_service.execute_get("/dsware/service/cluster/queryManageCluster")
        node_infos = json.loads(manager_cluster_info).get("nodeInfo", [])
        if len(node_infos) == 0:
            self.error_info.append(self.get_msg("no.node.error"))
            return ""
        node_info = node_infos[0]
        metadata_disk_type = node_info.get("diskInfo", {}).get("diskType", {})
        if not metadata_disk_type:
            self.error_info.append(self.get_msg("query.cluster.metadata.type.error"))
            return ""
        self.ori_info.append("Cluster Metadata: \n {}".format(node_info))
        return metadata_disk_type

    def _is_match_disk(self, disk_infos, conf_node, metadata_disk_type):
        disk_error = list()
        is_match_flag = False
        for disk_info in disk_infos:
            if conf_node.getMetaDisk() == disk_info.get("diskSn", {}):
                is_match_flag = True
                self.ori_info.append(common.get_format_header_ret(conf_node.getIp(), disk_info))
                self._disk_role(disk_info, disk_error)
                self._disk_type(disk_info, metadata_disk_type, disk_error)
                self._disk_size(disk_info, disk_error)
                self._disk_status(disk_info, disk_error)
                self._disk_sector_size(disk_info, disk_error)
        if not is_match_flag:
            self.error_info.append(self.get_msg("no.sn", conf_node.getIp()))
        return disk_error

    def _disk_sector_size(self, disk_info, disk_error):
        if self._is_judging_sector(disk_info) and disk_info.get("sectorSize", 0) != 512:
            disk_error.append(self.get_msg("disk.sector.size"))

    def _is_judging_sector(self, disk_info):
        return disk_info.get("diskType") == "ssd_card" and \
               (self.version.startswith("8.1.2")
                or self.version.startswith("8.1.3")
                or self.version.startswith("8.1.0"))

    def _disk_role(self, disk_info, disk_error):
        if not disk_info.get("diskRole", {}) == "no_use":
            disk_error.append(self.get_msg("disk.role"))

    def _disk_status(self, disk_info, disk_error):
        if not disk_info.get("diskStatus", 0) == 0:
            disk_error.append(self.get_msg("disk.status"))

    def _disk_size(self, disk_info, disk_error):
        if disk_info.get("diskType", "") == "ssd_card":
            disk_size = 75 * RATE_1000 * RATE_1000 * RATE_1000
        else:
            disk_size = 75 * RATE_1024 * RATE_1024 * RATE_1024
        if disk_info.get("mediaCapacityForByte", 0) < disk_size:
            disk_error.append(self.get_msg("disk.size"))

    def _disk_type(self, disk_info, metadata_disk_type, disk_error):
        if not metadata_disk_type == disk_info.get("diskType"):
            disk_error.append(self.get_msg("disk.type"))
